# -*- coding: utf-8 -*-

u"""
お守りID、スキルIDから取りうるスキルの最大値への組み合わせテーブルの作成・アクセスをするモジュール
2013/12/19 written by kei9
"""

import sqlite3
import csv

import skilltable
import amulettable
#import singleton

# for minmax table
MASTER_NAME = u"skill_minmax_master"
NAME = u"skill_minmax_{amulet_id}"
COL_AMULET_ID = u"amulet_id"
COL_SKILL_TABLE_NAME = u"skill_table_name"
COL_MIN1 = u"min1"
COL_MIN2 = u"min2"
COL_MAX1 = u"max1"
COL_MAX2 = u"max2"
COL_SKILL_ID = u"skill_id"
CREATE_MASTER_SQL = u"""create table if not exists {master}
    (id integer primary key, {amulet_id} integer, {table_name} varchar,
    foreign key(amulet_id) references {amulet_table}(id));""".format(
        master=MASTER_NAME,
        amulet_id=COL_AMULET_ID,
        amulet_table=amulettable.NAME,
        table_name=COL_SKILL_TABLE_NAME)
CREATE_SQL = u"""create table if not exists {{table_name}} 
    (id integer primary key, {skill_id} integer,
    {min1} integer, {max1} integer, {min2} integer, {max2} integer, 
    foreign key(skill_id) references {skill_table}({skill_id_col}));""".format(
        skill_id=COL_SKILL_ID,
        min1=COL_MIN1, min2=COL_MIN2,
        max1=COL_MAX1, max2=COL_MAX2,
        skill_table=skilltable.NAME,
        skill_id_col=skilltable.COL_SKILL_ID)
INSERT_MASTER_SQL = u"""insert into {master}
    ({amulet_id}, {table_col}) values(?,?);""".format(
        master=MASTER_NAME,
        table_col=COL_SKILL_TABLE_NAME,
        amulet_id=COL_AMULET_ID)
INSERT_SQL = u"""insert into {{table_name}}
    ({skill_id}, {min1}, {max1}, {min2}, {max2}) values(?,?,?,?,?)""".format(
        skill_id=COL_SKILL_ID,
        min1=COL_MIN1, min2=COL_MIN2,
        max1=COL_MAX1, max2=COL_MAX2)
SELECT_MASTER_ALL_SQL = u"""select {amu_id}, {table_col} from {table}""".format(
        amu_id=COL_AMULET_ID,
        table_col=COL_SKILL_TABLE_NAME,
        table=MASTER_NAME)
SELECT_ALL_SQL = u"""select {skill_id}, {min1}, {max1}, {min2}, {max2} from {{table_name}} """.format(
        skill_id=COL_SKILL_ID,
        min1=COL_MIN1, max1=COL_MAX1,
        min2=COL_MIN2, max2=COL_MAX2)
SELECT_NON_ZERO_SKILL1_ID_SQL = u"""select {skill_id} from {{table_name}} 
    where {min1} != 0 and {max1} != 0""".format(
        skill_id=COL_SKILL_ID,
        min1=COL_MIN1, max1=COL_MAX1)
SELECT_NON_ZERO_SKILL2_ID_SQL = u"""select {skill_id} from {{table_name}} 
    where {min2} != 0 and {max2} != 0""".format(
        skill_id=COL_SKILL_ID,
        min2=COL_MIN2, max2=COL_MAX2)
SELECT_MIN1_SQL = u"""select min({col}) from {{table_name}} """.format(
        col=COL_MIN1)
SELECT_MAX1_SQL = u"""select max({col}) from {{table_name}} """.format(
        col=COL_MAX1)
SELECT_MIN2_SQL = u"""select min({col}) from {{table_name}} """.format(
        col=COL_MIN2)
SELECT_MAX2_SQL = u"""select max({col}) from {{table_name}} """.format(
        col=COL_MAX2)
SELECT_MAX1_OF_SKILL_SQL = u"""select max({col}) from {{table_name}} 
    where skill_id={{skill_id}}""".format(
        col=COL_MAX1)
SELECT_MAX2_OF_SKILL_SQL = u"""select max({col}) from {{table_name}} 
    where skill_id={{skill_id}}""".format(
        col=COL_MAX2)

class SkillMinMaxTableGenerator(object):
    u"""お守りID、スキルIDから取りうるスキルの最大値への組み合わせテーブルの作成クラス"""
    def __init__(self):
        u""" コンストラクタ """
        self._amulet_id2skill1_filename_dict = {}
        self._amulet_id2skill2_filename_dict = {}

    def insert_master_data(self, db_cursor, csv_reader):
        u""" マスターテーブルを作成する。
        """
        db_cursor.execute(CREATE_MASTER_SQL)
        accessor = amulettable.AmuletTableAccessor(db_cursor)
        amu_id2name, amu_name2id = accessor.get_dict()

        csv_reader.next()   # skip header row
        for row in csv_reader:
            amulet_name = row[0].strip()
            #if not isinstance(amulet_name, unicode):
            #    amulet_name = unicode(amulet_name, u"utf-8")
            skill1_file, skill2_file = row[1].strip(), row[2].strip()
            amulet_id = amu_name2id[amulet_name]
            table_name = NAME.format(amulet_id=amulet_id)
            self._amulet_id2skill1_filename_dict[amulet_id] = skill1_file
            self._amulet_id2skill2_filename_dict[amulet_id] = skill2_file
            db_cursor.execute(INSERT_MASTER_SQL, (amulet_id, table_name))

    def get_skill_filenames(self):
        u""" スキル1, スキル2の最大最小が記載されたファイル名をお守りIDに関連付けて返す
        return {amulet_id:(skill1_filename, skill2_filename)}"""
        result_dict = {}
        for amu_id in self._amulet_id2skill1_filename_dict.keys():
            result_dict[amu_id] = (self._amulet_id2skill1_filename_dict[amu_id],
                    self._amulet_id2skill2_filename_dict[amu_id])
        return result_dict

    def insert_data(self, db_cursor, amulet_id2csv_readers_dict):
        u""" お守りIDと関連付けられたcsv_reader(skill1,skill2)からデータを読み込み、
        db_cursorへデータを挿入する。
        ammulet_id2csv_readers_dict:{amulet_id:(csv_reader_skill1, csv_reader_skill2)}"""
        accessor = skilltable.SkillTableAccessor(db_cursor)
        skill_id2name, skill_name2id = accessor.get_dict()

        for amulet_id in amulet_id2csv_readers_dict.keys():
            table_name = NAME.format(amulet_id=amulet_id)
            db_cursor.execute(CREATE_SQL.format(table_name=table_name))
            skill1_reader, skill2_reader = amulet_id2csv_readers_dict[amulet_id]
            insert_values = {}

            skill1_reader.next()   # skip header row
            for row in skill1_reader:  # (skill_name, min1, max1)
                skill_name = row[0].strip()
                #if not isinstance(skill_name, unicode):
                #    skill_name = unicode(skill_name, u"utf-8")
                min1, max1 = int(row[1].strip()), int(row[2].strip())
                skill_id = skill_name2id[skill_name]
                insert_values[skill_id] = (skill_id, min1, max1, 0, 0)

            if skill2_reader is not None:
                skill2_reader.next()   # skip header row
                for row in skill2_reader:  # (skill_name, min2, max2)
                    skill_name = row[0].strip()
                    #if not isinstance(skill_name, unicode):
                    #    skill_name = unicode(skill_name, u"utf-8")
                    min2, max2 = int(row[1].strip()), int(row[2].strip())
                    skill_id = skill_name2id[skill_name]
                    if skill_id in insert_values:
                        val = insert_values[skill_id]
                        insert_values[skill_id] = (skill_id, val[1], val[2], min2, max2)
                    else:
                        insert_values[skill_id] = (skill_id, 0, 0, min2, max2)
            sql = INSERT_SQL.format(table_name=table_name)
            for skill_id, values in insert_values.items():
                db_cursor.execute(sql, values)

class SkillMinMaxTableAccessor(object):
    u""" お守りID、スキルIDから取りうるスキルの最大値への組み合わせテーブルのアクセスクラス"""
    #__metaclass__ = singleton.Singleton
    def __init__(self, db_cursor):
        u""" db_cursor: cursor of sqlite3 database """
        self._cursor = db_cursor
        accessor = amulettable.AmuletTableAccessor(db_cursor)
        self._amu_id2name, self._amu_name2id = accessor.get_dict()

        accessor = skilltable.SkillTableAccessor(db_cursor)
        self._skill_id2name, self._skill_name2id = accessor.get_dict()
        self._amu_id2table_name = {}

        self._cursor.execute(SELECT_MASTER_ALL_SQL)
        for row in self._cursor.fetchall():
            amu_id, table_name = row
            self._amu_id2table_name[amu_id] = table_name

    def get_minmax_by_amulet(self, amulet_id):
        u""" 指定されたお守りIDにおけるスキルIDと最大値、最小値の組み合わせを返す
        return {skill1_id:(min1,max1)}, {skill2_id;(min2,max2)}"""
        skill1_dict, skill2_dict = {}, {}
        if amulet_id in self._amu_id2table_name:
            table_name = self._amu_id2table_name[amulet_id]
            #get non zero skill1_ids
            sql = SELECT_NON_ZERO_SKILL1_ID_SQL.format(table_name=table_name)
            self._cursor.execute(sql)
            skill1_ids = [row[0] for row in self._cursor.fetchall()]

            #get non zero skill2_ids
            sql = SELECT_NON_ZERO_SKILL2_ID_SQL.format(table_name=table_name)
            self._cursor.execute(sql)
            skill2_ids = [row[0] for row in self._cursor.fetchall()]

            self._cursor.execute(SELECT_ALL_SQL.format(table_name=table_name))
            for (skill_id, min1, max1, min2, max2) in self._cursor.fetchall():
                if skill_id in skill1_ids:
                    skill1_dict[skill_id] = (min1, max1)
                if skill_id in skill2_ids:
                    skill2_dict[skill_id] = (min2, max2)

        return (skill1_dict, skill2_dict)

    def get_minmax_by_id(self):
        u""" お守りIDごとのスキルIDと最大値、最小値の組み合わせを辞書として返す
        return {amulet_id:({skill1_id:(min1,max1)}, {skill2_id;(min2,max2)})}"""
        result_dict = {}
        for amu_id in self._amu_id2name.keys():
            result_dict[amu_id] = self.get_minmax_by_amulet(amu_id)
        return result_dict

    def get_minmax_by_name(self):
        u""" お守り名ごとのスキルIDと最大値、最小値の組み合わせを辞書として返す
        return {amulet_name:({skill1_name:(min1,max1)}, {skill2_name;(min2,max2)})}"""
        result_dict = {}
        for amu_id in self._amu_id2name.keys():
            id_dict1, id_dict2 = self.get_minmax_by_amulet(amu_id)
            name_dict1, name_dict2 = {}, {}
            for skill_id, vals in id_dict1.items():
                name_dict1[self._skill_id2name[skill_id]] = vals
            for skill_id, vals in id_dict2.items():
                name_dict2[self._skill_id2name[skill_id]] = vals
            result_dict[self._amu_id2name[amu_id]] = (name_dict1, name_dict2)
        return result_dict
