# -*- coding: utf-8 -*-

u"""
SEED1とスキル名、お守り名の組み合わせテーブルの作成・アクセスをするモジュール
2013/12/19 written by kei9
"""

import sqlite3
import csv

import mh4constnumbers
import skilltable
import amulettable
#import singleton

# for seed1 table by tenun
NAME = u"seed1_table_{alchemy_type}"
COL_TABLE = u"table_no"
COL_NO = u"no"
COL_SEED1 = u"seed1"
COL_RESULT_NO = u"result_no"
COL_AMULET1 = u"amulet_id1"
COL_AMULET2 = u"amulet_id2"
COL_AMULET3 = u"amulet_id3"
COL_AMULET4 = u"amulet_id4"
COL_AMULET5 = u"amulet_id5"
COL_AMULET6 = u"amulet_id6"
COL_AMULET7 = u"amulet_id7" # 555 doesn't has this col
COL_AMULET_LIST = [COL_AMULET1, COL_AMULET2, COL_AMULET3,
        COL_AMULET4, COL_AMULET5, COL_AMULET6, COL_AMULET7]
COL_SKILL1_1 = u"skill1_1"
COL_SKILL1_2 = u"skill1_2"
COL_SKILL1_3 = u"skill1_3"
COL_SKILL1_4 = u"skill1_4"
COL_SKILL1_5 = u"skill1_5"
COL_SKILL1_6 = u"skill1_6"
COL_SKILL1_7 = u"skill1_7"  # 555 doesn't has this col
COL_SKILL1_LIST = [COL_SKILL1_1, COL_SKILL1_2, COL_SKILL1_3,
        COL_SKILL1_4, COL_SKILL1_5, COL_SKILL1_6, COL_SKILL1_7]

# tenun
CREATE_SQL =  u"""create table if not exists {{table_name}}
    ({no} integer primary key, {table_no} integer, {seed} integer unique, {result_num} integer,
    {amulet_id1} integer, {amulet_id2} integer, {amulet_id3} integer,
    {amulet_id4} integer, {amulet_id5} integer, {amulet_id6} integer, {amulet_id7} integer,
    {skill_id1} integer, {skill_id2} integer, {skill_id3} integer,
    {skill_id4} integer, {skill_id5} integer, {skill_id6} integer, {skill_id7} integer,
    foreign key({amulet_id1}) references {amulet_table}(id),
    foreign key({amulet_id2}) references {amulet_table}(id),
    foreign key({amulet_id3}) references {amulet_table}(id),
    foreign key({amulet_id4}) references {amulet_table}(id),
    foreign key({amulet_id5}) references {amulet_table}(id),
    foreign key({amulet_id6}) references {amulet_table}(id),
    foreign key({amulet_id7}) references {amulet_table}(id),
    foreign key({skill_id1}) references {skill_table}(id),
    foreign key({skill_id2}) references {skill_table}(id),
    foreign key({skill_id3}) references {skill_table}(id),
    foreign key({skill_id4}) references {skill_table}(id),
    foreign key({skill_id5}) references {skill_table}(id),
    foreign key({skill_id6}) references {skill_table}(id),
    foreign key({skill_id7}) references {skill_table}(id)
    );""".format(
        no=COL_NO,
        seed=COL_SEED1,
        table_no=COL_TABLE,
        result_num=COL_RESULT_NO,
        amulet_id1=COL_AMULET1,
        amulet_id2=COL_AMULET2,
        amulet_id3=COL_AMULET3,
        amulet_id4=COL_AMULET4,
        amulet_id5=COL_AMULET5,
        amulet_id6=COL_AMULET6,
        amulet_id7=COL_AMULET7,
        skill_id1=COL_SKILL1_1,
        skill_id2=COL_SKILL1_2,
        skill_id3=COL_SKILL1_3,
        skill_id4=COL_SKILL1_4,
        skill_id5=COL_SKILL1_5,
        skill_id6=COL_SKILL1_6,
        skill_id7=COL_SKILL1_7,
        amulet_table=amulettable.NAME,
        skill_table=skilltable.NAME)
INSERT_SQL = u"""insert into {{table_name}}
    ({no}, {table_no}, {seed}, {result_num},
    {amulet_id1}, {amulet_id2}, {amulet_id3},
    {amulet_id4}, {amulet_id5}, {amulet_id6}, {amulet_id7},
    {skill_id1}, {skill_id2}, {skill_id3},
    {skill_id4}, {skill_id5}, {skill_id6}, {skill_id7})
    values(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?);""".format(
        no=COL_NO,
        seed=COL_SEED1,
        table_no=COL_TABLE,
        result_num=COL_RESULT_NO,
        amulet_id1=COL_AMULET1,
        amulet_id2=COL_AMULET2,
        amulet_id3=COL_AMULET3,
        amulet_id4=COL_AMULET4,
        amulet_id5=COL_AMULET5,
        amulet_id6=COL_AMULET6,
        amulet_id7=COL_AMULET7,
        skill_id1=COL_SKILL1_1,
        skill_id2=COL_SKILL1_2,
        skill_id3=COL_SKILL1_3,
        skill_id4=COL_SKILL1_4,
        skill_id5=COL_SKILL1_5,
        skill_id6=COL_SKILL1_6,
        skill_id7=COL_SKILL1_7)
SELECT_SQL = u""" select {seed} from {{table_name}} 
    """.format(seed=COL_SEED1)
SELECT_TABLE_NOS_FROM_SEED1_SQL = u"""select {table_no}, {no} from {{table_name}}
    where {seed1}={{seed1}}""".format(
        no=COL_NO,
        seed1=COL_SEED1,
        table_no=COL_TABLE)
SELECT_NEAR_SEED1S_FROM_NO_SQL = u"""select {no}, {seed1}, {result_num},
    {amulet_id1}, {amulet_id2}, {amulet_id3},
    {amulet_id4}, {amulet_id5}, {amulet_id6}, {amulet_id7},
    {skill1_id1}, {skill1_id2}, {skill1_id3},
    {skill1_id4}, {skill1_id5}, {skill1_id6}, {skill1_id7}
    from {{table_name}}
    where {table_no}={{table_no}} and {no}>={{min_no}} and {no}<={{max_no}}
    order by {no} ASC """.format(
        no=COL_NO,
        seed1=COL_SEED1,
        table_no=COL_TABLE,
        result_num=COL_RESULT_NO,
        amulet_id1=COL_AMULET1,
        amulet_id2=COL_AMULET2,
        amulet_id3=COL_AMULET3,
        amulet_id4=COL_AMULET4,
        amulet_id5=COL_AMULET5,
        amulet_id6=COL_AMULET6,
        amulet_id7=COL_AMULET7,
        skill1_id1=COL_SKILL1_1,
        skill1_id2=COL_SKILL1_2,
        skill1_id3=COL_SKILL1_3,
        skill1_id4=COL_SKILL1_4,
        skill1_id5=COL_SKILL1_5,
        skill1_id6=COL_SKILL1_6,
        skill1_id7=COL_SKILL1_7)
SELECT_NEAR_NAMES_FROM_NO_SQL = u"""select {no}, {seed1} from {{table_name}}
    where {table_no}={{table_no}} and {no}>={{min_no}} and {no}<={{max_no}}
    order by {no} ASC """.format(
        no=COL_NO,
        seed1=COL_SEED1,
        table_no=COL_TABLE)
SELECT_ALL_FROM_SEED1_SQL = u"""select {no}, {table_no}, {result_num},
    {amulet_id1}, {amulet_id2}, {amulet_id3},
    {amulet_id4}, {amulet_id5}, {amulet_id6}, {amulet_id7},
    {skill1_id1}, {skill1_id2}, {skill1_id3},
    {skill1_id4}, {skill1_id5}, {skill1_id6}, {skill1_id7}
    from {{table_name}} where {seed1}={{seed1}} """.format(
        no=COL_NO,
        seed1=COL_SEED1,
        table_no=COL_TABLE,
        result_num=COL_RESULT_NO,
        amulet_id1=COL_AMULET1,
        amulet_id2=COL_AMULET2,
        amulet_id3=COL_AMULET3,
        amulet_id4=COL_AMULET4,
        amulet_id5=COL_AMULET5,
        amulet_id6=COL_AMULET6,
        amulet_id7=COL_AMULET7,
        skill1_id1=COL_SKILL1_1,
        skill1_id2=COL_SKILL1_2,
        skill1_id3=COL_SKILL1_3,
        skill1_id4=COL_SKILL1_4,
        skill1_id5=COL_SKILL1_5,
        skill1_id6=COL_SKILL1_6,
        skill1_id7=COL_SKILL1_7)

# extract skill1 by place
SELECT_SEED1_SQL = u"""select
    {seed1} from {{table_name}} """.format(
        seed1=COL_SEED1)
# select seed1 number with skill1 and place
SELECT_SEED1_BY_SKILL1_SQL = u"""select
    {seed1} from {{table_name}} 
    where {{amulet_col}}={{amulet_id}} and 
    {{skill1_col}}={{skill1_id}}""".format(
        seed1=COL_SEED1)

# count seed1 number with skill1 and place
COUNT_SEED1_SQL = u"""select
    count({seed1}) from {{table_name}} """.format(
        seed1=COL_SEED1)
# count seed1 number with skill1 and place
COUNT_SEED1_BY_SKILL1_SQL = u"""select
    count({seed1}) from {{table_name}} 
    where {{amulet_col}}={{amulet_id}} and 
    {{skill1_col}}={{skill1_id}}""".format(
        seed1=COL_SEED1)

class Seed1TenunTableGenerator(object):
    u""" スキルIDとスキル名の組み合わせテーブルを作成するクラス """
    def __init__(self):
        u""" コンストラクタ """
        pass

    def insert_data(self, db_cursor, csv_reader, key_alchemy):
        u""" csv_readerからデータを読み込み、 db_cursorへデータを挿入する。
        key_alchemyはKEY_TENUN555 or KEY_TENUN888"""
        if (key_alchemy == mh4constnumbers.KEY_TENUN555 or
            key_alchemy == mh4constnumbers.KEY_TENUN888):
            table_name = NAME.format(alchemy_type=key_alchemy)
        else:
            raise NotImplementedError(u"this alchemy type is not supported")
        amu_accessor = amulettable.AmuletTableAccessor(db_cursor)
        skill_accessor = skilltable.SkillTableAccessor(db_cursor)
        amu_id2name, amu_name2id = amu_accessor.get_dict()
        skill_id2name, skill_name2id = skill_accessor.get_dict()

        db_cursor.execute(CREATE_SQL.format(table_name=table_name))

        insert_sql = INSERT_SQL.format(table_name=table_name)
        csv_reader.next()   # skip header row
        csv_reader.next()   # skip header row
        for row in csv_reader:
            # (table_no, no, seed1, result_no, amu1, amu2, ...,, amu7, skill_id1, skill_id2, ..., skill_id7)
            vals = [x.strip() for x in row]
            table_no, no, seed1, result_no = vals[0], vals[1], vals[2],vals[3]
            amu_ids = [amu_name2id[x] if x in amu_name2id else mh4constnumbers.NO_DATA for x in vals[4:11]]
            if len(amu_ids) < 7:
                amu_ids += [mh4constnumbers.NO_DATA] * (7-len(amu_ids))
            skill_ids = [skill_name2id[x] if x in skill_name2id else mh4constnumbers.NO_DATA for x in vals[11:18]]
            if len(skill_ids) < 7:
                skill_ids += [mh4constnumbers.NO_DATA] * (7-len(skill_ids))
            vals = [no, table_no, seed1, result_no] + amu_ids + skill_ids
            db_cursor.execute(insert_sql, tuple(vals))

class Seed1TenunTableAccessor(object):
    u""" スキルIDとスキル名の組み合わせテーブルへのアクセス用クラス """
    #__metaclass__ = singleton.Singleton
    def __init__(self, db_cursor):
        u""" db_cursor: cursor of sqlite3 database """
        self._cursor = db_cursor
        amu_accessor = amulettable.AmuletTableAccessor(db_cursor)
        skill_accessor = skilltable.SkillTableAccessor(db_cursor)
        self._amu_id2name, self._amu_name2id = amu_accessor.get_dict()
        self._skill_id2name, self._skill_name2id = skill_accessor.get_dict()

    def select_seed1s_from_ids(self, amuid_skillid_list, key_alchemy):
        u""" お守りIDとスキルIDのリストから、条件を満たすSeed1を検索する
        amuid_skillid_list: [(amu_id, skill_id),...]"""
        if (key_alchemy == mh4constnumbers.KEY_TENUN555 or
            key_alchemy == mh4constnumbers.KEY_TENUN888):
            table_name = NAME.format(alchemy_type=key_alchemy)
        else:
            raise NotImplementedError(u"this alchemy type is not supported")

        where_list = []
        result = set()
        for (amu_id, skill_id), amu_col, skill_col in zip(
                amuid_skillid_list, COL_AMULET_LIST, COL_SKILL1_LIST):
            if amu_id in self._amu_id2name and skill_id in self._skill_id2name:
                where_list.append("{0}={1} and {2}={3}".format(amu_id, amu_col, skill_id, skill_col))

        if len(where_list) > 0:
            sql = SELECT_SQL.format(table_name=table_name) + " where " + " and ".join(where_list)
            self._cursor.execute(sql)
            result = set([x[0] for x in self._cursor.fetchall()])

        return result

    def select_seed1s_from_names(self, amuname_skillname_list, key_alchemy):
        u""" お守りIDとスキルIDのリストから、条件を満たすSeed1を検索する
        amuname_skillname_list: [(amu_name, skill_name),...]"""
        amuid_skillid_list = [
                (self._amu_name2id[amu_name] if amu_name in self._amu_name2id else None,
                self._skill_name2id[skill_name] if skill_name in self._skill_name2id else None)
                for amu_name, skill_name in amuname_skillname_list]
        return self.select_seed1s_from_ids(amuid_skillid_list, key_alchemy)

    def select_table_nos_from_seed1(self, seed1, key_alchemy):
        u""" Seed1と錬金の種類から通しNo,テーブルNo,お守り個数を返す。
        return (no, table_no, result_num)"""
        if (key_alchemy == mh4constnumbers.KEY_TENUN555 or
            key_alchemy == mh4constnumbers.KEY_TENUN888):
            table_name = NAME.format(alchemy_type=key_alchemy)
        else:
            raise NotImplementedError(u"this alchemy type is not supported")

        sql = SELECT_ALL_FROM_SEED1_SQL.format(table_name=table_name, seed1=seed1)
        self._cursor.execute(sql)
        row = self._cursor.fetchone()
        try:
            no, table_no, result_num = row[0], row[1], row[2]
            return (no, table_no, result_num)
        except TypeError as e:
            raise KeyError(u"seed1:{0} not found".format(seed1))

    def select_ids_from_seed1(self, seed1, key_alchemy):
        u""" Seed1と錬金の種類から通しNo,テーブルNo,お守り個数, 
        お守りId(1-7)、スキルID(1-7)を返す。値が存在しない場所はNoneで埋める。
        return (no, table_no, result_num, (amulet_ids), (skill_ids))"""
        if (key_alchemy == mh4constnumbers.KEY_TENUN555 or
            key_alchemy == mh4constnumbers.KEY_TENUN888):
            table_name = NAME.format(alchemy_type=key_alchemy)
        else:
            raise NotImplementedError(u"this alchemy type is not supported")

        sql = SELECT_ALL_FROM_SEED1_SQL.format(table_name=table_name, seed1=seed1)
        self._cursor.execute(sql)
        row = self._cursor.fetchone()
        no, table_no, result_num = row[0], row[1], row[2]
        amulet_ids = [x if x in self._amu_id2name else None for x in row[3:10]]
        skill_ids = [x if x in self._skill_id2name else None for x in row[10:]]
        return (no, table_no, result_num, amulet_ids, skill_ids)

    def select_names_from_seed1(self, seed1, key_alchemy):
        u""" Seed1と錬金の種類から通しNo,テーブルNo,お守り個数, 
        お守り名(1-7)、スキル名(1-7)を返す。値が存在しない場所は空文字列で埋める。
        return (no, table_no, result_num, (amulet_names), (skill_names))"""
        (no, table_no, result_num, amulet_ids, skill_ids) = self.select_ids_from_seed1(seed1, key_alchemy)
        amulet_names = [self._amu_id2name[x] if x in self._amu_id2name else u"" for x in amulet_ids]
        skill_names = [self._skill_id2name[x] if x in self._skill_id2name else u"" for x in skill_ids]
        return (no, table_no, result_num, amulet_names, skill_names)

    def select_near_ids_from_seed1(self, no, table_no, smaller_num, larger_num, key_alchemy):
        u""" 通し番号とテーブル番および錬金の種類からその周囲のseed1の通し番号、テーブルNo.、
        お守り個数, お守りId(1-7)、スキルID(1-7)を辞書として返す。
        ただし辞書のキーは通し番号とする。
        値が存在しない場所はNoneで埋める。
        return {no: (seed1, result_num, (amulet_ids), (skill_ids))}"""
        if (key_alchemy == mh4constnumbers.KEY_TENUN555 or
            key_alchemy == mh4constnumbers.KEY_TENUN888):
            table_name = NAME.format(alchemy_type=key_alchemy)
        else:
            raise NotImplementedError(u"this alchemy type is not supported")
        max_no = no + larger_num
        min_no = no - smaller_num

        sql = SELECT_NEAR_SEED1S_FROM_NO_SQL.format(
                table_name=table_name,
                max_no=max_no, min_no=min_no, table_no=table_no)
        self._cursor.execute(sql)

        result_dict = {}
        for row in self._cursor.fetchall():
            no, seed1, result_num = row[0], row[1], row[2]
            amulet_ids = [x if x in self._amu_id2name else None for x in row[3:10]]
            skill_ids = [x if x in self._skill_id2name else None for x in row[10:]]
            result_dict[no] = (seed1, result_num, amulet_ids, skill_ids)

        return result_dict

    def select_near_names_from_seed1(self, no, table_no, smaller_num, larger_num, key_alchemy):
        u""" 通し番号とテーブル番および錬金の種類からその周囲のseed1の通し番号、テーブルNo.、
        お守り個数, お守り名(1-7)、スキル名(1-7)を辞書として返す。
        ただし辞書のキーは通し番号とする。
        値が存在しない場所は空文字列で埋める。
        return {no: (seed1, result_num, (amulet_names), (skill_names))}"""
        if (key_alchemy == mh4constnumbers.KEY_TENUN555 or
            key_alchemy == mh4constnumbers.KEY_TENUN888):
            table_name = NAME.format(alchemy_type=key_alchemy)
        else:
            raise NotImplementedError(u"this alchemy type is not supported")
        max_no = no + larger_num
        min_no = no - smaller_num

        sql = SELECT_NEAR_SEED1S_FROM_NO_SQL.format(
                table_name=table_name,
                max_no=max_no, min_no=min_no, table_no=table_no)
        self._cursor.execute(sql)

        result_dict = {}
        for row in self._cursor.fetchall():
            no, seed1, result_num = row[0], row[1], row[2]
            amulet_names = [self._amu_id2name[x] if x in self._amu_id2name else u"" for x in row[3:10]]
            skill_names = [self._skill_id2name[x] if x in self._skill_id2name else u"" for x in row[10:]]
            result_dict[no] = (seed1, result_num, amulet_names, skill_names)

        return result_dict

    def select_seed1s_from_skill_pos_ids(self, amulet_id, skill1_id, skill_pos, key_alchemy):
        u""" 指定されたお守りId,スキルIdを指定された枠に持つSeed1を返す """
        if (key_alchemy == mh4constnumbers.KEY_TENUN555 or
            key_alchemy == mh4constnumbers.KEY_TENUN888):
            table_name = NAME.format(alchemy_type=key_alchemy)
        else:
            raise NotImplementedError(u"this alchemy type is not supported")

        skill_col = COL_SKILL1_LIST[skill_pos]
        amulet_col = COL_AMULET_LIST[skill_pos]
        sql = SELECT_SEED1_BY_SKILL1_SQL.format(table_name=table_name,
                skill1_col=skill_col, skill1_id=skill1_id,
                amulet_col=amulet_col, amulet_id=amulet_id)
        self._cursor.execute(sql)
        return set([x[0] for x in self._cursor.fetchall()])

