# -*- coding: utf-8 -*-

# database のアクセス用スクリプト
# 2013/12/04 written by kei9 

import sqlite3
import sys

import db_supports
import mh4constnumbers
from skilltable import SkillTableAccessor
from amulettable import AmuletTableAccessor
from skillminmaxtable import SkillMinMaxTableAccessor
from seed2skill2table import Seed2Skill2TableAccessor
from seed2tablenumbertable import Seed2TableNumberTableAccessor
from seed2thresholdtable import Seed2ThresholdTableAccessor
from seed2inishietable import Seed2InishieTableAccessor
from sufficienttable import SufficientTableAccessor
from seed1tenuntable import Seed1TenunTableAccessor
from alchemysimulator import AlchemySimulator
from amuletsearcher import AmuletSearcher

class DataBaseAccessor(object):
    u""" this is access class to database """
    def __init__(self, db_name):
        u""" db_name is database name to access """
        self._db_name = db_name
        self._connect = sqlite3.connect(self._db_name)
        #self._connect.text_factory = unicode
        #self._connect.text_factory = str
        self._cursor = self._connect.cursor()

        self._acc_skill = SkillTableAccessor(self._cursor)
        self._acc_amulet = AmuletTableAccessor(self._cursor)
        self._acc_minmax = SkillMinMaxTableAccessor(self._cursor)
        self._acc_seed2skill2 = Seed2Skill2TableAccessor(self._cursor)
        self._acc_seed2no = Seed2TableNumberTableAccessor(self._cursor)
        self._acc_seed2threshold = Seed2ThresholdTableAccessor(self._cursor)
        self._acc_seed2inishie = Seed2InishieTableAccessor(self._cursor)
        self._acc_sufficient = SufficientTableAccessor(self._cursor)
        self._acc_seed1tenun = Seed1TenunTableAccessor(self._cursor)
        self._simulator = AlchemySimulator(self._cursor)
        self._amulet_searcher = AmuletSearcher(self._cursor)

        self._minmax_dict_by_name = self._acc_minmax.get_minmax_by_name()

    def select_seed2s(self, amu_name2skill_names):
        u""" お守り名とスキル名のリストの辞書からSeed2を特定する 
        skill_names: (skill_name1, skill_name2, ...., skill_name7)
        不明であればNoneあるいはNO_DATAを入れておくものとする。
        """
        return self._acc_seed2skill2.select_seed2s_by_name(amu_name2skill_names)

    def select_seed1s(self, amu_skill_name_list, alchemy_type):
        u""" お守り名とスキル名のリスト、および錬金の種類からseed1を返す
        amu_skill_name_list: [(amu_name, skill_name), ....]"""
        return self._acc_seed1tenun.select_seed1s_from_names(amu_skill_name_list, alchemy_type)

    def get_minmax_dict(self, copied=False):
        u""" お守り名と第1．第2スキル名ごとのの最大値を関連付けた辞書を返す
        return {amulet_name:({skill1_name:(min1,max1)}, {skill2_name:(min2,max2)})}"""
        if copied is True:
            return self._minmax_dict_by_name.copy()
        else:
            return self._minmax_dict_by_name

    def get_id_sorted_amulet_names(self, is_desc=False):
        u""" お守りId順でソートされたお守り名を返す(is_desc=Trueで降順) """
        return self._acc_amulet.get_id_sorted_names(is_desc)

    def get_id_sorted_skill_names(self, is_desc=False):
        u""" スキルId順でソートされたスキル名を返す(is_desc=Trueで降順) """
        return self._acc_skill.get_id_sorted_names(is_desc)

    def select_threshold_from_sufficient(self, amulet_name, sufficient_val):
        u""" お守り名と充足値から、スロットごとの判定値を得る
            return (slot1_th, slot2_th, slot3_th)
        """
        return self._acc_sufficient.select_thresholds_by_name(amulet_name, sufficient_val)

    def select_names_from_seed2(self, seed2):
        u""" 指定されたSeed2に対応するテーブルNo, 通しNo,
        お守り名ごとの第2スキル名(1-7枠)、
        判定値1(1-7枠)、判定値2(1-7枠)を返す
        return: table_no, no, {amu_name:(skill2_names)}, threshold1s, threshold2s
        """
        #table no, no
        table_no, no = self._acc_seed2no.get_table_no_from_seed2(seed2)
        # skill2 dict
        skill_dict = self._acc_seed2skill2.select_skill_names_by_seed2(seed2)
        # threshold1s
        th1s = self._acc_seed2threshold.select_thresholds_from_seed2(seed2, 1)
        # threshold2s
        th2s = self._acc_seed2threshold.select_thresholds_from_seed2(seed2, 2)

        return table_no, no, skill_dict, th1s, th2s

    def select_inishie_skill2_from_seed2(self, seed2):
        u""" get inishie's (skill2_name, threshold1, threshold2) from seed2 value """
        return self._acc_seed2inishie.select_names_from_seed2(seed2)

    def select_names_from_seed1(self, seed1, key_alchemy):
        u""" seed1に対応する(no, table_no, result_num, 
        (amulet_name1, ..., amulet_name7), (skill1_name1, ..., skill1_name7))
        の辞書を得る。存在しない場所は空文字列で置き換えてある。
        """
        return self._acc_seed1tenun.select_names_from_seed1(seed1, key_alchemy)

    def select_table_nos_from_seed1(self, seed1, key_alchemy):
        u""" Seed1と錬金の種類から通しNo,テーブルNo,お守り個数を返す。
        return (no, table_no, result_num)"""
        return self._acc_seed1tenun.select_table_nos_from_seed1(seed1, key_alchemy)

    def select_near_seed1s_from_table_no(self, no, table_no, smaller_num, larger_num, alchemy_type):
        u""" 通し番号とテーブル番および錬金の種類からその周囲のseed1の通し番号、テーブルNo.、
        お守り個数, お守り名(1-7)、スキル名(1-7)を辞書として返す。
        ただし辞書のキーは通し番号とする。
        値が存在しない場所は空文字列で埋める。
        return {no: (seed1, result_num, (amulet_names), (skill_names))}"""
        return self._acc_seed1tenun.select_near_names_from_seed1(no, table_no, smaller_num, larger_num, alchemy_type)

    def get_sufficient_value(self, amulet_name, skill1_name, skill2_name, skill1_val, skill2_val):
        u""" 充足値を計算する。
        指定したスキルが見つからない場合や、充足値を計算できない場合はNoneを返す 
        skill2_nameが存在しないスキルであればskill1のみから計算する
        return sufficient_val
        """
        if amulet_name in self._minmax_dict_by_name:
            skill1_minmax, skill2_minmax = self._minmax_dict_by_name[amulet_name]
            if skill1_name in skill1_minmax:
                min1, max1 = skill1_minmax[skill1_name]
                skill1_val = 0 if skill1_val < 0 else skill1_val
            else:
                # cannot find skill1
                return None

            if skill2_name in skill2_minmax:
                min2, max2 = skill2_minmax[skill2_name]
                skill2_val = 0 if skill2_val < 0 else skill2_val
            else:
                max2 = 1
                skill2_val = 0

            if skill1_val > max1 or skill2_val > max2:
                # over value
                return None

            suff_val = (10 * skill1_val) // max1 + (10 * skill2_val) // max2
            return suff_val
        else:
            return None

    u""" お守り検索 """
    def simple_select_seed2s_from_names(self, 
            amulet_name, skill1_name, skill1_val, skill2_name, skill2_val, slot_num):
        u""" 指定されたスキルId,スキル値、スロット数を元にして、
        Seed1との組み合わせを無視したSeed2候補の簡易検索を行う
        return (suff_val, threshold, th1_seed2s, th2_seed2s)"""
        return self._amulet_searcher.simple_select_seed2s_from_names(
                amulet_name, skill1_name, skill1_val, skill2_name, skill2_val, slot_num)

    def select_seed1s_from_fixed_seed2(self,
            seed2, amulet_name, skill1_name, skill1_val, skill2_name, skill2_val, slot_num):
        u""" 指定されたSeed2, スキルId、スキル値、スロット数において、
        条件を満たすお守りが発生するSeed1を検索する
        return (seed1s_555, seed1s_888)"""
        return self._amulet_searcher.select_seed1s_from_fixed_seed2_names(
                seed2, amulet_name, skill1_name, skill1_val, skill2_name, skill2_val, slot_num)

    def select_seed1s_from_names(self, 
            amulet_name, skill1_name, skill1_val, skill2_name, skill2_val, slot_num):
        u""" 指定されたお守り名、スキル名、スキル値、スロット数において、
        条件を満たすお守りが発生するSeed2とSeed1のペアを検索する。
        return {seed2:( seed1s_555, seed1s_888)} """
        u"""
        return self._amulet_searcher.select_seed1s_from_names(amulet_name, 
                skill1_name, skill1_val, skill2_name, skill2_val, slot_num)
        """
        raise NotImplementedError(u"Too heavy and long to calculate!")

    u""" 錬金シミュレーター """
    def simulate_nazo(self, seed1, seed2):
        u""" なぞの錬金
        return (skill1_name, skill1_val, skill2_name, skill2_val, slot_num, type_name)
        """
        return self._simulator.alchemy_nazo_name(seed1, seed2)

    def simulate_komyou(self, seed1, seed2):
        u""" 光明の錬金
        return (skill1_name, skill1_val, skill2_name, skill2_val, slot_num, type_name)
        """
        return self._simulator.alchemy_komyou_name(seed1, seed2)

    def simulate_inishie(self, seed1, seed2):
        u""" いにしえの錬金
        return (skill1_name, skill1_val, skill2_name, skill2_val, slot_num, type_name)
        """
        return self._simulator.alchemy_inishie_name(seed1, seed2)

    def simulate_tenun(self, seed1, seed2, alchemy_key):
        u""" 天運の錬金
        return list of (skill1_name, skill1_val, skill2_name, skill2_val, slot_num, type_name)
        """
        return self._simulator.alchemy_tenun_name(seed1, seed2, alchemy_key)

    def close(self):
        u""" close database accessor """
        self._cursor.close()
        self._connect.close()

if __name__ == "__main__":
    db = DataBaseAccessor("test.sqlite3")
    #db._print_dicts()
    hikari = [None, 57, None, None, None, None, None]
    huru = [54, None, None, None, None, None, None]
    #yuga = [None, None, 98, 75, None, None, None]
    yuga = [None, None, 98, None, None, None, None]
    dic = {2:hikari, 3:huru, 4:yuga}
    #seeds = db.select_seeds(dic)
    seeds = set([58241, 176])
    skill_dic, slot_dic = db.select_skills_from_seeds(seeds)

    print "seeds: ", seeds
    print "amu_id to seed2skill dict: ", skill_dic
    print "seed2slot dict", slot_dic
    db.close()
