# -*- coding: utf-8 -*-

# GUIのメインフレームにおけるお守り検索Notebookのview
# 2013/12/15 written by kei9
import wx
from wx import xrc
import wx.grid

import constnumbers

class NoteBookAmuletView():
    u""" メインのフレームのお守り検索タブ """
    def __init__(self, frame):
        self.frame = frame
        self._init_view()
        self._init_event()
        self.update_filter_radio()
        self._last_search_type = None

    def _init_view(self):
        # initialize view

        # button
        self.button_search = xrc.XRCCTRL(self.frame, "ButtonAmuletSearchSearch")
        self.button_clear = xrc.XRCCTRL(self.frame, "ButtonAmuletSearchClear")
        self.button_simulate = xrc.XRCCTRL(self.frame, "ButtonAmuletToSimulator")
        self.ID_BUTTON_SEARCH = xrc.XRCID("ButtonAmuletSearchSearch")
        self.ID_BUTTON_CLEAR = xrc.XRCID("ButtonAmuletSearchClear")
        self.ID_BUTTON_SIMULATE = xrc.XRCID("ButtonAmuletToSimulator")

        # radio button
        self.radiobox_amulet = xrc.XRCCTRL(self.frame, "RadioBoxAmulet")
        self.radiobox_search_type = xrc.XRCCTRL(self.frame, "RadioBoxSearchType")
        self.radiobox_filter = xrc.XRCCTRL(self.frame, "RadioBoxFilter")

        # seeds grid
        self.grid_seeds = xrc.XRCCTRL(self.frame, "GridAmuletSeeds")
        self.grid_seeds.CreateGrid(1, constnumbers.NUM_AMULET_SEEDS_GRID_COL)
        self.grid_seeds.SetColLabelValue(0, constnumbers.LABEL_FORMAT_AMULET_SEEDS_GRID_COL1)
        self.grid_seeds.SetColLabelValue(1, constnumbers.LABEL_FORMAT_AMULET_SEEDS_GRID_COL2)
        self.grid_seeds.DisableDragGridSize()
        self.grid_seeds.EnableEditing(False)
        self.grid_seeds.SetRowLabelSize(30)
        self.grid_seeds.SetSelectionMode(wx.grid.Grid.wxGridSelectRows)

        # spin ctrl
        self.spin_ctrl_skill1_val = xrc.XRCCTRL(self.frame, "SpinCtrlAmuletSearchSkill1Value")
        self.spin_ctrl_skill2_val = xrc.XRCCTRL(self.frame, "SpinCtrlAmuletSearchSkill2Value")
        self.spin_ctrl_slot_num = xrc.XRCCTRL(self.frame, "SpinCtrlAmuletSearchSlotNumber")
        self.ID_SPIN_CTRL_SKILL1_VAL = xrc.XRCID("SpinCtrlAmuletSearchSkill1Value")
        self.ID_SPIN_CTRL_SKILL2_VAL = xrc.XRCID("SpinCtrlAmuletSearchSkill2Value")

        # combobox
        self.combo_box_skill1 = xrc.XRCCTRL(self.frame, "ComboBoxAmuletSearchSkill1")
        self.combo_box_skill2 = xrc.XRCCTRL(self.frame, "ComboBoxAmuletSearchSkill2")
        self.ID_COMBO_BOX_SKILL1 = xrc.XRCID("ComboBoxAmuletSearchSkill1")
        self.ID_COMBO_BOX_SKILL2 = xrc.XRCID("ComboBoxAmuletSearchSkill2")

        # text ctrl
        self.text_ctrl_result = xrc.XRCCTRL(self.frame, "TextCtrlAmuletSearchResult")
        self.text_ctrl_seed2 = xrc.XRCCTRL(self.frame, "TextCtrlFixedSeed2")

        # set min & max of slot
        self.set_slot_minmax(constnumbers.SLOT_MIN, constnumbers.SLOT_MAX)

    def _init_event(self):
        # initialize event
        self.grid_seeds.Bind(wx.EVT_SIZE, self.OnSeedGridSizeChanged)
        self.combo_box_skill1.Bind(wx.EVT_COMBOBOX, self.OnComboSkill1Changed)
        self.combo_box_skill2.Bind(wx.EVT_COMBOBOX, self.OnComboSkill2Changed)
        self.radiobox_amulet.Bind(wx.EVT_RADIOBOX, self.OnRadioAmuletChanged)
        self.radiobox_search_type.Bind(wx.EVT_RADIOBOX, self.OnRadioSearchTypeChanged)
        self.radiobox_filter.Bind(wx.EVT_RADIOBOX, self.OnRadioFilterChanged)

    def OnSeedGridSizeChanged(self, event):
        u""" SeedGridサイズの変更時 """
        width, height = self.grid_seeds.GetParent().GetSize()
        each_width = (width-40) / (constnumbers.NUM_AMULET_SEEDS_GRID_COL) # consider margin size
        #self.grid_seeds.SetRowLabelValue(0, u"")
        for col in range(constnumbers.NUM_AMULET_SEEDS_GRID_COL):
            self.grid_seeds.SetColSize(col, each_width)

    def OnComboSkill1Changed(self, event):
        u""" スキル1のコンボボックスの変更時 """
        self.update_skill_minmax(1)

    def OnComboSkill2Changed(self, event):
        u""" スキル2のコンボボックスの変更時 """
        self.update_skill_minmax(2)

    def OnRadioAmuletChanged(self, event):
        u""" お守り選択のラジオボタン変更時 """
        self.update_skill_names()
        self.update_skill_minmax()

    def OnRadioSearchTypeChanged(self, event):
        u""" お守り選択のラジオボタン変更時 """
        search_type = self.get_selected_search_type()
        if search_type == constnumbers.SEARCH_TYPE_FIXED_SEED2:
            self.text_ctrl_seed2.Enable()
        else:
            self.text_ctrl_seed2.Disable()

    def OnRadioFilterChanged(self, event):
        u""" 結果表示のフィルターラジオボックスの変更時 """
        if self._last_search_type == None:
            pass
        elif self._last_search_type == constnumbers.SEARCH_TYPE_SEED2:
            self._set_items_dict_only_seed2()
        elif self._last_search_type == constnumbers.SEARCH_TYPE_FIXED_SEED2:
            self._set_items_dict_fixed_seed2()
        else:
            raise NotImplementedError("irregular radio button selected")

    def bind_radiobox_search_type(self, event_func, evt=wx.EVT_RADIOBOX):
        u""" 判定値の種類のラジオボタンが押された時のイベントをセットする """
        self.radiobox_search_type.Bind(evt, event_func)

    def bind_button_search(self, event_func, evt=wx.EVT_BUTTON):
        u""" 検索ボタンが押された時のイベントをセットする """
        self.button_search.Bind(evt, event_func)

    def bind_button_clear(self, event_func, evt=wx.EVT_BUTTON):
        u""" クリアボタンが押された時のイベントをセットする """
        self.button_clear.Bind(evt, event_func)

    def bind_button_simulate(self, event_func, evt=wx.EVT_BUTTON):
        u""" シミュレーター移動ボタンが押された時のイベントをセットする """
        self.button_simulate.Bind(evt, event_func)

    def set_skillminmax_dict(self, skill_names, amulet_names, minmax_dict):
        u"""スキル名のリスト、お守り名のリスト、
        そのお守りに対応する第1,2スキルの最大最小の辞書をセットする
        {amulet_name:({skill1_name:(min1,max1)}, {skill2_name:(min2,max2)})}
        """
        self._skill_names = skill_names
        self._amulet_names = amulet_names
        self._minmax_dict = minmax_dict
        self.set_selected_amulet(constnumbers.NAME_AMULET1)
        self.update_skill_names()
        self.update_skill_minmax()

    def set_skillnames_ordered(self, skillnames):
        u""" 順番に並べられたスキル名のシーケンスをセットする """
        self._skillnames = skillnames

    def update_skill_names(self):
        u""" 選択されたお守り名に応じてスキル名のリストを更新する """
        amu_name = self.get_selected_amulet()
        skill1_minmax, skill2_minmax = self._minmax_dict[amu_name]
        self.combo_box_skill1.SetItems([x for x in self._skill_names if x in skill1_minmax])
        self.combo_box_skill2.SetItems([constnumbers.VAL_NO_SKILL] + 
                [x for x in self._skill_names if x in skill2_minmax])
        self.combo_box_skill1.SetSelection(0)
        self.combo_box_skill2.SetSelection(0)

    def update_skill_minmax(self, index=None):
        u""" 選択されたスキル名に応じて値の範囲を設定する """
        amu_name = self.get_selected_amulet()
        skill1_minmax, skill2_minmax = self._minmax_dict[amu_name]

        if index is None or index == 1:
            skill1_name = self.combo_box_skill1.GetStringSelection()
            if skill1_name in skill1_minmax:
                min1, max1 = skill1_minmax[skill1_name]
                self.spin_ctrl_skill1_val.SetRange(min1, max1)
            else:
                self.spin_ctrl_skill1_val.SetRange(0, 0)

        if index is None or index == 2:
            skill2_name = self.combo_box_skill2.GetStringSelection()
            if skill2_name in skill2_minmax:
                min2, max2 = skill2_minmax[skill2_name]
                self.spin_ctrl_skill2_val.SetRange(min2, max2)
            else:
                self.spin_ctrl_skill2_val.SetRange(0, 0)

    def update_filter_radio(self):
        u""" 検索方式に応じて結果表示のフィルターを変える """
        search_type = self.get_selected_search_type()
        if search_type == constnumbers.SEARCH_TYPE_SEED2:
            self.radiobox_filter.SetItemLabel(0, constnumbers.SEARCH_TYPE_SEED2_FILTER1)
            self.radiobox_filter.SetItemLabel(1, constnumbers.SEARCH_TYPE_SEED2_FILTER2)
            self.radiobox_filter.SetItemLabel(2, constnumbers.SEARCH_TYPE_SEED2_FILTER3)
            self.grid_seeds.SetColLabelValue(0, constnumbers.SEARCH_TYPE_SEED2_COL1)
            self.grid_seeds.SetColLabelValue(1, constnumbers.SEARCH_TYPE_SEED2_COL2)
        elif search_type == constnumbers.SEARCH_TYPE_FIXED_SEED2:
            self.radiobox_filter.SetItemLabel(0, constnumbers.SEARCH_TYPE_FIXED_SEED2_FILTER1)
            self.radiobox_filter.SetItemLabel(1, constnumbers.SEARCH_TYPE_FIXED_SEED2_FILTER2)
            self.radiobox_filter.SetItemLabel(2, constnumbers.SEARCH_TYPE_FIXED_SEED2_FILTER3)
            self.grid_seeds.SetColLabelValue(0, constnumbers.SEARCH_TYPE_FIXED_SEED2_COL1)
            self.grid_seeds.SetColLabelValue(1, constnumbers.SEARCH_TYPE_FIXED_SEED2_COL2)
        else:
            raise NotImplementedError("irregular radio button selected")

    def set_slot_minmax(self, slot_min, slot_max):
        u""" スロットの最大値と最小値をセットする """
        self.spin_ctrl_slot_num.SetRange(slot_min, slot_max)

    def get_skill_values(self):
        u""" 第1,2スキルの値を得る 
        return (skill1_val, skill2_val)"""
        skill1_val = self.spin_ctrl_skill1_val.GetValue()
        skill2_val = self.spin_ctrl_skill2_val.GetValue()
        return (skill1_val, skill2_val)

    def get_slot_value(self):
        u""" スロットの値を得る """
        return self.spin_ctrl_slot_num.GetValue()

    def clear_input_values(self):
        u""" 入力された値を初期化する """
        self.combo_box_skill1.SetSelection(0)
        self.combo_box_skill2.SetSelection(0)
        self.spin_ctrl_skill1_val.SetValue(1)
        self.spin_ctrl_skill2_val.SetValue(0)
        self.spin_ctrl_slot_num.SetValue(1)
        self.text_ctrl_seed2.SetValue(u"")
        self.text_ctrl_result.SetValue(u"")

    def get_fixed_seed2(self):
        u""" seed2固定検索時のSeed2を返す """
        seed2 = self.text_ctrl_seed2.GetValue()
        if seed2.isdigit():
            return int(seed2)
        else:
            return None

    def set_selected_amulet(self, value, amulet_name=constnumbers.NAME_AMULET1):
        u""" ラジオボタンの値をセットする 
            arg: (value, amulet_name)
        """
        if amulet_name == constnumbers.NAME_AMULET1:
            idx = 0
        elif amulet_name == constnumbers.NAME_AMULET2:
            idx = 1
        elif amulet_name == constnumbers.NAME_AMULET3:
            idx = 2
        self.radiobox_amulet.SetSelection(idx)

    def get_selected_amulet(self):
        u""" 選択されているお守り名を得る 
            return: (NAME_OF_AMULET)
        """
        idx = self.radiobox_amulet.GetSelection()
        if idx == 0:
            return constnumbers.NAME_AMULET1
        elif idx == 1:
            return constnumbers.NAME_AMULET2
        elif idx == 2:
            return constnumbers.NAME_AMULET3
        else:
            raise NotImplementedError("irregular radio button selected")

    def get_selected_search_type(self):
        u""" 選択されているラジオボタンの検索種類IDを得る
            return: (THRESHOLD_TYPE_ID)
        """
        idx = self.radiobox_search_type.GetSelection()
        if idx == 0:
            return constnumbers.SEARCH_TYPE_SEED2
        elif idx == 1:
            return constnumbers.SEARCH_TYPE_FIXED_SEED2
        elif idx == 2:
            return constnumbers.SEARCH_TYPE_SEED1_SEED2
        else:
            raise NotImplementedError("irregular radio button selected")

    def get_selected_tenun_type(self):
        u""" 固定Seed2における天運の種類を返す。判別できない場合はNone """
        if self._last_search_type == constnumbers.SEARCH_TYPE_FIXED_SEED2:
            filter_str = self.radiobox_filter.GetStringSelection()
            if filter_str == constnumbers.SEARCH_TYPE_FIXED_SEED2_FILTER1:
                return constnumbers.KEY_TENUN888
            elif filter_str == constnumbers.SEARCH_TYPE_FIXED_SEED2_FILTER2:
                return constnumbers.KEY_TENUN555
            elif filter_str == constnumbers.SEARCH_TYPE_FIXED_SEED2_FILTER3:
                return constnumbers.KEY_TENUN888
        else:
            return None

    def get_skill_names(self):
        u""" skill1, skill2で選択されているスキル名を返す
            return (skill1_name, skill2_name)"""
        skill1_name = self.combo_box_skill1.GetStringSelection()
        skill2_name = self.combo_box_skill2.GetStringSelection()
        return (skill1_name, skill2_name)

    def set_skill_names(self, skill1_names, skill2_names):
        u""" スキル1、スキル2のコンボボックスの一覧をセットする """
        self.combo_box_skill1.SetItems(skill1_names)
        self.combo_box_skill2.SetItems(skill2_names)

    def set_simualte_button_enable(self, value):
        u""" 対応スキル検索ボタンの有効・無効の切り替え """
        if value is True:
            self.button_simulate.Enable()
        else:
            self.button_simulate.Disable()

    def set_result(self, value):
        u""" 結果表示用のTextCtrlに値をセットする """
        self.text_ctrl_result.SetValue(value)

    def set_result_only_seed2(self, suff_val, threshold, th1_seed2s, th2_seed2s):
        u""" Seed2のみからの検索結果表示用に値をセットする """
        self.update_filter_radio()
        seed2s = th1_seed2s | th2_seed2s
        self.text_ctrl_result.SetValue(u"""※簡易検索のため、不正確な場合があります
充足値: {0}, 必要な判定値: {1}
{1}以上の判定値1を持つSeed2: {2}
{1}以上の判定値2を持つSeed2: {3}
{1}以上の判定値を持つSeed2: {4}""".format(
            suff_val, threshold, len(th1_seed2s), len(th2_seed2s), len(seed2s)))
        self._last_search_type = constnumbers.SEARCH_TYPE_SEED2
        self._last_values = (suff_val, threshold, th1_seed2s, th2_seed2s)
        self._set_items_dict_only_seed2()

    def _set_items_dict_only_seed2(self):
        u""" Seed2のみからの検索時結果をフィルターに応じて表示 """
        if self._last_search_type == constnumbers.SEARCH_TYPE_SEED2:
            filter_str = self.radiobox_filter.GetStringSelection()
            if filter_str == constnumbers.SEARCH_TYPE_SEED2_FILTER1:
                th1_include, th2_include = True, True
            elif filter_str == constnumbers.SEARCH_TYPE_SEED2_FILTER2:
                th1_include, th2_include = True, False
            elif filter_str == constnumbers.SEARCH_TYPE_SEED2_FILTER3:
                th1_include, th2_include = False, True

            suff_val, threshold, th1_seed2s, th2_seed2s = self._last_values
            seed2s = th1_seed2s | th2_seed2s
            items_dict = {}
            for seed2 in seed2s:
                texts = []
                is_include = False
                if seed2 in th1_seed2s:
                    texts.append(u"判定値1")
                    if th1_include:
                        is_include = True
                if seed2 in th2_seed2s:
                    texts.append(u"判定値2")
                    if th2_include:
                        is_include = True
                if is_include:
                    texts = u", ".join(texts)
                    texts = u"Seed2({0})".format(texts) if len(texts) > 0 else texts
                    items_dict[seed2] = texts
            self.set_grid_seeds_items(items_dict)

    def set_result_fixed_seed2(self, suff_val, threshold, seed2, seed1s_555, seed1s_888):
        u""" 固定Seed2からの検索結果表示用に値をセットする """
        self.update_filter_radio()
        seed1s = seed1s_555 | seed1s_888
        self.text_ctrl_result.SetValue(u"""Seed2:\t{0}
充足値: {1}, 必要な判定値: {2}
天運555におけるSeed1:\t{3}
天運888におけるSeed1:\t{4}
Seed1の合計:\t{5}""".format(seed2, suff_val, threshold, len(seed1s_555), len(seed1s_888), len(seed1s)))

        self._last_search_type = constnumbers.SEARCH_TYPE_FIXED_SEED2
        self._last_values = (suff_val, threshold, seed2, seed1s_555, seed1s_888)
        self._set_items_dict_fixed_seed2()

    def _set_items_dict_fixed_seed2(self):
        u""" 固定Seed2からの検索時結果をフィルターに応じて表示 """
        if self._last_search_type == constnumbers.SEARCH_TYPE_FIXED_SEED2:
            filter_str = self.radiobox_filter.GetStringSelection()
            if filter_str == constnumbers.SEARCH_TYPE_FIXED_SEED2_FILTER1:
                include_555, include_888 = True, True
            elif filter_str == constnumbers.SEARCH_TYPE_FIXED_SEED2_FILTER2:
                include_555, include_888 = True, False
            elif filter_str == constnumbers.SEARCH_TYPE_FIXED_SEED2_FILTER3:
                include_555, include_888 = False, True

            items_dict = {}
            suff_val, threshold, seed2, seed1s_555, seed1s_888 = self._last_values
            seed1s = seed1s_555 | seed1s_888
            for seed1 in seed1s:
                texts = []
                is_include = False
                if seed1 in seed1s_555:
                    texts.append(u"天運555")
                    if include_555:
                        is_include = True
                if seed1 in seed1s_888:
                    texts.append(u"天運888")
                    if include_888:
                        is_include = True
                if is_include:
                    texts = u", ".join(texts)
                    texts = u"Seed1({0})".format(texts) if len(texts) > 0 else texts
                    items_dict[seed1] = texts
            self.set_grid_seeds_items(items_dict)

    def set_result_seeds(self, suff_val, result_dict):
        u""" 全検索の結果を表示する。
        Seed2から各天運のSeed1の集合の辞書を引数とする。
        result_dict={seed2:(seed1s_555, seed1s_888)}"""
        u"""
        self.text_ctrl_result.SetValue(u"充足値: {0} Seed2の総数:\t{1}".format(suff_val, len(result_dict)))
        items_dict = {}
        for seed2, (seed1s_555, seed1s_888) in result_dict.items():
            seed1s = seed1s_555 | seed1s_888
            for seed1 in seed1s:
                texts = []
                if seed1 in seed1s_555:
                    texts.append(u"天運555")
                if seed1 in seed1s_888:
                    texts.append(u"天運888")
                texts = u", ".join(texts)
                texts = u"Seed1({0})".format(texts) if len(texts) > 0 else texts
                items_dict[(seed1, seed2)] = texts
        self.set_grid_seeds_items(items_dict)
        """
        raise NotImplementedError(u"give up to implement all search because of ttoo heavy and long calc")

    def set_grid_seeds_items(self, items_dict):
        u""" Seed一覧のGridに値をセットする。
        引数はSeed -> 文字列の辞書とする。
        Noneが与えられた時はClearする"""
        if items_dict is not None and len(items_dict) > 0:
            row_num = len(items_dict)
            diff = row_num - self.grid_seeds.GetNumberRows()
            if diff > 0:
                self.grid_seeds.AppendRows(diff, False)
            elif diff < 0:
                self.grid_seeds.DeleteRows(0, -diff, False)

            for row, seed in enumerate(sorted(items_dict.keys())):
                #self.grid_seeds.SetRowLabelValue(row, u"")
                type_str = items_dict[seed]
                self.grid_seeds.SetCellValue(row, 0, u"{0}".format(seed))
                self.grid_seeds.SetCellValue(row, 1, type_str)
            self.set_simualte_button_enable(True)
        else:
            self.clear_grid_seeds()
            self.set_simualte_button_enable(False)

    def get_selected_seeds(self):
        u""" Seed一覧のGridの選択された(Seed1,Seed2)の値を得る。"""
        rows = [x for x in self.grid_seeds.GetSelectedRows()]
        if len(rows) > 0:
            if self._last_search_type == constnumbers.SEARCH_TYPE_FIXED_SEED2:
                seed1 = int(self.grid_seeds.GetCellValue(rows[0], 0))
                seed2 = self._last_values[2]
                return seed1, seed2
            elif self._last_search_type == constnumbers.SEARCH_TYPE_SEED2:
                seed1 = None
                seed2 = int(self.grid_seeds.GetCellValue(rows[0], 0))
                return seed1, seed2
            else:
                return None
        else:
            return None

    def set_grid_seeds_selected_idx(self, idx):
        u""" Seed一覧のGridの選択された行を選択状態にする。"""
        if idx >= 0 and idx < self.grid_seeds.GetNumberRows():
            self.grid_seeds.MakeCellVisible(idx, 0)
            self.grid_seeds.SelectRow(idx)

    def clear_grid_seeds(self):
        u""" Seed一覧のGridから値をClearする"""
        for row in range(self.grid_seeds.GetNumberRows()):
            self.grid_seeds.SetRowLabelValue(row, u"")
            for col in range(self.grid_seeds.GetNumberCols()):
                self.grid_seeds.SetCellValue(row, col, u"")
        self.grid_seeds.ClearSelection()
        self.set_simualte_button_enable(False)
        self._last_search_type = None
        self._last_valuse = None

    def clear_grid(self):
        u"""Gridの値をクリアする"""
        self.clear_grid_seeds()
