# -*- coding: utf-8 -*-

# databaseの生成用スクリプト
# 2013/12/09 written by kei9 

import sqlite3
import csv
import os.path
import os
import sys
import tempfile
import zipfile
import cStringIO

import db_supports
import mh4constnumbers
import skilltable
import amulettable
import skillminmaxtable
import seed2skill2table
import seed2tablenumbertable
import seed2thresholdtable
import seed2inishietable
import sufficienttable
import seed1tenuntable

class DataBaseGenerator(object):
    u""" this is generate class of database """
    def __init__(self, db_name=None):
        if db_name is None:
            self._db_name = ":memory:"
        else :
            self._db_name = db_name
        self._tmp_db = None
        self._dict_skill_id = {}
        self._dict_amulet_id = {}

    def generate_db(self, over_write=True):
        u""" DBを作成する関数。over_writeがTrueで既存のDBがある場合は上書きする """
        if self._db_name == ":memory:":
            # nothing to do
            pass
        elif os.path.exists(self._db_name) and os.path.isfile(self._db_name):
            if over_write is False:
                return
            with tempfile.NamedTemporaryFile(delete=False) as tmp_file:
                self._tmp_db = tmp_file.name

        self._open()

        # read data from zip file
        self._zipfile = zipfile.ZipFile(db_supports.ZIP_FILE, "r")

        # db generation
        self._create_skill_table()
        self._create_amulet_table()
        self._create_min_max_table()
        self._create_seed2_skill2_table()
        self._create_seed2_threshold_table()
        self._create_seed2_inishie_table()
        self._create_sufficient_value_table()
        self._create_seed1_tenun_table()

        self._zipfile.close()
        self._close()

        # rename operation of tmp file
        if self._tmp_db is not None:
            with tempfile.NamedTemporaryFile(delete=True) as tmp_file:
                tmp_name = tmp_file.name
            os.rename(self._db_name, tmp_name)
            os.rename(self._tmp_db, self._db_name)
            os.remove(tmp_name)

    def _open(self):
        if self._tmp_db is None:
            self._connect = sqlite3.connect(self._db_name)
        else:
            self._connect = sqlite3.connect(self._tmp_db)
        self._connect.text_factory = str   # for usage of utf-8
        self._cursor = self._connect.cursor()

    def _close(self):
        self._connect.commit()
        self._cursor.close()
        self._connect.close()

    def _get_skill_id(self, skill_name):
        u""" 与えられたスキル名からスキルのIDを得る関数。スキルが存在しなければNoneを返す """
        skill_name = unicode(skill_name, 'utf-8')

        if skill_name in self._dict_skill_id:
            return self._dict_skill_id[skill_name]
        else:
            sql = db_supports.SKILL_TABLE_SKILL2ID_SQL.format(skill_name=skill_name)
            self._cursor.execute(sql)
            skill_ids = []
            for val in self._cursor.fetchall():
                skill_ids.append(val[0])
            if len(skill_ids) < 1:
                return None
            else:
                self._dict_skill_id[skill_name] = skill_ids[0]
                return skill_ids[0]

    def _get_amulet_id(self, amulet_name):
        u""" 与えられたお守り名からお守りのIDを得る関数。スキルが存在しなければNoneを返す """
        amulet_name = unicode(amulet_name, 'utf-8')

        if amulet_name in self._dict_amulet_id:
            return self._dict_amulet_id[amulet_name]
        else:
            sql = db_supports.AMULET_TABLE_AMULET2ID_SQL.format(amulet_name=amulet_name)
            self._cursor.execute(sql)
            amulet_ids = []
            for val in self._cursor.fetchall():
                amulet_ids.append(val[0])
            if len(amulet_ids) < 1:
                return None
            else:
                self._dict_amulet_id[amulet_name] = amulet_ids[0]
                return amulet_ids[0]

    def _create_skill_table(self):
        u"""スキルとIDの組み合わせテーブルを作成する"""
        print "create skill table"
        generator = skilltable.SkillTableGenerator()

        # read from zip file with StringIO wrapper
        f = cStringIO.StringIO(self._zipfile.read(db_supports.SKILL_FILE_NAME, "r"))
        reader = csv.reader(f)  # (id, skillName)
        generator.insert_data(self._cursor, reader)

        self._connect.commit()
        f.close()

    def _create_amulet_table(self):
        u"""お守り名とIDの組み合わせテーブルを作成する"""
        print "create amulet table"
        generator = amulettable.AmuletTableGenerator()
        # read from zip file with StringIO wrapper
        f = cStringIO.StringIO(self._zipfile.read(db_supports.AMULET_FILE_NAME, "r"))
        reader = csv.reader(f)  # (amuleteName)
        generator.insert_data(self._cursor, reader)

        self._connect.commit()
        f.close()

    def _create_min_max_table(self):
        u"""お守り名と対応するスキルの最大最小値の記載されたcsvファイルから
        お守りごとのスキルの最大最小値を記載したテーブルを作成する
        """
        print "load min & max of skill" 
        # create master table of skill min max
        generator = skillminmaxtable.SkillMinMaxTableGenerator()
        # read from zip file with StringIO wrapper
        f = cStringIO.StringIO(self._zipfile.read(db_supports.MIN_MAX_FILE_NAME, "r"))
        reader = csv.reader(f)  # (name, filename of minmax1, filename of minmax2)

        generator.insert_master_data(self._cursor, reader)
        filenames_dict = generator.get_skill_filenames()
        reader_dict, filelist = {}, []
        for key, value in filenames_dict.items():
            fname1, fname2 = value
            if fname1 in self._zipfile.namelist():
                file_minmax1 = cStringIO.StringIO(self._zipfile.read(fname1, "r"))
                reader1 = csv.reader(file_minmax1)  # (name of skill, min1, max1)
                filelist.append(file_minmax1)
            else:
                reader1 = None

            if fname2 in self._zipfile.namelist():
                file_minmax2 = cStringIO.StringIO(self._zipfile.read(fname2, "r"))
                reader2 = csv.reader(file_minmax2)  # (name of skill, min1, max1)
                filelist.append(file_minmax2)
            else:
                reader2 = None
            reader_dict[key] = (reader1, reader2)
        generator.insert_data(self._cursor, reader_dict)

        for _file in filelist:
            _file.close()

        self._connect.commit()
        f.close()

    def _create_seed2_skill2_table(self):
        u"""csvファイルからSEED2から第2スキルへの表を生成する
        """
        print "load Second skill"
        # create master table of seed2 to skill2
        generator = seed2skill2table.Seed2Skill2TableGenerator()
        # read from zip file with StringIO wrapper
        f = cStringIO.StringIO(self._zipfile.read(db_supports.SEED2_SKILL2_FILE_NAME, "r"))
        reader = csv.reader(f)  # (omamori_name, filename of Second_skill)

        generator.insert_master_data(self._cursor, reader)
        filenames_dict = generator.get_skill_filenames()
        reader_dict, filelist = {}, []
        for key, fname in filenames_dict.items():
            if fname in self._zipfile.namelist():
                file_skill2 = cStringIO.StringIO(self._zipfile.read(fname, "r"))
                reader = csv.reader(file_skill2)  # (seed2, skillname1,..., skillname7)
                filelist.append(file_skill2)
                reader_dict[key] = reader
        generator.insert_data(self._cursor, reader_dict)

        for _file in filelist:
            _file.close()

        self._connect.commit()
        f.close()

        self._create_seed2_table_no_table()
        self._connect.commit()

    def _create_seed2_table_no_table(self):
        u""" Seed2に対応するテーブル値を格納するテーブルを作成する """
        generator = seed2tablenumbertable.Seed2TableNumberTableGenerator()

        # read from zip file with StringIO wrapper
        f = cStringIO.StringIO(self._zipfile.read(db_supports.SEED2_TABLE_NO_FILE_NAME, "r"))
        reader = csv.reader(f)  # (seed2, table_no, no)
        generator.insert_data(self._cursor, reader)
        self._connect.commit()
        f.close()

    def _create_seed2_inishie_table(self):
        u""" いにしえの錬金の第2Seed対応テーブルを作成する """
        generator = seed2inishietable.Seed2InishieTableGenerator()

        # read from zip file with StringIO wrapper
        f = cStringIO.StringIO(self._zipfile.read(db_supports.SEED2_INISHIE_FILE_NAME, "r"))
        reader = csv.reader(f)  # (seed2, skill_name, threshold1, threshold2)
        generator.insert_data(self._cursor, reader)
        self._connect.commit()
        f.close()

    def _create_seed2_threshold_table(self):
        u""" csvファイルよりSEED2から判定値1,2へのテーブルを作成する
        """
        print "load Threshold1,2"
        generator = seed2thresholdtable.Seed2ThresholdTableGenerator()

        # for threshold1 table
        # read from zip file with StringIO wrapper
        f = cStringIO.StringIO(self._zipfile.read(db_supports.SEED2_THRESHOLD1_FILE_NAME, "r"))
        reader = csv.reader(f)  # (seed2, threshold1_1, threshold1_2, ..., threshold1_7)
        generator.insert_data(self._cursor, reader, 1)
        f.close()

        # for threshold2 table
        # read from zip file with StringIO wrapper
        f = cStringIO.StringIO(self._zipfile.read(db_supports.SEED2_THRESHOLD2_FILE_NAME, "r"))
        reader = csv.reader(f)  # (seed2, threshold2_1, threshold2_2, ..., threshold2_7)
        generator.insert_data(self._cursor, reader, 2)
        f.close()

        self._connect.commit()

    def _create_sufficient_value_table(self):
        u"""csvファイルから充足値の表を生成する
        """
        print "load Sufficient Value"
        # create master table of sufficient value
        generator = sufficienttable.SufficientTableGenerator()
        # read from zip file with StringIO wrapper
        f = cStringIO.StringIO(self._zipfile.read(db_supports.SUFFICIENT_FILE_NAME, "r"))
        reader = csv.reader(f)  # (omamori_name, filename of sufficient values)

        generator.insert_master_data(self._cursor, reader)
        filenames_dict = generator.get_skill_filenames()
        reader_dict, filelist = {}, []
        for key, fname in filenames_dict.items():
            if fname in self._zipfile.namelist():
                file_suff = cStringIO.StringIO(self._zipfile.read(fname, "r"))
                reader = csv.reader(file_suff)  # (seed2, skillname1,..., skillname7)
                filelist.append(file_suff)
                reader_dict[key] = reader
        generator.insert_data(self._cursor, reader_dict)

        for _file in filelist:
            _file.close()

        self._connect.commit()
        f.close()

    def _create_seed1_tenun_table(self):
        u""" csvファイルより天運の錬金結果からSEED1へのテーブルを作成する
        """
        print "load seed1"
        # for seed1 table
        # tenun555
        generator = seed1tenuntable.Seed1TenunTableGenerator()
        # read from zip file with StringIO wrapper
        f = cStringIO.StringIO(self._zipfile.read(db_supports.SEED1_TENUN555_FILE_NAME, "r"))
        reader = csv.reader(f)
        generator.insert_data(self._cursor, reader, mh4constnumbers.KEY_TENUN555)
        f.close()

        # tenun888
        # read from zip file with StringIO wrapper
        f = cStringIO.StringIO(self._zipfile.read(db_supports.SEED1_TENUN888_FILE_NAME, "r"))
        reader = csv.reader(f)
        generator.insert_data(self._cursor, reader, mh4constnumbers.KEY_TENUN888)
        f.close()
        self._connect.commit()


if __name__ == "__main__":
    #db = DataBaseGenerator(DB_FILE_NAME)
    db = DataBaseGenerator("test.sqlite3")
    #db = DataBaseGenerator()    # for memory
    db.generate_db()

