# -*- coding: utf-8 -*-

u"""
お守り検索を行うクラス
2013/12/20 written by kei9
"""

import sqlite3
import csv

import amulettable
import skilltable
import skillminmaxtable
import seed2thresholdtable
import seed2skill2table
import sufficienttable
import seed1tenuntable
import alchemysimulator
import mh4constnumbers

SELECT_SEED2S_FROM_THRESHOLD_AND_SKILL2_SQL = u"""
    select {{threshold_table}}.{threshold_seed2} from {{threshold_table}} 
    inner join {{skill2_table}} on {{threshold_table}}.{threshold_seed2} = {{skill2_table}}.{skill2_seed2}
    where 
    ({{skill2_table}}.{skill2_id1} = {{skill2_id}} and {{threshold_table}}.{threshold_1} >= {{threshold}}) or 
    ({{skill2_table}}.{skill2_id2} = {{skill2_id}} and {{threshold_table}}.{threshold_2} >= {{threshold}}) or 
    ({{skill2_table}}.{skill2_id3} = {{skill2_id}} and {{threshold_table}}.{threshold_3} >= {{threshold}}) or 
    ({{skill2_table}}.{skill2_id4} = {{skill2_id}} and {{threshold_table}}.{threshold_4} >= {{threshold}}) or 
    ({{skill2_table}}.{skill2_id5} = {{skill2_id}} and {{threshold_table}}.{threshold_5} >= {{threshold}}) or 
    ({{skill2_table}}.{skill2_id6} = {{skill2_id}} and {{threshold_table}}.{threshold_6} >= {{threshold}}) or 
    ({{skill2_table}}.{skill2_id7} = {{skill2_id}} and {{threshold_table}}.{threshold_7} >= {{threshold}})
    """.format(
        skill2_seed2=seed2skill2table.COL_SEED2,
        skill2_id1=seed2skill2table.COL_SKILL2_ID1,
        skill2_id2=seed2skill2table.COL_SKILL2_ID2,
        skill2_id3=seed2skill2table.COL_SKILL2_ID3,
        skill2_id4=seed2skill2table.COL_SKILL2_ID4,
        skill2_id5=seed2skill2table.COL_SKILL2_ID5,
        skill2_id6=seed2skill2table.COL_SKILL2_ID6,
        skill2_id7=seed2skill2table.COL_SKILL2_ID7,
        threshold_seed2=seed2thresholdtable.COL_SEED2,
        threshold_1=seed2thresholdtable.COL_THRESHOLD1,
        threshold_2=seed2thresholdtable.COL_THRESHOLD2,
        threshold_3=seed2thresholdtable.COL_THRESHOLD3,
        threshold_4=seed2thresholdtable.COL_THRESHOLD4,
        threshold_5=seed2thresholdtable.COL_THRESHOLD5,
        threshold_6=seed2thresholdtable.COL_THRESHOLD6,
        threshold_7=seed2thresholdtable.COL_THRESHOLD7)


class AmuletSearcher(object):
    u"""お守りの検索を行うクラス"""
    def __init__(self, db_cursor):
        u""" db_cursor: cursor of sqlite3 database """
        self._cursor = db_cursor
        accessor = amulettable.AmuletTableAccessor(db_cursor)
        self._amu_id2name, self._amu_name2id = accessor.get_dict()

        accessor = skilltable.SkillTableAccessor(db_cursor)
        self._skill_id2name, self._skill_name2id = accessor.get_dict()

        accessor = skillminmaxtable.SkillMinMaxTableAccessor(db_cursor)
        self._minmax_dict = accessor.get_minmax_by_id()

        self._acc_seed2skill2 = seed2skill2table.Seed2Skill2TableAccessor(db_cursor)
        self._amu_id2skill2_table = self._acc_seed2skill2.get_amulet_id2table_name_dict()

        self._acc_sufficient = sufficienttable.SufficientTableAccessor(db_cursor)
        self._acc_threshold = seed2thresholdtable.Seed2ThresholdTableAccessor(db_cursor)
        self._acc_seed1tenun = seed1tenuntable.Seed1TenunTableAccessor(db_cursor)
        self._simulator = alchemysimulator.AlchemySimulator(db_cursor)

    def get_sufficient_value(self, amulet_id, skill1_id, skill1_val, skill2_id, skill2_val):
        u""" 充足値を計算する。
        指定したスキルが見つからない場合や、充足値を計算できない場合はNoneを返す 
        skill2_nameが存在しないスキルであればskill1のみから計算する
        return sufficient_val
        """
        skill1_minmax, skill2_minmax = self._minmax_dict[amulet_id]
        if skill1_id in skill1_minmax:
            min1, max1 = skill1_minmax[skill1_id]
            skill1_val = 0 if skill1_val < 0 else skill1_val
        else:
            # cannot find skill1
            return None

        if skill2_id in skill2_minmax:
            min2, max2 = skill2_minmax[skill2_id]
            skill2_val = 0 if skill2_val < 0 else skill2_val
        else:
            max2 = 1
            skill2_val = 0

        if skill1_val > max1 or skill2_val > max2:
            # over value
            return None

        suff_val = (10 * skill1_val) // max1 + (10 * skill2_val) // max2
        return suff_val

    def simple_select_seed2s_from_ids(self, amulet_id, skill1_id, skill1_val, skill2_id, skill2_val, slot_num):
        u""" 指定されたスキルId,スキル値、スロット数を元にして、
        Seed1との組み合わせを無視したSeed2候補の簡易検索を行う
        return (suff_val, threshold, th1_seed2s, th2_seed2s)"""
        suff_val = self.get_sufficient_value(amulet_id, skill1_id, skill1_val, skill2_id, skill2_val)
        if suff_val is not None:
            slot_ths = self._acc_sufficient.select_thresholds_by_id(amulet_id, suff_val)
            threshold = slot_ths[slot_num - 1]
            if skill2_id in self._skill_id2name:
                skill2_table = self._amu_id2skill2_table[amulet_id]
                # for threshold 1
                threshold_table = seed2thresholdtable.NAME.format(index=1)
                sql = SELECT_SEED2S_FROM_THRESHOLD_AND_SKILL2_SQL.format(
                        threshold_table=threshold_table,
                        skill2_table=skill2_table,
                        threshold=threshold,
                        skill2_id=skill2_id)
                self._cursor.execute(sql)
                th1_seed2s = set([x[0] for x in self._cursor.fetchall()])

                # for threshold 2
                threshold_table = seed2thresholdtable.NAME.format(index=2)
                sql = SELECT_SEED2S_FROM_THRESHOLD_AND_SKILL2_SQL.format(
                        threshold_table=threshold_table,
                        skill2_table=skill2_table,
                        threshold=threshold,
                        skill2_id=skill2_id)
                self._cursor.execute(sql)
                th2_seed2s = set([x[0] for x in self._cursor.fetchall()])

            else:
                th1_seed2s = self._acc_threshold.select_seed2s_from_all_threshold(threshold, 1)
                th2_seed2s = self._acc_threshold.select_seed2s_from_all_threshold(threshold, 2)
        else:
            th1_ssed2s, th2_seed2s = set(), set()
            threshold = None

        return (suff_val, threshold, th1_seed2s, th2_seed2s)

    def simple_select_seed2s_from_names(self, 
            amulet_name, skill1_name, skill1_val, skill2_name, skill2_val, slot_num):
        u""" 指定されたお守り名、スキル名,スキル値、スロット数を元にして、
        Seed1との組み合わせを無視したSeed2候補の簡易検索を行う
        return (suff_val, threshold, th1_seed2s, th2_seed2s)"""
        amulet_id = self._amu_name2id[amulet_name]
        skill1_id = self._skill_name2id[skill1_name] if skill1_name in self._skill_name2id else None
        skill2_id = self._skill_name2id[skill2_name] if skill2_name in self._skill_name2id else None
        return self.simple_select_seed2s_from_ids(amulet_id, skill1_id, skill1_val, skill2_id, skill2_val, slot_num)


    def select_seed1s_from_fixed_seed2_ids(self, 
            seed2, amulet_id, skill1_id, skill1_val, skill2_id, skill2_val, slot_num):
        u""" 指定されたSeed2, スキルId、スキル値、スロット数において、
        条件を満たすお守りが発生するSeed1を検索する
        return (seed1s_555, seed1s_888)"""
        skill2_positions = self._acc_seed2skill2.select_skill_position_from_seed2_ids(
                amulet_id, seed2, skill2_id)
        if len(skill2_positions) > 0:
            seed1s_555, seed1s_888 = set(), set()
            for seed1s, key in zip([seed1s_555, seed1s_888],
                    [mh4constnumbers.KEY_TENUN555, mh4constnumbers.KEY_TENUN888]):
                for pos in skill2_positions:
                    seed1s |= self._acc_seed1tenun.select_seed1s_from_skill_pos_ids(
                            amulet_id, skill1_id, pos, key)

            for seed1s, key in zip([seed1s_555, seed1s_888],
                    [mh4constnumbers.KEY_TENUN555, mh4constnumbers.KEY_TENUN888]):
                tmp_list = []
                for seed1 in seed1s:
                    result = self._simulator.alchemy_tenun_id(seed1, seed2, key)
                    for _skill1_id, _skill1_val, _skill2_id, _skill2_val, _slot_num, type_name in result:
                        if (skill1_id == _skill1_id and skill1_val <= _skill1_val and
                                skill2_id == _skill2_id and skill2_val <= _skill2_val and
                                slot_num <= _slot_num):
                            tmp_list.append(seed1)
                            break
                seed1s &= set(tmp_list)

            return (seed1s_555, seed1s_888)
        else:
            return set(), set()

    def select_seed1s_from_fixed_seed2_names(self, 
            seed2, amulet_name, skill1_name, skill1_val, skill2_name, skill2_val, slot_num):
        u""" 指定されたSeed2, スキル名、スキル値、スロット数において、
        条件を満たすお守りが発生するSeed1を検索する
        return (seed1s_555, seed1s_888)"""
        if (amulet_name in self._amu_name2id and 
                skill1_name in self._skill_name2id and
                skill2_name in self._skill_name2id):
            amulet_id = self._amu_name2id[amulet_name]
            skill1_id = self._skill_name2id[skill1_name]
            skill2_id = self._skill_name2id[skill2_name]
            return self.select_seed1s_from_fixed_seed2_ids(seed2, amulet_id,
                    skill1_id, skill1_val, skill2_id, skill2_val, slot_num)
        else:
            return set(), set()

    def select_seed1s_from_ids(self, 
            amulet_id, skill1_id, skill1_val, skill2_id, skill2_val, slot_num):
        u""" 指定されたお守りId、スキルId、スキル値、スロット数において、
        条件を満たすお守りが発生するSeed2とSeed1のペアを検索する。
        return {seed2s: (seed1s_555, seed1s_888)} """
        u"""
        seed2s = self._acc_seed2skill2.select_seed2s_from_skill2_id(amulet_id, skill2_id)
        print u"first seed2's length: {0}".format(len(seed2s))
        result_dict = {}
        count = 0
        for seed2 in seed2s:
            _seed1s_555, _seed1s_888 = self.select_seed1s_from_fixed_seed2_ids(seed2, amulet_id,
                    skill1_id, skill1_val, skill2_id, skill2_val, slot_num)
            if len(_seed1s_555) == 0 and len(_seed1s_888) == 0:
                continue
            else:
                result_dict[seed2] = (_seed1s_555, _seed1s_888)
            count += 1
            if count % 100 == 0:
                print u"loop {0}".format(count)

        return result_dict
        """
        raise NotImplementedError(u"too heavy and long to calc")

    def select_seed1s_from_names(self, 
            amulet_name, skill1_name, skill1_val, skill2_name, skill2_val, slot_num):
        u""" 指定されたお守り名、スキル名、スキル値、スロット数において、
        条件を満たすお守りが発生するSeed2とSeed1のペアを検索する。
        return {seed2s: (seed1s_555, seed1s_888} """
        u"""
        if (amulet_name in self._amu_name2id and 
                skill1_name in self._skill_name2id and
                skill2_name in self._skill_name2id):
            amulet_id = self._amu_name2id[amulet_name]
            skill1_id = self._skill_name2id[skill1_name]
            skill2_id = self._skill_name2id[skill2_name]
            return self.select_seed1s_from_ids(amulet_id,
                    skill1_id, skill1_val, skill2_id, skill2_val, slot_num)
        else:
            return {}
        """
        raise NotImplementedError(u"too heavy and long to calc")
