%feature("docstring") OT::StandardDistributionPolynomialFactory
"Build orthonormal or orthogonal univariate polynomial families.

Parameters
----------
arg : :class:`~openturns.Distribution` or :class:`~openturns.OrthonormalizationAlgorithm`
    Either a :class:`~openturns.Distribution` implementing the probability
    measure according to which the polynomial family is orthonormal **or** an
    :class:`~openturns.OrthonormalizationAlgorithm`.

    In the first case, the implementation will switch to the suitable specific
    orthonormal univariate polynomial family if any (see the notes below), or
    it will default to the :class:`~openturns.AdaptiveStieltjesAlgorithm` to build
    an orthonormal univariate polynomial family.

Notes
-----
Use this functionality with caution:

   - The polynomials exist if and only if the distribution admits finite
     moments of all orders. Even if some algorithms manage to compute something, it
     will be plain numerical noise.
   - Even if the polynomials exist, they form an Hilbertian basis with respect to the
     dot product induced by the distribution if and only if the distribution is
     **determinate**, ie is characterized by its moments. For example, the
     :class:`~openturns.LogNormal` distribution has orthonormal polynomials of
     arbitrary degree but the projection onto the functional space generated by
     these polynomials (see :class:`~openturns.FunctionalChaosAlgorithm`) may
     converge to a function that differs significantly from the function being
     projected (see [ernst2012]_).
     See :ref:`functional_chaos` for more details on this topic.

OpenTURNS implements the following **specific orthonormal** [#orthonormal]_
univariate polynomial families together with their associated **standard
distributions**.

.. table::

    +-----------------------------------------------------------------------+---------------------------------------+
    | Standard distribution                                                 | Polynomial                            |
    +=======================================================================+=======================================+
    | :class:`~openturns.Normal`                                            | :class:`~openturns.HermiteFactory`    |
    | :math:`\\cN(\\mu = 0, \\sigma = 1)`                                      |                                       |
    +-----------------------------------------------------------------------+---------------------------------------+
    | :class:`~openturns.Uniform`                                           | :class:`~openturns.LegendreFactory`   |
    | :math:`\\cU(a = -1, b = 1)`                                            |                                       |
    +-----------------------------------------------------------------------+---------------------------------------+
    | :class:`~openturns.Gamma`                                             | :class:`~openturns.LaguerreFactory`   |
    | :math:`\\Gamma(k = k_a + 1, \\lambda = 1, \\gamma = 0)`                  |                                       |
    +-----------------------------------------------------------------------+---------------------------------------+
    | :class:`~openturns.Beta`                                              | :class:`~openturns.JacobiFactory`     |
    | :math:`{\\rm B}(r = \\beta + 1, t = \\alpha + \\beta + 2, a = -1, b = 1)` |                                       |
    +-----------------------------------------------------------------------+---------------------------------------+
    | :class:`~openturns.Poisson`                                           | :class:`~openturns.CharlierFactory`   |
    | :math:`\\cP(\\lambda)`                                                  |                                       |
    +-----------------------------------------------------------------------+---------------------------------------+
    | :class:`~openturns.Binomial`                                          | :class:`~openturns.KrawtchoukFactory` |
    | :math:`\\cB(n, p)`                                                     |                                       |
    +-----------------------------------------------------------------------+---------------------------------------+
    | :class:`~openturns.NegativeBinomial`                                  | :class:`~openturns.MeixnerFactory`    |
    | :math:`\\cB^-(r, p)`                                                   |                                       |
    +-----------------------------------------------------------------------+---------------------------------------+

Aside, OpenTURNS also implements generic algorithms for building orthonormal
univariate polynomial families with respect to any arbitrary probability
measure (implemented as a :class:`~openturns.Distribution`).
OpenTURNS implements the following :class:`~openturns.OrthonormalizationAlgorithm`'s, 
:class:`~openturns.AdaptiveStieltjesAlgorithm` (default).

.. [#orthonormal] A polynomial family :math:`(P_n)_{n \\geq 0}` is said to be
    orthonormal with respect to the probability measure :math:`w(z) \\di{z}` if
    and only if:

    .. math::

        \\scalarproduct{P_i}{P_j} = \\int_{\\supp{w}} P_i(z) P_j(z) w(z) \\di{z}
                                 = \\delta_{ij}, \\quad i, j = 1, \\ldots, n

    where :math:`\\delta_{ij}` denotes Kronecker's delta.

See also
--------
:ref:`orthogonal_polynomials`, AdaptiveStieltjesAlgorithm

Examples
--------
>>> import openturns as ot

Build the specific orthonormal polynomial factory associated to the normal
distribution (Hermite):

>>> polynomial_factory = ot.StandardDistributionPolynomialFactory(ot.Normal())
>>> for i in range(3):
...     print(polynomial_factory.build(i))
1
X
-0.707107 + 0.707107 * X^2

Build an orthonormal polynomial factory for the WeibullMin distribution with
the default orthonormalization algorithm:

>>> polynomial_factory = ot.StandardDistributionPolynomialFactory(ot.WeibullMin())
>>> for i in range(3):
...     print(polynomial_factory.build(i))
1
-1 + X
1 - 2 * X + 0.5 * X^2

Build an orthonormal polynomial factory for the log-normal distribution with
Chebychev's othonormalization algorithm:

>>> algorithm = ot.AdaptiveStieltjesAlgorithm(ot.WeibullMin())
>>> polynomial_factory = ot.StandardDistributionPolynomialFactory(algorithm)
>>> for i in range(3):
...     print(polynomial_factory.build(i))
1
-1 + X
1 - 2 * X + 0.5 * X^2
"

// ---------------------------------------------------------------------

%feature("docstring") OT::StandardDistributionPolynomialFactory::getHasSpecificFamily
"Accessor to the specific family boolean.

Returns
-------
hasSpecificFamily : bool
    True if the factory has a specific orthogonal polynomial family."

// ---------------------------------------------------------------------

%feature("docstring") OT::StandardDistributionPolynomialFactory::getOrthonormalizationAlgorithm
"Accessor to the orthonormalization algorithm.

Returns
-------
orthonormalizationAlgorithm : :class:`~openturns.OrthonormalizationAlgorithm`
    The univariate polynomial orthonormalization algorithm."

// ---------------------------------------------------------------------

%feature("docstring") OT::StandardDistributionPolynomialFactory::getSpecificFamily
"Accessor to the specific orthonormal polynomial family.

Returns
-------
specificFamily : :class:`~openturns.OrthogonalUniVariatePolynomialFamily`
    The specific orthonormal univariate polynomial family."
