/*
* Author: Scott Bailey
* License: BSD
*/

/*****************************************************************
*                             CONFIG
*****************************************************************/

/*
*  These settings controls how much to increment or decrement when
*  adding numeric values to date/time values.
*  Common values:
*  '1us'   or   '1 microsecond' (default for temporal period)
*  '1ms'        '1 millisecond'
*  '1s'         '1 second' (default for timestamp, period and time)
*  '1m'         '1 minute'
*  '1h'         '1 hour'
*  '1d'         '1 day'    (default for date)
*/

-- sets granularity for timestamp calculation
CREATE OR REPLACE FUNCTION timestamp_granularity()
RETURNS INTERVAL AS
$$
    SELECT INTERVAL '1s';
$$ LANGUAGE 'sql' IMMUTABLE COST 1;


-- sets granularity for date calculations
CREATE OR REPLACE FUNCTION date_granularity()
RETURNS INTERVAL AS
$$
    SELECT INTERVAL '1 day';
$$ LANGUAGE 'sql' IMMUTABLE COST 1;

-- granularity for time calculations
CREATE OR REPLACE FUNCTION time_granularity()
RETURNS INTERVAL AS
$$
    SELECT INTERVAL '1s';
$$ LANGUAGE 'sql' IMMUTABLE COST 1;

-- sets granularity for period calculations
CREATE OR REPLACE FUNCTION period_granularity()
RETURNS INTERVAL AS
$$
    SELECT INTERVAL '1s';
$$ LANGUAGE 'sql' IMMUTABLE COST 1;



/*****************************************************************
*                      timestamp functions
*****************************************************************/

-- 1 unit(seconds) prior to ts
CREATE OR REPLACE FUNCTION prior(timestampTz)
RETURNS timestampTz AS
$$
    SELECT $1 - timestamp_granularity();
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- 1 unit(seconds) after ts
CREATE OR REPLACE FUNCTION next(timestampTz)
RETURNS timestampTz AS
$$
    SELECT $1 + timestamp_granularity();
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- add n units(seconds) to timestampTz
CREATE OR REPLACE FUNCTION timestamp_add(timestampTz, numeric)
RETURNS timestampTz AS
$$
    SELECT $1 + timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- add n units(seconds) to timestamp
CREATE OR REPLACE FUNCTION timestamp_add(timestamp, numeric)
RETURNS timestamp AS
$$
    SELECT $1 + timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;

-- subtract n units(seconds) from timestamp
CREATE OR REPLACE FUNCTION timestamp_subtract(timestampTz, numeric)
RETURNS timestampTz AS
$$
    SELECT $1 - timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- subtract n units(seconds) to timestamp
CREATE OR REPLACE FUNCTION timestamp_subtract(timestamp, numeric)
RETURNS timestamp AS
$$
    SELECT $1 - timestamp_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


/*****************************************************************
*                       date functions
*****************************************************************/

-- add n units(days) to date 
-- default date add only allows interger addition
CREATE OR REPLACE FUNCTION date_add(date, numeric)
RETURNS timestampTz AS
$$
    SELECT $1::timestampTz + date_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- subtract n units(days) from date
CREATE OR REPLACE FUNCTION date_subtract(date, numeric)
RETURNS timestampTz AS
$$
    SELECT $1::timestampTz - date_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


/*****************************************************************
*                        time functions
*****************************************************************/


-- add n units to time
CREATE OR REPLACE FUNCTION time_add(time, numeric)
RETURNS time AS
$$
    SELECT $1 + time_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- subtract n units to time
CREATE OR REPLACE FUNCTION time_subtract(time, numeric)
RETURNS time AS
$$
    SELECT $1 - time_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- add n units to time
CREATE OR REPLACE FUNCTION time_add(timeTz, numeric)
RETURNS timeTz AS
$$
    SELECT $1 + time_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


-- subtract n units from time
CREATE OR REPLACE FUNCTION time_subtract(timeTz, numeric)
RETURNS timeTz AS
$$
    SELECT $1 - time_granularity() * $2;
$$ LANGUAGE 'sql' IMMUTABLE STRICT COST 1;


/*****************************************************************
*                        interval functions
*****************************************************************/

-- Number of seconds in interval
CREATE OR REPLACE FUNCTION duration(interval)
RETURNS numeric AS
$$
   SELECT EXTRACT(EPOCH FROM $1)::numeric;
$$ LANGUAGE 'sql' IMMUTABLE STRICT;

/*****************************************************************
*                          operators
*****************************************************************/

CREATE OPERATOR +(
  PROCEDURE = timestamp_add,
  LEFTARG = timestampTz,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = timestamp_subtract,
  LEFTARG = timestampTz,
  RIGHTARG = numeric
);

CREATE OPERATOR +(
  PROCEDURE = timestamp_add,
  LEFTARG = timestamp,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = timestamp_subtract,
  LEFTARG = timestamp,
  RIGHTARG = numeric
);

CREATE OPERATOR +(
  PROCEDURE = date_add,
  LEFTARG = date,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = date_subtract,
  LEFTARG = date,
  RIGHTARG = numeric
);

CREATE OPERATOR +(
  PROCEDURE = time_add,
  LEFTARG = time,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = time_subtract,
  LEFTARG = time,
  RIGHTARG = numeric
);

CREATE OPERATOR +(
  PROCEDURE = time_add,
  LEFTARG = timetz,
  RIGHTARG = numeric
);

CREATE OPERATOR -(
  PROCEDURE = time_subtract,
  LEFTARG = timetz,
  RIGHTARG = numeric
);

