
/************************************************************************
 *                                                                      *
 * gm2pgsql V1.0                                                        *
 * Simple Geomedia Warehouseto postgis exporter                         *
 * Copyright (C) 2009 Claudio Rocchini                                  *
 * Istituto Geografico Militare Italiano                                *
 * web:   www.igmi.org                                                  *
 * email: ad2prod@geomil.esercito.difesa.it                             *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the  Free Software Foundation, either  version 3 of the  License, or *
 * any later version.                                                   *
 *                                                                      *
 *   This program is distributed in the hope that it will be useful,    *
 *   but WITHOUT ANY WARRANTY; without  even the implied warranty of    *
 *   MERCHANTABILITY or  FITNESS  FOR A PARTICULAR PURPOSE.  See the    *
 *   GNU General Public License for more details.                       *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program.  If not, see <http://www.gnu.org/licenses/> *
 *                                                                      *
 ************************************************************************/


#include <stdio.h>
#include <stdlib.h>

#include <vector>
#include <string>

#include "rodbc2.h"
#include "blob4.h"


const char * text_field_name = "GM2P_TEXT";

//#define SHQN_UTF_CONVERSION 

#ifdef SHQN_UTF_CONVERSION 

// TODO text conversion support

static void convert_utf8( byte c, std::vector< byte > & data)
{
		// In realt basterebbe un buffer di 2 byte (massima lunghezza unicode)
		// ma non e' il caso dei fare i tirchi.

	const size_t BSIZE = 16;
	unsigned short buf2[BSIZE];
	unsigned char  buf3[BSIZE];

		// Da ASCCI a unicode 16
	int s1 = MultiByteToWideChar(CP_ACP,MB_ERR_INVALID_CHARS, (char *)&c,1,buf2,BSIZE);
	if(s1==0)
	{
		printf("****ERROR**** direct conversion: %c\n",c);
		exit(0);
	}
	
		// Da unicode 16 a UTF8
	int s2 = WideCharToMultiByte(CP_UTF8,0,buf2,s1,(LPSTR)buf3,BSIZE,0,0);
	if(s2==0)
	{
		printf("****ERROR**** direct conversion: %c\n",c);
		exit(0);
	}

	for(int i=0;i<s2;++i)
		data.push_back(buf3[i]);
}

	// Funzione di conversione di una stringa

static void convert( std::vector< byte > & data )
{
	if(data.empty()) return;

	std::vector< byte > t(data.size());
	std::vector< byte >::iterator i;

	std::copy(data.begin(),data.end(),t.begin());
	data.clear();

	for(i=t.begin();i!=t.end();++i)
	{
		if(*i>=32 && *i<=126)		// ASCII standard
			data.push_back(*i);
		else						// ASCII esteso : necessita conversione
		{
			convert_utf8(*i,data);
		}
	}
}

#endif



class Feature
{
public:
	std::string FeatureName;
	int  GeometryType;
	std::string PrimaryGeometryFieldName;
	std::string RealPrimaryGeometryFieldName;

	Feature() {}
	Feature( const std::string & fn, int gt, const std::string pgfm) :
		FeatureName(fn), GeometryType(gt), PrimaryGeometryFieldName(pgfm) {}
};


class variant_buffer
{
public:
	enum BType { L,D,S,B,F,T };

	BType  t;
	long   l;
	double f;
	double d;
	TIMESTAMP_STRUCT tm;
	std::vector<byte> s; long ls;

	bool ignore;
};

#ifdef SHQN_UTF_CONVERSION 
void out_str( FILE * fo, const std::vector<byte> & os, long ls )
#else
void out_str( FILE * fo, const std::vector<byte> & s, long ls )
#endif
{
	if(ls==-1) fprintf(fo,"NULL");
	else
	{
#ifdef SHQN_UTF_CONVERSION 
		std::vector<byte> s(ls);
		std::copy(os.begin(),os.begin()+ls,s.begin());
		convert(s);
#endif

		fprintf(fo,"'");
		std::vector<byte>::const_iterator i;
		for(i=s.begin();i!=s.begin()+ls;++i)
		{
			if( *i=='\'' ) fputc('\'',fo);
			fputc(*i,fo);
		}
		fprintf(fo,"'");
	}
}


void print_num( FILE * fo, double n, long size = 42 )
{
	if(size==-1) fprintf(fo,"NULL");
	else
	{
		static char buff[32];
		sprintf(buff,"%30.15f",n);
		char * p = 0;
		p = buff+strlen(buff);
		
		while( *(p-1)=='0' && *(p-2)!='.' && p>buff ) --p;
		*p = 0;
		p = buff;
		while(*p==' ') ++p;
		fputs(p,fo);
	}
}

void toWKT( FILE * fo, const blo::BlobCache & bb )
{
	switch(bb.type)
	{
	case blo::BlobCache::T_POINT:
		fputs("POINT(",fo);
		print_num(fo,bb.pt.x); fputs(" ",fo);
		print_num(fo,bb.pt.y); fputs(" ",fo);
		print_num(fo,bb.pt.z); fputs(")",fo);
		break;

	case blo::BlobCache::T_OPOINT:
		fputs("POINT(",fo);
		print_num(fo,bb.pt.x); fputs(" ",fo);
		print_num(fo,bb.pt.y); fputs(" ",fo);
		print_num(fo,bb.pt.z); fputs(")",fo);
		// TODO how to save angle ??
		break;

	case blo::BlobCache::T_LINE:
		{
			fputs("LINESTRING(",fo);
			std::vector<blo::point>::const_iterator i;
			for(i=bb.pl.l.begin();i!=bb.pl.l.end();++i)
			{
				print_num(fo,i->x); fputs(" ",fo);
				print_num(fo,i->y); fputs(" ",fo);
				print_num(fo,i->z);

				if(i!=bb.pl.l.end()-1) fputs(",",fo);
			}
			fputs(")",fo);
		}
		break;

	case blo::BlobCache::T_AREA:
		{
			fprintf(fo,"POLYGON(");
			std::vector< blo::ring >::const_iterator i;
			for(i=bb.pg.r.begin();i!=bb.pg.r.end();++i)
			{
				fprintf(fo,"(");
				std::vector<blo::point>::const_iterator j;
				for(j=(*i).begin();j!=(*i).end();++j)
				{
					print_num(fo,j->x); fputs(" ",fo);
					print_num(fo,j->y); fputs(" ",fo);
					print_num(fo,j->z); fputs(",",fo);
				}
				print_num(fo,(*i).begin()->x); fputs(" ",fo);
				print_num(fo,(*i).begin()->y); fputs(" ",fo);
				print_num(fo,(*i).begin()->z);
				fprintf(fo,")");
				if(i!=bb.pg.r.end()-1) fprintf(fo,",");
			}
			fprintf(fo,")");
		}
		break;

	case blo::BlobCache::T_ARC:
		fputs("POINT(",fo);
		print_num(fo,bb.ar.start.x); fputs(" ",fo);
		print_num(fo,bb.ar.start.y); fputs(" ",fo);
		print_num(fo,bb.ar.start.z); fputs(")",fo);
		// TODO arc
		break;

	case blo::BlobCache::T_TEXT:
		fputs("POINT(",fo);
		print_num(fo,bb.pt.x); fputs(" ",fo);
		print_num(fo,bb.pt.y); fputs(" ",fo);
		print_num(fo,bb.pt.z); fputs(")",fo);
		break;

	case blo::BlobCache::T_COLLECTION:
		{
			fprintf(fo,"GeometryCollection(");
			std::vector<blo::BlobCache>::const_iterator i;
			for(i=bb.co.begin();i!=bb.co.end();++i)
			{
				toWKT(fo,*i);
				if(i!=bb.co.end()-1) fprintf(fo,",");
			}
			fprintf(fo,")");
		}
		break;

	default:
		printf("INTERNAL*Unsupported Blob type %d\n",bb.type);
		exit(0);
	}
}

int main( int argc, char * argv[] )
{
	char * pcas = "";

	printf("GM2PGSQL V1.0 (C) 2009 IGMI\n\n");

	const size_t VC_SIZE = 32768;		// lunghezza string var
	const size_t GM_SIZE = 4*1024*1024;	// buffer geometrico

	std::auto_ptr<unsigned char> static_buffer( new unsigned char[GM_SIZE] );
	long sb_size = 0;

	if(argc<5)
	{
		printf(
			"usage: [options] gm2pgsql warefile.mdb outfile.sql db_schema_name SRID\n"
			"options:\n"
			" -p : preserve case of names (tables and attributes)\n"
			);
		exit(0);
	}

	int pbase = 1;

	while(argv[pbase][0]=='-')
	{
		switch(argv[pbase][1])
		{
		case 'p': pcas = "\""; break;
		default:
			printf("warning: unknow option: %c\n",argv[pbase][1]);
		}
		++pbase;
	}

	static char msg[1024];
	CONST INT BSIZE = 4096;
	static char buff1[BSIZE]; long bsize1 = 0;
	static char buff2[BSIZE]; long bsize2 = 0;
	long n;
	SQLRETURN r;

	std::string query;
	std::string feature_table;
	std::string plist_table;
	std::vector< Feature > features;
	std::string schema = argv[2+pbase];
	std::string srid   = argv[3+pbase];

	std::auto_ptr<sql2::Environment>  env( new sql2::Environment() );
	std::auto_ptr<sql2::Connection >  con( new sql2::Connection(*env) );

		// ******** Connection to access file ********

	con->connect_file("Access",argv[pbase]);
	if( !con->is_connected() )
	{
		con->get_message(msg);
		printf("ERROR*Fail to connect ODBC: %s\n",msg);
		exit(0);
	}

	std::auto_ptr<sql2::Statement> sta( new sql2::Statement(*con) );

		// Open out file

	FILE * fo = fopen(argv[1+pbase],"w");
	if(fo==0)
	{
		printf("ERROR*Fail to write file %s\n",argv[1+pbase]);
		exit(0);
	}

	fprintf(fo,"-- gm2pgsql export of %s\n\n",argv[0+pbase]);

		// ******** Get feature table name ********

	sta->bind(1,buff1,BSIZE,bsize1);
	sta->execute("SELECT TableName FROM GAliasTable WHERE TableType='INGRFeatures'");

	if(	sta->get_status()!=SQL_SUCCESS && sta->get_status()!=SQL_SUCCESS_WITH_INFO )
	{
		sta->get_message(msg);
		printf("ERROR*Fail to read Alias Table: %s\n",msg);
		exit(0);
	}
	
	r = sta->fetch();

	if(r!=SQL_SUCCESS && r!=SQL_SUCCESS_WITH_INFO)
	{
		sta->get_message(msg);
		printf("ERROR*Fail to read Alias Table: %s\n",msg);
		exit(0);
	}

	feature_table = buff1;

	sta->close_cursor();
	sta->unbind();

		// ******** Get feature list ********

	sta->bind(1,buff1,BSIZE,bsize1);
	sta->bind(2,n);
	sta->bind(3,buff2,BSIZE,bsize2);

	query = "SELECT FeatureName, GeometryType, PrimaryGeometryFieldName FROM ";
	query += feature_table;

	sta->execute(query.c_str());

	if(	sta->get_status()!=SQL_SUCCESS && sta->get_status()!=SQL_SUCCESS_WITH_INFO )
	{
		sta->get_message(msg);
		printf("ERROR*Fail to read Feature List: %s\n",msg);
		exit(0);
	}
	
	for(;;)
	{
		n = 0;
		buff1[0] = 0;
		buff2[0] = 0;

		r = sta->fetch();
		if(r==SQL_NO_DATA) break;

		if(r!=SQL_SUCCESS && r!=SQL_SUCCESS_WITH_INFO)
		{
			sta->get_message(msg);
			printf("ERROR*Fail to read Feature List: %s\n",msg);
			exit(0);
		}

		features.push_back( Feature(buff1,n,buff2) );
	}

	sta->close_cursor();
	sta->unbind();

	printf("Found %u features\n",features.size());

		// ******** Feature exporting cycle ********

	for(std::vector< Feature >::iterator fi=features.begin();fi!=features.end();++fi)
	{
		printf("Feature %s (%u of %u): "
			,fi->FeatureName.c_str()
			,(fi-features.begin())+1
			,features.size()
		); fflush(stdout);

		if(fi->GeometryType==4)
		{
			printf("Raster data skipped (for now)\n");
			continue;
		}
			// ******** Get feature structure ********

		std::vector< sql2::ColumnDesc > cols;

		if( ! sta->columns(fi->FeatureName.c_str(),cols) )
		{
			sta->get_message(msg);
			printf("ERROR*Fail to get table structure of %s: %s\n",fi->FeatureName.c_str(),msg);
			exit(0);
		}

		std::vector<variant_buffer> buffers( cols.size() );
		std::vector<variant_buffer>::iterator bi;
		std::vector< sql2::ColumnDesc >::const_iterator ci;
		int nc;

				// ******** Check bads and goods ********

		std::string last_col;
		std::string primary_key = "";
		
		for(ci=cols.begin(),bi=buffers.begin(); ci!=cols.end(); ++ci,++bi)
		{
				// TODO poco solido! How to solid this control???

			if( ci->name.length()>3 &&
				( ci->name.substr( ci->name.length()-3,3 ) == "_sk" ||
                  ci->name.substr( ci->name.length()-3,3 ) == "_SK" ) ) 
			{
				bi->ignore = true;
			}
			else
			{
				bi->ignore = false;
				last_col   = ci->name;
			}

			if(ci->nullable=="0" && ci->type==SQL_INTEGER && primary_key=="") 
				primary_key = ci->name;
		}

			// ******** Create table Command Construsction ********
		
		fprintf(fo,"\nCREATE TABLE %s.%s%s%s (\n",schema.c_str(),pcas,fi->FeatureName.c_str(),pcas);

		for(ci=cols.begin(),bi=buffers.begin(); ci!=cols.end(); ++ci,++bi)
		{
			if(bi->ignore) continue;

			fprintf(fo,"\t%s%s%s",pcas,ci->name.c_str(),pcas);

			switch(ci->type)
			{
			case SQL_BIT:            fprintf(fo," BOOLEAN"); break;
			case SQL_SMALLINT:
			case SQL_TINYINT:        fprintf(fo," SMALLINT"); break;
			case SQL_INTEGER:        fprintf(fo," INTEGER"); break;
			case SQL_FLOAT:
			case SQL_REAL:           fprintf(fo," REAL"); break;
			case SQL_DOUBLE:         fprintf(fo," DOUBLE PRECISION"); break;
			case SQL_VARCHAR:        fprintf(fo," CHARACTER VARYING(%s)",ci->size.c_str()); break;
			case SQL_LONGVARCHAR:    fprintf(fo," CHARACTER VARYING",ci->size.c_str()); break;
			case SQL_TYPE_TIMESTAMP: fprintf(fo," TIMESTAMP"); break;
			case SQL_NUMERIC       : fprintf(fo," NUMERIC(%s,%s)",ci->size.c_str(),ci->digit.c_str()); break;
			case SQL_LONGVARBINARY:
					// TODO debole! What is the text geometry field name??
				if(  fi->PrimaryGeometryFieldName==ci->name||
					(fi->PrimaryGeometryFieldName=="" && fi->GeometryType==33) )
				{
					fprintf(fo," GEOMETRY");
					fi->RealPrimaryGeometryFieldName = ci->name;
				}
				else
				{
					printf("ERROR*Unsupported LONGVARBINARY type on table %s\n",fi->FeatureName.c_str());
					exit(0);
				}
				break;
			default:
				printf("ERROR*Unsupported SQL type in def: %ld on table %s\n",ci->type,fi->FeatureName.c_str());
				exit(0);
			}

			if(ci->name==primary_key) fprintf(fo," PRIMARY KEY");
			else if(ci->nullable=="0") fprintf(fo," NOT NULL");

				// TODO SUPPORT for default value

			if( ci->name==last_col)
			{
				if(fi->GeometryType==33)
				{
					fprintf(fo,
						",\n"
						"\t%s%s%s CHARACTER VARYING"
						,pcas
						,text_field_name
						,pcas
					);
				}
				fprintf(fo,"\n");
			}
			else fprintf(fo,",\n");
		}
	
		fprintf(fo,");\n\n");

			// ******** Buffer bindings for data ********
		
		for(ci=cols.begin(),bi=buffers.begin(),nc=1; ci!=cols.end(); ++ci,++bi,++nc)
		{
			if(bi->ignore) continue;

			int size = atoi(ci->size.c_str());
			bool gm_just_binded = false;

			switch(ci->type)
			{
			case SQL_BIT:
			case SQL_TINYINT:
			case SQL_SMALLINT:
			case SQL_INTEGER:
				bi->t = variant_buffer::L;
				sta->bind(nc,bi->l,bi->ls);
				break;
			case SQL_FLOAT:
			case SQL_REAL:
				bi->t = variant_buffer::F;
				sta->bind(nc,bi->f,bi->ls);
				break;
			case SQL_NUMERIC:
			case SQL_DOUBLE:
				bi->t = variant_buffer::D;
				sta->bind(nc,bi->d,bi->ls);
				break;
			case SQL_VARCHAR:
				bi->t = variant_buffer::S;
				bi->s.resize(size+1);
				sta->bind(nc,&(bi->s.front()),size+1,bi->ls);
				break;
			case SQL_LONGVARCHAR:
				bi->t = variant_buffer::S;
				bi->s.resize(VC_SIZE);
				sta->bind(nc,&(bi->s.front()),VC_SIZE,bi->ls);
				break;
			case SQL_TYPE_TIMESTAMP:
				bi->t = variant_buffer::T;
				sta->bind(nc,bi->tm,bi->ls);
				break;
			case SQL_LONGVARBINARY:
				if(gm_just_binded)
				{
					printf("ERROR*Multiple Bin Column on table %s\n",fi->FeatureName.c_str());
					exit(0);
				}
				bi->t = variant_buffer::B;
				sta->bind(nc,static_buffer.get(),GM_SIZE,sb_size);
				gm_just_binded = true;
				break;
			default:
				printf("ERROR*Unsupported SQL type in read: %ld on table %s\n",ci->type,fi->FeatureName.c_str());
				exit(0);
			}
		}

			// ******** Execute Data Query ********

		query = "SELECT * FROM ";
		query += fi->FeatureName;

		sta->execute(query.c_str());

		if(	sta->get_status()!=SQL_SUCCESS && sta->get_status()!=SQL_SUCCESS_WITH_INFO )
		{
			sta->get_message(msg);
			printf("ERROR*Fail to read Feature %s : %s\n", fi->FeatureName.c_str(), msg);
			exit(0);
		}

			// ******** Data Row Read cycle ********

		size_t nitem = 0;
		for(;;)
		{
				// ******** Fetch data ********

			r = sta->fetch();
			if(r==SQL_NO_DATA) break;

			if(r!=SQL_SUCCESS && r!=SQL_SUCCESS_WITH_INFO)
			{
				sta->get_message(msg);
				printf("ERROR*Fail to read Feature %s: %s\n",fi->FeatureName.c_str(), msg);
				exit(0);
			}

				// ******** Dump Data SQL ********

			fprintf(fo,"INSERT INTO %s.%s%s%s VALUES(",schema.c_str(),pcas,fi->FeatureName.c_str(),pcas);

			for(ci=cols.begin(),bi=buffers.begin(); ci!=cols.end(); ++ci,++bi)
			{
				blo::BlobCache bb;

				if(bi->ignore) continue;

				switch(bi->t)
				{
				case variant_buffer::L:
					if(ci->type==SQL_BIT)
					{
						if(bi->ls==-1) fprintf(fo,"NULL");
						else           fprintf(fo,"%s",bi->l ? "TRUE" : "FALSE" );
					}
					else
					{
						if(bi->ls==-1) fprintf(fo,"NULL");
						else           fprintf(fo,"%d",bi->l);
					}
					break;
				case variant_buffer::F: print_num(fo,bi->f,bi->ls); break;
				case variant_buffer::D: print_num(fo,bi->d,bi->ls); break;
				case variant_buffer::S: out_str(fo,bi->s,bi->ls); break;
				case variant_buffer::T:
					if(bi->ls==-1) fprintf(fo,"NULL");
					else
					{
						fprintf(fo,"'%d-%02d-%02d %02d:%02d:%02d'::TIMESTAMP"
							,bi->tm.year
							,bi->tm.month
							,bi->tm.day
							,bi->tm.hour
							,bi->tm.minute
							,bi->tm.second
						);
					}
					break;
				case variant_buffer::B:
					{
						if(sb_size==-1) fprintf(fo,"NULL");
						else
						{
							if(!bb.decode(static_buffer.get()/*,sb_size*/))
							{
								printf("ERROR*Decoding blob of %s\n",fi->FeatureName.c_str());
								exit(0);
							}
							fprintf(fo,"GeomFromEWKT('SRID=%s;",srid.c_str());
							toWKT(fo,bb);
							fprintf(fo,"')");
						}
					}
					break;
				default:
					printf("INTERNAL 1\n");
					exit(0);
				}

				if(ci->name!=last_col)
					fprintf(fo,",");
				else
				{
					if(fi->GeometryType==33 && bb.type==blo::BlobCache::T_TEXT)
					{
						fputs(",",fo);
						out_str(fo,bb.tx,bb.tx.size());
					}
				}
			}

			fprintf(fo,");\n");

			++nitem;
		}

		printf("%u items\n",nitem);
	
		sta->close_cursor();
		sta->unbind();

		fprintf(fo,"\n");
	}

	fprintf(fo,"\n");

		// ******** Generating additional commands ********

	{
		for(std::vector< Feature >::const_iterator fi=features.begin();fi!=features.end();++fi)
		if(fi->GeometryType!=4)
		{
				fprintf(fo,"ALTER TABLE %s.%s%s%s ADD CONSTRAINT enforce_dims CHECK (ndims(%s%s%s)=3);\n"
					,schema.c_str()
					,pcas
					,fi->FeatureName.c_str()
					,pcas
					,pcas
					,fi->RealPrimaryGeometryFieldName.c_str()
					,pcas
				);

				fprintf(fo,"ALTER TABLE %s.%s%s%s ADD CONSTRAINT enforce_srid CHECK (srid(%s%s%s) = %s);\n"
					,schema.c_str()
					,pcas
					,fi->FeatureName.c_str()
					,pcas
					,pcas
					,fi->RealPrimaryGeometryFieldName.c_str()
					,pcas
					,srid.c_str()
				);

							
				fprintf(fo,"CREATE INDEX %s_%s_gist ON %s.%s%s%s USING GIST(%s%s%s);\n"
					,fi->FeatureName.c_str()
					,fi->RealPrimaryGeometryFieldName.c_str()
					,schema.c_str()
					,pcas
					,fi->FeatureName.c_str()
					,pcas
					,pcas
					,fi->RealPrimaryGeometryFieldName.c_str()
					,pcas
				);
		}
	}
	
	{
		for(std::vector< Feature >::const_iterator fi=features.begin();fi!=features.end();++fi)
		{
			char * tipo = 0;

			switch(fi->GeometryType)
			{
			case  1: tipo = "MULTILINE"; break;
			case  2: tipo = "MULTIPOLYGON"; break;
			case  3: tipo = "GEOMETRY"; break;
			case 10: tipo = "MULTIPOINT"; break;
			case 33: tipo = "POINT"; break;
			case  4: /* Nothing to do */ break;
			default:
				printf("INTERNAL*2 : bad type %d\n",fi->GeometryType);
				exit(0);
				break;
			}

			fprintf(fo,"INSERT INTO geometry_columns VALUES ('''''','%s','%s','%s',3,'%s','%s');\n"
				,schema.c_str()
				,fi->FeatureName.c_str()
				,fi->RealPrimaryGeometryFieldName.c_str()
				,srid.c_str()
				,tipo
			);

		}
	}

	fclose(fo);
	
	return 0;
}
