
module.exports = exports = configure

/**
 * Module dependencies.
 */

var fs = require('fs')
  , path = require('path')
  , glob = require('glob')
  , semver = require('semver')
  , mkdirp = require('./util/mkdirp')
  , createHook = require('./util/hook')
  , asyncEmit = require('./util/asyncEmit')
  , win = process.platform == 'win32'

exports.usage = 'Generates ' + (win ? 'MSVC project files' : 'a Makefile') + ' for the current module'

function configure (gyp, argv, callback) {

  var python = gyp.opts.python || 'python'
    , buildDir = path.resolve('build')
    , configPath
    , emitter
    , versionStr
    , version

  // Very first step is to load up the user-defined 'gyp-configure.js' file if it
  // exists. We relay filecycle events using the eventemitter returned from this
  createHook('gyp-configure.js', function (err, _e) {
    if (err) return callback(err)
    emitter = _e
    getVersion()
  })

  function getVersion () {
    if (gyp.opts.target) {
      // if --target was given, then ensure that version is installed
      versionStr = gyp.opts.target
      gyp.verbose('compiling against --target node version', versionStr)
    } else {
      // if no --target was specified then use the current host node version
      versionStr = process.version
      gyp.verbose('no --target version specified, falling back to host node version', versionStr)
    }
    version = semver.parse(versionStr)
    if (!version) {
      return callback(new Error('Invalid version number: ' + versionStr))
    }
    version = version.slice(1, 4).join('.')
    gyp.opts.ensure = true
    gyp.commands.install([ version ], createBuildDir)
  }

  function createBuildDir (err) {
    if (err) return callback(err)
    gyp.verbose('attempting to create "build" dir', buildDir)
    mkdirp(buildDir, function (err, isNew) {
      if (err) return callback(err)
      gyp.verbose('"build" dir needed to be created?', isNew)
      createConfigFile()
    })
  }

  function createConfigFile (err) {
    if (err) return callback(err)
    gyp.verbose('creating build/config.gypi file')

    var config = {}
    configPath = path.resolve(buildDir, 'config.gypi')

    config.target_defaults = {
        cflags: []
      , default_configuration: gyp.opts.debug ? 'Debug' : 'Release'
      , defines: []
      , include_dirs: []
      , libraries: []
    }

    config.variables = {
        target_arch: gyp.opts.arch || process.arch || 'ia32'
    }

    var prefix = '# Do not edit. File was generated by node-gyp\'s "configure" step'
      , json = JSON.stringify(config, null, 2)
    gyp.verbose('writing out config file', configPath)
    fs.writeFile(configPath, [prefix, json, ''].join('\n'), runGypAddon)
  }

  function runGypAddon (err) {
    if (err) return callback(err)

    var devDir = path.resolve(process.env.HOME, '.node-gyp', version)
      , gyp_addon = path.resolve(devDir, 'tools', 'gyp_addon')

    if (!win && !~argv.indexOf('-f') && !~argv.indexOf('--format')) {
      gyp.verbose('gyp format was not specified; forcing "make"')
      // force the 'make' target for non-Windows
      argv.push('-f', 'make')
    }

    // include the "config.gypi" file that was generated
    argv.unshift('-I' + configPath)

    // enforce use of the "binding.gyp" file
    argv.unshift('binding.gyp')

    // execute `gyp_addon` from the current target node version
    argv.unshift(gyp_addon)

    asyncEmit(emitter, 'before', function (err) {
      if (err) return callback(err)

      var cp = gyp.spawn(python, argv)
      cp.on('exit', onCpExit)
    })
  }

  /**
   * Called when the `gyp_addon` child process exits.
   */

  function onCpExit (code, signal) {
    asyncEmit(emitter, 'after', function (err) {
      if (err) {
        callback(err)
      } else if (code !== 0) {
        callback(new Error('`gyp_addon` failed with exit code: ' + code))
      } else {
        // we're done
        callback()
      }
    })
  }

}
