%feature("docstring") OT::GaussianProcessRegressionResult
"Gaussian process regression (aka kriging) result.

.. warning::
    This class is experimental and likely to be modified in future releases.
    To use it, import the ``openturns.experimental`` submodule.


Parameters
----------
gpfResult : :class:`~openturns.experimental.GaussianProcessFitterResult`
    Structure result of a gaussian process fitter.
covarianceCoefficients : 2-d sequence of float
    The :math:`\\vect{\\gamma}` defined in :eq:`gammaEq`.

Notes
-----
The Gaussian Process Regression (aka Kriging) meta model :math:`\\tilde{\\cM}` is defined by:

.. math::
    :label: metaModelGP
    
    \\tilde{\\cM}(\\vect{x}) =  \\vect{\\mu}(\\vect{x}) + \\Expect{\\vect{Y}(\\omega, \\vect{x})\\,| \\,\\cC}

where :math:`\\cC` is the condition :math:`\\vect{Y}(\\omega, \\vect{x}_k) = \\vect{y}_k` for each :math:`k \\in [1, \\sampleSize]`.
    
Equation :eq:`metaModelKrig` writes:

.. math::

    \\tilde{\\cM}(\\vect{x}) = \\vect{\\mu}(\\vect{x}) + \\Cov{\\vect{Y}(\\omega, \\vect{x}), (\\vect{Y}(\\omega,\\vect{x}_1),\\dots,\\vect{Y}(\\omega, \\vect{x}_{\\sampleSize}))}\\vect{\\gamma}

where 

.. math::

    \\Cov{\\vect{Y}(\\omega, \\vect{x}), (\\vect{Y}(\\omega, \\vect{x}_1),\\dots,\\vect{Y}(\\omega, \\vect{x}_{\\sampleSize}))} = \\left(\\mat{C}(\\vect{x},\\vect{x}_1)|\\dots|\\mat{C}(\\vect{x},\\vect{x}_{\\sampleSize})\\right)\\in \\cM_{\\outputDim,\\sampleSize \\times \\outputDim}(\\Rset)

and 

.. math::
    :label: gammaEqGP

    \\vect{\\gamma} = \\mat{C}^{-1}(\\vect{y}-\\vect{m})

At the end, the meta model writes:

.. math::
    :label: metaModelGPFinal

    \\tilde{\\cM}(\\vect{x}) = \\vect{\\mu}(\\vect{x}) + \\sum_{i=1}^{\\sampleSize} \\gamma_i  \\mat{C}(\\vect{x},\\vect{x}_i)




Examples
--------
Create the model :math:`\\cM: \\Rset \\mapsto \\Rset` and the samples:

>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> g = ot.SymbolicFunction(['x'],  ['x * sin(x)'])
>>> sampleX = [[1.0], [2.0], [3.0], [4.0], [5.0], [6.0]]
>>> sampleY = g(sampleX)

Create the algorithm:

>>> basis = ot.Basis([ot.SymbolicFunction(['x'], ['x']), ot.SymbolicFunction(['x'], ['x^2'])])
>>> covarianceModel = ot.GeneralizedExponential([2.0], 2.0)

>>> fit_algo = otexp.GaussianProcessFitter(sampleX, sampleY, covarianceModel, basis)
>>> fit_algo.run()

>>> algo = otexp.GaussianProcessRegression(fit_algo.getResult())
>>> algo.run()

Get the result:

>>> result = algo.getResult()

Get the meta model:

>>> metaModel = result.getMetaModel()
"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessRegressionResult::getCovarianceCoefficients
"Accessor to the covariance coefficients.

Returns
-------
covCoeff : :class:`~openturns.Sample`
    The :math:`\\vect{\\gamma}` defined in :eq:`gammaEq`.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessRegressionResult::getBasis
"Accessor to the collection of basis.

Returns
-------
basis : :class:`~openturns.Basis`
    Functional basis of size :math:`b` : :math:`(\\varphi^\\ell: \\Rset^{\\inputDim} \\rightarrow \\Rset^{\\outputDim})` for each :math:`l \\in [1, b]`.

Notes
-----
If the trend is not estimated, the basis is empty. "


// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessRegressionResult::getCovarianceModel
"Accessor to the covariance model.

Returns
-------
covModel : :class:`~openturns.CovarianceModel`
    The covariance model of the Gaussian process *W*.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::GaussianProcessRegressionResult::getLinearAlgebraMethod
"Accessor to the used linear algebra method to fit.

Returns
-------
linAlgMethod : int
    The used linear algebra method to fit the model:

    - otexp.GaussianProcessFitterResult.LAPACK or 0: using `LAPACK` to fit the model,

    - otexp.GaussianProcessFitterResult.HMAT or 1: using `HMAT` to fit the model.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessRegressionResult::getMetaModel
"Accessor to the metamodel.

Returns
-------
metaModel : :class:`~openturns.Function`
    The meta model :math:`\\tilde{\\cM}: \\Rset^{\\inputDim} \\rightarrow \\Rset^{\\outputDim}`, defined in :eq:'metaModelGPF'.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessRegressionResult::getNoise
"Accessor to the Gaussian process.

Returns
-------
process : :class:`~openturns.Process`
    Returns the Gaussian process :math:`W` with the optimized parameters.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessRegressionResult::getOptimalLogLikelihood
"Accessor to the optimal log-likelihood of the model.

Returns
-------
optimalLogLikelihood : float
    The value of the log-likelihood corresponding to the model.
"

// ---------------------------------------------------------------------    

%feature("docstring") OT::GaussianProcessRegressionResult::getRegressionMatrix
"Accessor to the regression matrix.

Returns
-------
process : :class:`~openturns.Matrix`
    Returns the regression matrix.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessRegressionResult::getTrendCoefficients
"Accessor to the trend coefficients.

Returns
-------
trendCoef : sequence of float
    The trend coefficients vectors :math:`(\\vect{\\alpha}^1, \\dots, \\vect{\\alpha}^{\\outputDim})` as a :class:`~openturns.Point` 
"

// ---------------------------------------------------------------------

%feature("docstring") OT::GaussianProcessRegressionResult::getTrendFunction
"Accessor to the trend function.

Returns
-------
trendFunc : :class:`~openturns.Function`
    The trend function.
"
// ---------------------------------------------------------------------