/*
 *  linux/fs/fat/cache.c
 *
 *  Written 1992,1993 by Werner Almesberger
 */

#include <linux/msdos_fs.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/stat.h>

#include "msbuffer.h"

static struct fat_cache *fat_cache,cache[FAT_CACHE];

/* Returns the this'th FAT entry, -1 if it is an end-of-file entry. If
   new_value is != -1, that FAT entry is replaced by it. */

int fat_access(struct super_block *sb,int nr,int new_value)
{
	struct buffer_head *bh,*bh2,*c_bh,*c_bh2;
	unsigned char *p_first,*p_last;
	int first,last,next,copy;

	if ((unsigned) (nr-2) >= MSDOS_SB(sb)->clusters) return 0;
	if (MSDOS_SB(sb)->fat_bits == 16) first = last = nr*2;
	else {
		first = nr*3/2;
		last = first+1;
	}
	if (!(bh = bread(sb->s_dev,MSDOS_SB(sb)->fat_start+(first >>
	    SECTOR_BITS),SECTOR_SIZE))) {
		printk("bread in fat_access failed\n");
		return 0;
	}
	if ((first >> SECTOR_BITS) == (last >> SECTOR_BITS))
		bh2 = bh;
	else {
		if (!(bh2 = bread(sb->s_dev,MSDOS_SB(sb)->fat_start+(last
		    >> SECTOR_BITS),SECTOR_SIZE))) {
			brelse(bh);
			printk("bread in fat_access failed\n");
			return 0;
		}
	}
	if (MSDOS_SB(sb)->fat_bits == 16) {
		p_first = p_last = NULL; /* GCC needs that stuff */
		next = CF_LE_W(((unsigned short *) bh->b_data)[(first &
		    (SECTOR_SIZE-1)) >> 1]);
		if (next >= 0xfff7) next = -1;
	}
	else {
		p_first = &((unsigned char *) bh->b_data)[first & (SECTOR_SIZE-1)];
		p_last = &((unsigned char *) bh2->b_data)[(first+1) &
		    (SECTOR_SIZE-1)];
		if (nr & 1) next = ((*p_first >> 4) | (*p_last << 4)) & 0xfff;
		else next = (*p_first+(*p_last << 8)) & 0xfff;
		if (next >= 0xff7) next = -1;
	}
	if (new_value != -1) {
		if (MSDOS_SB(sb)->fat_bits == 16)
			((unsigned short *) bh->b_data)[(first & (SECTOR_SIZE-1)) >>
			    1] = CT_LE_W(new_value);
		else {
			if (nr & 1) {
				*p_first = (*p_first & 0xf) | (new_value << 4);
				*p_last = new_value >> 4;
			}
			else {
				*p_first = new_value & 0xff;
				*p_last = (*p_last & 0xf0) | (new_value >> 8);
			}
			mark_buffer_dirty(bh2, 1);
		}
		mark_buffer_dirty(bh, 1);
		for (copy = 1; copy < MSDOS_SB(sb)->fats; copy++) {
			if (!(c_bh = bread(sb->s_dev,MSDOS_SB(sb)->
			    fat_start+(first >> SECTOR_BITS)+MSDOS_SB(sb)->
			    fat_length*copy,SECTOR_SIZE))) break;
			memcpy(c_bh->b_data,bh->b_data,SECTOR_SIZE);
			mark_buffer_dirty(c_bh, 1);
			if (bh != bh2) {
				if (!(c_bh2 = bread(sb->s_dev,
				    MSDOS_SB(sb)->fat_start+(first >>
				    SECTOR_BITS)+MSDOS_SB(sb)->fat_length*copy
				    +1,SECTOR_SIZE))) {
					brelse(c_bh);
					break;
				}
				memcpy(c_bh2->b_data,bh2->b_data,SECTOR_SIZE);
				brelse(c_bh2);
			}
			brelse(c_bh);
		}
	}
	brelse(bh);
	if (bh != bh2) brelse(bh2);
	return next;
}


void cache_init(void)
{
	static int initialized = 0;
	int count;

	if (initialized) return;
	fat_cache = &cache[0];
	for (count = 0; count < FAT_CACHE; count++) {
		cache[count].device = 0;
		cache[count].next = count == FAT_CACHE-1 ? NULL :
		    &cache[count+1];
	}
	initialized = 1;
}


void cache_lookup(struct inode *inode,int cluster,int *f_clu,int *d_clu)
{
	struct fat_cache *walk;

#ifdef DEBUG
printk("cache lookup: <%s,%d> %d (%d,%d) -> ", kdevname(inode->i_dev),
       inode->i_ino, cluster, *f_clu, *d_clu);
#endif
	for (walk = fat_cache; walk; walk = walk->next)
		if (inode->i_dev == walk->device
		    && walk->ino == inode->i_ino
		    && walk->file_cluster <= cluster
		    && walk->file_cluster > *f_clu) {
			*d_clu = walk->disk_cluster;
#ifdef DEBUG
printk("cache hit: %d (%d)\n",walk->file_cluster,*d_clu);
#endif
			if ((*f_clu = walk->file_cluster) == cluster) return;
		}
#ifdef DEBUG
printk("cache miss\n");
#endif
}


#ifdef DEBUG
static void list_cache(void)
{
	struct fat_cache *walk;

	for (walk = fat_cache; walk; walk = walk->next) {
		if (walk->device)
			printk("<%s,%d>(%d,%d) ", kdevname(walk->device),
			       walk->ino, walk->file_cluster, walk->disk_cluster);
		else printk("-- ");
	}
	printk("\n");
}
#endif


void cache_add(struct inode *inode,int f_clu,int d_clu)
{
	struct fat_cache *walk,*last;

#ifdef DEBUG
printk("cache add: <%s,%d> %d (%d)\n", kdevname(inode->i_dev),
       inode->i_ino, f_clu, d_clu);
#endif
	last = NULL;
	for (walk = fat_cache; walk->next; walk = (last = walk)->next)
		if (inode->i_dev == walk->device
		    && walk->ino == inode->i_ino
		    && walk->file_cluster == f_clu) {
			if (walk->disk_cluster != d_clu) {
				printk("FAT cache corruption");
				fat_cache_inval_inode(inode);
				return;
			}
			/* update LRU */
			if (last == NULL) return;
			last->next = walk->next;
			walk->next = fat_cache;
			fat_cache = walk;
#ifdef DEBUG
list_cache();
#endif
			return;
		}
	walk->device = inode->i_dev;
	walk->ino = inode->i_ino;
	walk->file_cluster = f_clu;
	walk->disk_cluster = d_clu;
	last->next = NULL;
	walk->next = fat_cache;
	fat_cache = walk;
#ifdef DEBUG
list_cache();
#endif
}


/* Cache invalidation occurs rarely, thus the LRU chain is not updated. It
   fixes itself after a while. */

void fat_cache_inval_inode(struct inode *inode)
{
	struct fat_cache *walk;

	for (walk = fat_cache; walk; walk = walk->next)
		if (walk->device == inode->i_dev
		    && walk->ino == inode->i_ino)
			walk->device = 0;
}


void fat_cache_inval_dev(kdev_t device)
{
	struct fat_cache *walk;

	for (walk = fat_cache; walk; walk = walk->next)
		if (walk->device == device)
			walk->device = 0;
}


int get_cluster(struct inode *inode,int cluster)
{
	int nr,count;

	if (!(nr = MSDOS_I(inode)->i_start)) return 0;
	if (!cluster) return nr;
	count = 0;
	for (cache_lookup(inode,cluster,&count,&nr); count < cluster;
	    count++) {
		if ((nr = fat_access(inode->i_sb,nr,-1)) == -1) return 0;
		if (!nr) return 0;
	}
	cache_add(inode,cluster,nr);
	return nr;
}


int fat_smap(struct inode *inode,int sector)
{
	struct msdos_sb_info *sb;
	int cluster,offset;

	sb = MSDOS_SB(inode->i_sb);
	if (inode->i_ino == MSDOS_ROOT_INO || (S_ISDIR(inode->i_mode) &&
	    !MSDOS_I(inode)->i_start)) {
		if (sector >= sb->dir_entries >> MSDOS_DPS_BITS) return 0;
		return sector+sb->dir_start;
	}
	cluster = sector/sb->cluster_size;
	offset = sector % sb->cluster_size;
	if (!(cluster = get_cluster(inode,cluster))) return 0;
	return (cluster-2)*sb->cluster_size+sb->data_start+offset;
}


/* Free all clusters after the skip'th cluster. Doesn't use the cache,
   because this way we get an additional sanity check. */

int fat_free(struct inode *inode,int skip)
{
	int nr,last;

	if (!(nr = MSDOS_I(inode)->i_start)) return 0;
	last = 0;
	while (skip--) {
		last = nr;
		if ((nr = fat_access(inode->i_sb,nr,-1)) == -1) return 0;
		if (!nr) {
			printk("fat_free: skipped EOF\n");
			return -EIO;
		}
	}
	if (last)
		fat_access(inode->i_sb,last,MSDOS_SB(inode->i_sb)->fat_bits ==
		    12 ? 0xff8 : 0xfff8);
	else {
		MSDOS_I(inode)->i_start = 0;
		inode->i_dirt = 1;
	}
	lock_fat(inode->i_sb);
	while (nr != -1) {
		if (!(nr = fat_access(inode->i_sb,nr,0))) {
			fat_fs_panic(inode->i_sb,"fat_free: deleting beyond EOF");
			break;
		}
		if (MSDOS_SB(inode->i_sb)->free_clusters != -1)
			MSDOS_SB(inode->i_sb)->free_clusters++;
		inode->i_blocks -= MSDOS_SB(inode->i_sb)->cluster_size;
	}
	unlock_fat(inode->i_sb);
	fat_cache_inval_inode(inode);
	return 0;
}
