/*
 * Copyright (c) 2021 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */

package org.eclipse.trace4cps.ui.view;

import java.text.DecimalFormat;
import java.util.concurrent.TimeUnit;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.trace4cps.common.jfreechart.chart.axis.SectionAxis;
import org.eclipse.trace4cps.common.jfreechart.ui.viewers.ChartPanelContentViewer;
import org.eclipse.trace4cps.common.jfreechart.ui.viewers.ChartPanelSelectionHandler.SelectionType;
import org.eclipse.trace4cps.vis.jfree.TracePlotManager;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.plot.XYMeasureWithAnnotations;
import org.jfree.chart.plot.XYMeasureWithAnnotations.MeasurementAxis;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.Range;
import org.jfree.data.RangeAlign;

/**
 * This is a wrapper around the {@link TracePlotManager} that embeds it in the Eclipse UI.
 */
public class TraceViewer extends ChartPanelContentViewer {
    private static final DecimalFormat FORMAT = new DecimalFormat(".###");

    private final TracePlotManager plotMgr;

    public TraceViewer(Composite parent) {
        super(parent);
        setPreserveSelection(true);
        plotMgr = new TracePlotManager();
        plotMgr.setDataItemFactory(new EclipseDataItemFactory());
        plotMgr.setDelegateToolTipGenerator(new EclipseToolTipGenerator(plotMgr::getViewConfig));
    }

    public TracePlotManager getPlotManager() {
        return plotMgr;
    }

    public ChartPanel getChartPanel() {
        return super.getChartPanelComposite().getChartPanel();
    }

    @Override
    protected ChartPanelControlTuple createControl(Composite parent, int style, SelectionType selectionType) {
        final ChartPanelControlTuple controlTuple = super.createControl(parent, style, selectionType);
        final ChartPanel chartPanel = controlTuple.getChartPanelComposite().getChartPanel();

        // Measuring
        XYMeasureWithAnnotations measurer = XYMeasureWithAnnotations.addToChartPanel(chartPanel,
                MeasurementAxis.DOMAIN);
        measurer.setAnnotationLabelProvider((from, to) -> formatTimestamp(
                Math.abs(to.doubleValue() - from.doubleValue()), plotMgr.getTraces().get(0).getTimeUnit()));

        XYPlot xyPlot = new XYPlot();
        NumberAxis domainAxis = new NumberAxis();
        domainAxis.setAutoRangeIncludesZero(false);
        domainAxis.setAutoRangeAlign(RangeAlign.LOWER);
        // Set a maximum zoom level to prevent display issues
        domainAxis.setRangeMinimumSize(10e-6);
        xyPlot.setDomainAxis(domainAxis);

        SectionAxis rangeAxis = new SectionAxis();
        rangeAxis.setRangeMinimumSize(1);
        xyPlot.setRangeAxis(rangeAxis);

        // Setup the datasets and coloring etc of the plot:
        plotMgr.initializePlot(xyPlot);

        JFreeChart chart = new JFreeChart(null, null, xyPlot, false);
        ChartFactory.getChartTheme().apply(chart);
        chartPanel.setChart(chart);

        return controlTuple;
    }

    protected XYPlot getXYPlot() {
        return (XYPlot)super.getPlot();
    }

    @Override
    protected void refreshChart() {
        plotMgr.update(getXYPlot());
        ChartFactory.getChartTheme().apply(getChart());
    }

    public Range getRange() {
        return getXYPlot().getDomainAxis().getRange();
    }

    private String formatTimestamp(double value, TimeUnit tu) {
        if (1 <= value && value < 1000) {
            return " " + FORMAT.format(value) + " " + asString(tu) + " ";
        }
        // switch time unit:
        if (value < 1) {
            return upscale(value, tu);
        } else { // value >= 1000
            return downscale(value, tu);
        }
    }

    private String upscale(double value, TimeUnit tu) {
        switch (tu) {
            case NANOSECONDS:
                return FORMAT.format(value) + " " + asString(tu);
            case MICROSECONDS:
                return formatTimestamp(value * 1000d, TimeUnit.NANOSECONDS);
            case MILLISECONDS:
                return formatTimestamp(value * 1000d, TimeUnit.MICROSECONDS);
            case SECONDS:
                return formatTimestamp(value * 1000d, TimeUnit.MILLISECONDS);
            case MINUTES:
                return formatTimestamp(value * 60d, TimeUnit.SECONDS);
            case HOURS:
                return formatTimestamp(value * 60d, TimeUnit.MINUTES);
            case DAYS:
                return formatTimestamp(value * 24d, TimeUnit.HOURS);
            default:
                throw new IllegalStateException();
        }
    }

    private String downscale(double value, TimeUnit tu) {
        switch (tu) {
            case NANOSECONDS:
                return formatTimestamp(value / 1000d, TimeUnit.MICROSECONDS);
            case MICROSECONDS:
                return formatTimestamp(value / 1000d, TimeUnit.MILLISECONDS);
            case MILLISECONDS:
                return formatTimestamp(value / 1000d, TimeUnit.SECONDS);
            case SECONDS:
                return formatTimestamp(value / 60d, TimeUnit.MINUTES);
            case MINUTES:
                return formatTimestamp(value / 60d, TimeUnit.HOURS);
            case HOURS:
                return formatTimestamp(value / 24d, TimeUnit.DAYS);
            case DAYS:
                return FORMAT.format(value) + " " + asString(tu);
            default:
                throw new IllegalStateException();
        }
    }

    private String asString(TimeUnit tu) {
        switch (tu) {
            case NANOSECONDS:
                return "ns";
            case MICROSECONDS:
                return "us";
            case MILLISECONDS:
                return "ms";
            case SECONDS:
                return "s";
            case MINUTES:
                return "min";
            case HOURS:
                return "hr";
            case DAYS:
                return "day";
            default:
                throw new IllegalStateException();
        }
    }
}
