/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import java.net.URI;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;

/**
 * Java Regular Expression based {@link ProxyMapping} implementation.
 * <p>
 * For example, suppose this should cover the following mapping cases:
 * <ul>
 * <li>/portals/applications/a/b/c.html &lt;-&gt; http://portals.apache.org/applications/a/b/c.html</li>
 * <li>/portals/pluto/a/b/c.html &lt;-&gt; http://portals.apache.org/pluto/a/b/c.html</li>
 * <li>/portals/jetspeed-2/a/b/c.html &lt;-&gt; http://portals.apache.org/jetspeed-2/a/b/c.html</li>
 * </ul>
 * Then you might configure the properties of this like the following example to achieve
 * the mappings shown above:
 * <ul>
 * <li><code>localPattern</code>: "^/portals/([A-Za-z]+)/(.*)$"</li>
 * <li><code>remoteReplace</code>: "http://portals.apache.org/$1/$2"</li>
 * <li><code>remotePattern</code>: "^http://portals\\.apache\\.org/([A-Za-z]+)/(.*)$"</li>
 * <li><code>localReplace</code>: "/portals/$1/$2"</li>
 * </ul>
 * </p>
 * <p>
 * <em>Note: </em>
 * The <code>localPattern</code> and <code>remoteReplace</code> properties are used
 * in order to translate the local request path info to a remote target URI, whereas
 * the <code>remotePattern</code> and <code>localReplace</code> properties are used
 * in order to translate the remote location URI (e.g, redirection location) to a local
 * request path info.
 * </p>
 */
public class RegexProxyMapping extends AbstractProxyMapping
{

    /**
     * Regular expression to evaluate a local request path info.
     */
    private String localPattern;

    /**
     * Replace string to translate to a remote URI based on the regular expression
     * result from {@link #localPattern}.
     */
    private String remoteReplace;

    /**
     * Regular expression to evaluate a remote URL.
     */
    private String remotePattern;

    /**
     * Replace string to translate to a local request path info based on the regular expression
     * result from {@link #remotePattern}.
     */
    private String localReplace;

    /**
     * Internal pattern instance created from {@link #localPattern}.
     */
    private Pattern localPatternObject;

    /**
     * Internal pattern instance created from {@link #remotePattern}.
     */
    private Pattern remotePatternObject;

    /**
     * Zero-argument default constructor.
     */
    public RegexProxyMapping()
    {
    }

    /**
     * Returns the regular expression to evaluate a local request path info.
     * @return
     */
    public String getLocalPattern()
    {
        return localPattern;
    }

    /**
     * Sets the regular expression to evaluate a local request path info.
     * @param localPattern
     */
    public void setLocalPattern(String localPattern)
    {
        this.localPattern = localPattern;
        this.localPatternObject = Pattern.compile(localPattern);
    }

    /**
     * Returns the replace string to translate to a remote URI based on the regular expression
     * result from {@link #getLocalPattern()}.
     * @return
     */
    public String getRemoteReplace()
    {
        return remoteReplace;
    }

    /**
     * Sets the replace string to translate to a remote URI based on the regular expression
     * result from {@link #getLocalPattern()}.
     * @param remoteReplace
     */
    public void setRemoteReplace(String remoteReplace)
    {
        this.remoteReplace = remoteReplace;
    }

    /**
     * Returns the regular expression to evaluate a remote URL.
     * @return
     */
    public String getRemotePattern()
    {
        return remotePattern;
    }

    /**
     * Sets the regular expression to evaluate a remote URL.
     * @param remotePattern
     */
    public void setRemotePattern(String remotePattern)
    {
        this.remotePattern = remotePattern;
        this.remotePatternObject = Pattern.compile(remotePattern);
    }

    /**
     * Returns the replace string to translate to a local request path info based on the regular expression
     * result from {@link #getRemotePattern}.
     * @return
     */
    public String getLocalReplace()
    {
        return localReplace;
    }

    /**
     * Sets the replace string to translate to a local request path info based on the regular expression
     * result from {@link #getRemotePattern}.
     * @param localReplace
     */
    public void setLocalReplace(String localReplace)
    {
        this.localReplace = localReplace;
    }

    /**
     * {@inheritDoc}
     */
    public boolean matchesLocal(String localPath)
    {
        Matcher m = localPatternObject.matcher(localPath);
        return m.matches();
    }

    /**
     * {@inheritDoc}
     */
    public String resolveRemoteFromLocal(String localPath)
    {
        Matcher m = localPatternObject.matcher(localPath);

        if (m.matches())
        {
            return m.replaceFirst(remoteReplace);
        }

        return null;
    }

    /**
     * {@inheritDoc}
     */
    public boolean matchesRemote(URI remoteURI)
    {
        Matcher m = remotePatternObject.matcher(remoteURI.toString());
        return m.matches();
    }

    /**
     * {@inheritDoc}
     */
    public String resolveLocalFromRemote(URI remoteURI)
    {
        Matcher m = remotePatternObject.matcher(remoteURI.toString());

        if (m.matches())
        {
            return m.replaceFirst(localReplace);
        }

        return null;
    }

}
