/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.command;

import java.io.IOException;
import java.net.URI;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.chain.Command;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * {@link Command} responsible for handling HTTP Redirection response from the remote target content.
 * <p>
 * This tries to translate the <code>Location</code> HTTP response header value to a local path info
 * by using the currently resolved {@link ProxyMapping} first.
 * </p>
 * <p>
 * If it cannot resolve the mapped local path info from the currently resolved {@link ProxyMapping},
 * then it tries to translate the <code>Location</code> HTTP response header value to a local path info
 * by invoking {@link ProxyMappingRegistry.findProxyMappingByRemoteURI(URI)} afterward.
 * </p>
 */
public class HandleRedirectionCommand extends AbstractProxyCommand
{

    private static Logger log = LoggerFactory.getLogger(HandleRedirectionCommand.class);

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean executeInternal(final ProxyContext context) throws ReverseProxyException, IOException
    {
        HttpResponse httpResponse = context.getHttpResponse();

        int statusCode = httpResponse.getStatusLine().getStatusCode();

        // Check if the proxy response is a redirect
        if (statusCode < HttpServletResponse.SC_MULTIPLE_CHOICES /* 300 */
            || statusCode >= HttpServletResponse.SC_NOT_MODIFIED /* 304 */)
        {
            return false;
        }

        String location = null;
        Header locationHeader = httpResponse.getFirstHeader("Location");

        if (locationHeader != null)
        {
            location = locationHeader.getValue();
        }

        if (location == null)
        {
            throw new ReverseProxyException(HttpServletResponse.SC_BAD_GATEWAY, "No location header.");
        }

        // According to rfc2616, "Location" header value must be an absolute URI.
        // However, in case it is not, prepend the host URI of the proxy target URL.
        if (location.startsWith("/")) {
            URI targetURI = context.getRemoteURI();
            String scheme = targetURI.getScheme();
            int port = targetURI.getPort();
            StringBuilder uriBuilder = new StringBuilder(40).append(targetURI.getScheme()).append("://").append(targetURI.getHost());

            if (port > 0 && ((port != 80 && "http".equals(scheme)) || (port != 443 && "https".equals(scheme)))) {
                uriBuilder.append(':').append(port);
            }

            uriBuilder.append(location);
            location = uriBuilder.toString();
        }

        URI locationURI = null;

        try
        {
            locationURI = URI.create(location);
        }
        catch (Exception e)
        {
            log.error("Invalid redirect location: '{}'.", location);
            return true;
        }

        // Modify the redirect to go to this proxy servlet rather that the proxied host
        String localPath = context.getResolvedMapping().resolveLocalFromRemote(locationURI);

        // if the current proxy path mapper cannot map the remote location to local path, then
        // try to find out a possible path mapper instead one more...
        if (localPath == null)
        {
            ProxyMapping proxyMapping = context.getProxyMappingRegistry().findProxyMappingByRemoteURI(locationURI);

            if (proxyMapping != null)
            {
                localPath = proxyMapping.resolveLocalFromRemote(locationURI);
            }
        }

        String redirectLocation = null;

        if (localPath == null)
        {
            log.warn("Cannot translate the redirect location to local path. {}", location);
            redirectLocation = location;
        }
        else
        {
            redirectLocation = context.getRequestContext().getRequestBasePath() + localPath;
        }

        context.getRequestContext().sendRedirect(redirectLocation);

        return true;
    }

}
