/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.renderkit.html.util;

import javax.faces.context.FacesContext;

/**
 * Represents a single resource that a component in a page needs a
 * browser to fetch. This class helps generate the URI that is emitted
 * into the page, and specifies the class that should be invoked to
 * handle the request for that URI when the browser makes it.
 *
 * @author Mathias Broekelmann
 */
public interface ResourceHandler
{
    /**
     * Return a Class object whose instance can decode the url generated
     * by this class in the getResourceUri method and use that info to
     * locate the resource data represented by this object. When a
     * browser requests the data in the URL generated by this class
     * and its callers, an instance of the returned class shall be
     * created to decode the remainder of the url and serve the
     * resource.
     * 
     * @return a class which implements
     *   org.apache.myfaces.component.html.util.ResourceLoader
     *
     * @see ResourceLoader
     */
    public Class getResourceLoaderClass(); 
    
    /**
     * Returns the uri part which is used by the resourceloader to
     * identify the resource to load. This URI will be interpreted
     * by an instance of the class returned by getResourceLoaderClass. 
     * 
     * @see org.apache.myfaces.renderkit.html.util.ResourceLoader#serveResource(javax.servlet.ServletContext, javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, String)
     */
    public String getResourceUri(FacesContext context);
    
    /**
     * Must always be implemented when equals is overridden.
     * 
     * @see java.lang.Object#hashCode()
     */
    public int hashCode();
    
    /**
     * Must be implemented to avoid loading the same resource multiple times.
     * <p>
     * When the same component is used multiple times in a page and that
     * component needs an external resource such as a script, multiple calls
     * will be made to the AddResource methods for the same resource. The
     * AddResource class will create an instance of this class for each such
     * call. However if there is already a ResourceHandler instance existing
     * which is "equal" to the newly created one then a duplicate will not
     * be queued for output. 
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object obj);
}
