/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.commons.exporter.util;

import java.util.ArrayList;
import java.util.List;

import javax.faces.FacesException;
import javax.faces.component.EditableValueHolder;
import javax.faces.component.UIColumn;
import javax.faces.component.UIComponent;
import javax.faces.component.UIViewRoot;
import javax.faces.component.ValueHolder;
import javax.faces.component.html.HtmlDataTable;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;
import javax.faces.el.PropertyNotFoundException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;


public final class ComponentUtils
{

    private static final Log log = LogFactory.getLog(ComponentUtils.class);

    public static UIComponent findComponentById(FacesContext context,
            UIComponent root, String id) {

        UIComponent component = null;

        for (int i = 0; i < root.getChildCount() && component == null; i++)
        {
            UIComponent child = (UIComponent) root.getChildren().get(i);
            component = findComponentById(context, child, id);
        }

        if (root.getId() != null)
        {
            if (component == null && root.getId().equals(id))
            {
                component = root;
            }
        }
        return component;
    }

    public static String getPathToComponent(UIComponent component) {
        StringBuffer buf = new StringBuffer();

        if (component == null)
        {
            buf.append("{Component-Path : ");
            buf.append("[null]}");
            return buf.toString();
        }

        getPathToComponent(component, buf);

        buf.insert(0, "{Component-Path : ");
        buf.append("}");

        return buf.toString();
    }

    private static void getPathToComponent(UIComponent component,
            StringBuffer buf) {

        if (component == null)
            return;

        StringBuffer intBuf = new StringBuffer();

        intBuf.append("[Class: ");
        intBuf.append(component.getClass().getName());
        if (component instanceof UIViewRoot)
        {
            intBuf.append(",ViewId: ");
            intBuf.append(((UIViewRoot) component).getViewId());
        }
        else
        {
            intBuf.append(",Id: ");
            intBuf.append(component.getId());
        }
        intBuf.append("]");

        buf.insert(0, intBuf.toString());

        getPathToComponent(component.getParent(), buf);
    }

    private static Object getValue(UIComponent component) {
        Object value;
        try
        {
            value = ((ValueHolder) component).getValue();
        }
        catch (Exception ex)
        {
            throw new FacesException(
                    "Could not retrieve value of component with path : "
                            + getPathToComponent(component), ex);
        }
        return value;
    }

    public static String getStringValue(FacesContext facesContext,
            UIComponent component) {
        try
        {
            if (!(component instanceof ValueHolder))
            {
                throw new IllegalArgumentException("Component : "
                        + getPathToComponent(component)
                        + "is not a ValueHolder");
            }

            if (component instanceof EditableValueHolder)
            {
                Object submittedValue = ((EditableValueHolder) component)
                        .getSubmittedValue();
                if (submittedValue != null)
                {
                    if (submittedValue instanceof String)
                    {
                        if (log.isDebugEnabled())
                            log.debug("returning 1 '" + submittedValue + "'");
                        return (String) submittedValue;
                    }

                    throw new IllegalArgumentException(
                            "Expected submitted value of type String for component : "
                                    + getPathToComponent(component));
                }
            }

            Object value;

            if (component instanceof EditableValueHolder)
            {

                EditableValueHolder holder = (EditableValueHolder) component;

                if (holder.isLocalValueSet())
                {
                    value = holder.getLocalValue();
                }
                else
                {
                    value = getValue(component);
                }
            }
            else
            {
                value = getValue(component);
            }

            Converter converter = ((ValueHolder) component).getConverter();
            if (converter == null && value != null)
            {

                try
                {
                    converter = facesContext.getApplication().createConverter(
                            value.getClass());
                    if (log.isDebugEnabled())
                        log.debug("the created converter is " + converter);
                }
                catch (FacesException e)
                {
                    log.error("No converter for class "
                            + value.getClass().getName()
                            + " found (component id=" + component.getId()
                            + ").");
                    // converter stays null
                }
            }

            if (converter == null)
            {
                if (value == null)
                {
                    if (log.isDebugEnabled())
                        log.debug("returning an empty string");
                    return "";
                }

                if (log.isDebugEnabled())
                    log.debug("returning an .toString");
                return value.toString();

            }

            if (log.isDebugEnabled())
                log.debug("returning converter get as string " + converter);
            return converter.getAsString(facesContext, component, value);

        }
        catch (PropertyNotFoundException ex)
        {
            log.error("Property not found - called by component : "
                    + getPathToComponent(component), ex);

            throw ex;
        }
    }

    public static List getHTMLDataTableColumns(HtmlDataTable table) {
        List columns = new ArrayList();

        for (int i = 0; i < table.getChildCount(); i++)
        {
            UIComponent child = (UIComponent) table.getChildren().get(i);
            if (child instanceof UIColumn)
            {
                columns.add(child);
            }
        }
        return columns;
    }
}
