// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.cloudstack.api.command.user.backup;

import java.util.ArrayList;
import java.util.List;

import javax.inject.Inject;

import org.apache.cloudstack.acl.RoleType;
import org.apache.cloudstack.api.APICommand;
import org.apache.cloudstack.api.ApiConstants;
import org.apache.cloudstack.api.ApiErrorCode;
import org.apache.cloudstack.api.BaseListProjectAndAccountResourcesCmd;
import org.apache.cloudstack.api.Parameter;
import org.apache.cloudstack.api.ServerApiException;
import org.apache.cloudstack.api.response.BackupOfferingResponse;
import org.apache.cloudstack.api.response.BackupResponse;
import org.apache.cloudstack.api.response.ListResponse;
import org.apache.cloudstack.api.response.UserVmResponse;
import org.apache.cloudstack.api.response.ZoneResponse;
import org.apache.cloudstack.backup.Backup;
import org.apache.cloudstack.backup.BackupManager;
import org.apache.cloudstack.context.CallContext;

import com.cloud.exception.ConcurrentOperationException;
import com.cloud.exception.InsufficientCapacityException;
import com.cloud.exception.NetworkRuleConflictException;
import com.cloud.exception.ResourceAllocationException;
import com.cloud.exception.ResourceUnavailableException;
import com.cloud.utils.Pair;

@APICommand(name = "listBackups",
        description = "Lists VM backups",
        responseObject = BackupResponse.class, since = "4.14.0",
        authorized = {RoleType.Admin, RoleType.ResourceAdmin, RoleType.DomainAdmin, RoleType.User})
public class ListBackupsCmd extends BaseListProjectAndAccountResourcesCmd {

    @Inject
    private BackupManager backupManager;

    /////////////////////////////////////////////////////
    //////////////// API parameters /////////////////////
    /////////////////////////////////////////////////////

    @Parameter(name = ApiConstants.ID,
            type = CommandType.UUID,
            entityType = BackupResponse.class,
            description = "id of the backup")
    private Long id;

    @Parameter(name = ApiConstants.VIRTUAL_MACHINE_ID,
            type = CommandType.UUID,
            entityType = UserVmResponse.class,
            description = "id of the VM")
    private Long vmId;

    @Parameter(name = ApiConstants.ZONE_ID,
            type = CommandType.UUID,
            entityType = ZoneResponse.class,
            description = "list backups by zone id")
    private Long zoneId;

    @Parameter(name = ApiConstants.NAME,
            type = CommandType.STRING,
            since = "4.21.0",
            description = "list backups by name")
    private String name;

    @Parameter(name = ApiConstants.BACKUP_OFFERING_ID,
            type = CommandType.UUID,
            entityType = BackupOfferingResponse.class,
            since = "4.21.0",
            description = "list backups by backup offering")
    private Long backupOfferingId;

    @Parameter(name = ApiConstants.LIST_VM_DETAILS,
            type = CommandType.BOOLEAN,
            since = "4.21.0",
            description = "list backups with VM details")
    private Boolean listVmDetails;

    /////////////////////////////////////////////////////
    /////////////////// Accessors ///////////////////////
    /////////////////////////////////////////////////////

    public Long getId() {
        return id;
    }

    public Long getVmId() {
        return vmId;
    }

    public String getName() {
        return name;
    }

    public Long getBackupOfferingId() {
        return backupOfferingId;
    }

    public Long getZoneId() {
        return zoneId;
    }

    public Boolean getListVmDetails() {
        return listVmDetails;
    }

    /////////////////////////////////////////////////////
    /////////////// API Implementation///////////////////
    /////////////////////////////////////////////////////

    protected void setupResponseBackupList(final List<Backup> backups, final Integer count) {
        final List<BackupResponse> responses = new ArrayList<>();
        for (Backup backup : backups) {
            if (backup == null) {
                continue;
            }
            BackupResponse backupResponse = backupManager.createBackupResponse(backup, this.getListVmDetails());
            responses.add(backupResponse);
        }
        final ListResponse<BackupResponse> response = new ListResponse<>();
        response.setResponses(responses, count);
        response.setResponseName(getCommandName());
        setResponseObject(response);
    }

    @Override
    public void execute() throws ResourceUnavailableException, InsufficientCapacityException, ServerApiException, ConcurrentOperationException, ResourceAllocationException, NetworkRuleConflictException {
        try{
            Pair<List<Backup>, Integer> result = backupManager.listBackups(this);
            setupResponseBackupList(result.first(), result.second());
        } catch (Exception e) {
            logger.debug("Exception while listing backups", e);
            throw new ServerApiException(ApiErrorCode.INTERNAL_ERROR, e.getMessage());
        }
    }

    @Override
    public long getEntityOwnerId() {
        return CallContext.current().getCallingAccount().getId();
    }

}
