/* $Id: conv.c,v 1.7 2005/05/12 19:29:35 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * imp/src/conv.c,v 1.7 2004/12/15 04:58:42 flaw
 * if/src/conv.c,v 1.5 2004/09/28 15:28:11 flaw
 *//*
 * Conversion functions
 */
#include <setjmp.h>
#include <postgres.h>
#include <access/tupdesc.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <catalog/pg_cast.h>
#include <catalog/catversion.h>
#include <executor/tuptable.h>
#include <nodes/params.h>
#include <parser/parse_type.h>
#include <utils/array.h>
#include <utils/datum.h>
#include <utils/elog.h>
#include <utils/rel.h>
#include <utils/builtins.h>
#include <utils/palloc.h>
#include <utils/relcache.h>
#include <utils/syscache.h>
#include <utils/tuplestore.h>
#include <tcop/tcopprot.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>
#include <pypg/heaptuple.h>
#include <pypg/tupledesc.h>
#include <pypg/object.h>
#include <pypg/type.h>
#include <pypg/utils.h>
#include <pypg/conv.h>

PyObj
PyObject_FromTypeTupleAndDatum(HeapTuple tt, Datum d)
{
	PyObj rob = NULL;
	rob = PyPgObject_FromTypeOidAndDatum(HeapTupleGetOid(tt), d);
	return(rob);
}

#ifdef SYNTAX_ERRORS
void
PySequence_InfixWithTypesDatumsNulls(
	PyObj s,
	Oid *types,
	Datum *datums,
	bool *nulls)
{
	unsigned int c = PySequence_Length(s);
	int i;

	for (i = 0; i < c; ++i)
	{
		Datum d = datums[i];
		Oid t = types[i];
		bool n = nulls[i];
		PyObj to;
		Form_pg_type ts;

		to = PyPgType_FromOid(t);
		/* Borrow the reference from the cache */
		DECREF(to);

		ts = PyPgType_FetchTypeStruct(tt);

		if (n == true)
			arg = PyPgObject_New(NULL, to);
		else
			arg = PyPgObject_FromPyPgTypeAndDatum(to, d);

		if (arg == NULL || PySequence_SetItem(sq, i, arg) < 0)
			return;
	}
}
#endif

PyObj
PySequence_FromPyPgTupleDescAndTuplestore(PyObj tdo, Tuplestorestate *ts)
{
	PyObj hto, rob;
	bool should_free;
	HeapTuple ht;

	rob = PyList_New(0);
	if (rob == NULL)
		return(NULL);

	while ((ht = (HeapTuple) tuplestore_gettuple(ts, true, &should_free)))
	{
		HeapTuple htc;
		PythonMemoryContext(htc = heap_copytuple(ht));
		if (should_free)
			heap_freetuple(ht);
		if (htc == NULL)
			goto error;

		hto = PyPgHeapTuple_New(tdo, htc);
		if (hto == NULL)
		{
			heap_freetuple(htc);
			goto error;
		}

		PyList_Append(rob, hto);
		DECREF(hto);
		if (PyErr_Occurred())
			goto error;
	}

	return(rob);
error:
	DECREF(rob);
	return(NULL);
}

PyObj
PySequence_FromTupleDescAndTuplestore(TupleDesc td, Tuplestorestate *ts)
{
	PyObj tdo = PyPgTupleDesc_New(td), rob;
	if (tdo == NULL) return(NULL);
	rob = PySequence_FromPyPgTupleDescAndTuplestore(tdo, ts);
	DECREF(tdo);
	return(rob);
}

/*
 * PyPgFill_PySequence - fill a sequence with the datums and nulls
 *
 * Use expectation is datums and nulls coming from FunctionCallInfo
 */
PyObj
PyPgFill_PySequence
(
 	PyObj args,
	PyObj argtypes,
	FunctionCallInfo fcinfo
)
{
	int i, nargs = PyObject_Length(args);
	Datum *datums = fcinfo->arg;
	bool *nulls = fcinfo->argnull;

	if (nargs > 0) for (i = 0; i < nargs; ++i)
	{
		PyObj arg, argtype = PySequence_GetItem(argtypes, i);
		DECREF(argtype);

		if (nulls[i])
			arg = PyPgType_FetchNULL(argtype);
		else
			arg = PyPgObject_FromPyPgTypeAndDatum(argtype, datums[i]);
		
		if (arg == NULL) goto error;
		PyList_SetItem(args, i, arg);
	}

	return(args);
error:
	DECREF(args);
	return(NULL);
}

/*
 * HeapTuple_FromTupleDescAndPyObject
 *
 * Place a Python object into a new single attribute HeapTuple
 */
HeapTuple
HeapTuple_FromTupleDescAndPyObject(TupleDesc td, PyObj ob)
{
	char null = 'n';
	Datum value = 0;
	HeapTuple ht = NULL;

	Assert(td->natts == 1);

	if (ob == Py_None ||(PyPgObject_Check(ob) && PyPgObject_IsNULL(ob)))
	{
		null = 'n';
		value = 0;
	}
	else
	{
		null = ' ';
		value = Datum_FromTypeOidAndPyObject(td->attrs[0]->atttypid, ob);
	}

	ht = heap_formtuple(td, &value, &null);
	if (null == ' ' && !(td->attrs[0]->attbyval))
		pfree(DatumGetPointer(value));

	return(ht);
}

/*
 * HeapTuple_FromTupleDescAndIterable
 *
 * Convert a Python iterable to a HeapTuple
 */
HeapTuple
HeapTuple_FromTupleDescAndIterable(TupleDesc td, PyObj so)
{
	int natts = td->natts;
	PyObj attr = NULL;
	
	unsigned int i = 0;
	char *nulls = NULL;
	Datum *values = NULL;
	HeapTuple ht = NULL;
	PyObj iter;

	nulls = palloc(sizeof(char) * natts);
	values = palloc(sizeof(Datum) * natts);
	
	iter = PyObject_GetIter(so);
	while ((attr = PyIter_Next(iter)))
	{
		/* Ignore the overflow */
		if (i >= natts)
			break;

		if (attr == Py_None ||(PyPgObject_Check(attr) && PyPgObject_IsNULL(attr)))
		{
			nulls[i] = 'n';
			values[i] = 0;
		}
		else
		{
			nulls[i] = ' ';
			values[i] = Datum_FromTypeOidAndPyObject(td->attrs[i]->atttypid, attr);
		}

		DECREF(attr);
		++i;
	}
	DECREF(iter);
	
	/* Handle underflow by NULLing out the rest */
	if (i < td->natts)
		for (; i < td->natts; ++i)
		{
			nulls[i] = 'n';
			values[i] = 0;
		}

	ht = heap_formtuple(td, values, nulls);

	while (i-- != 0)
	{
		if (!(td->attrs[i]->attbyval) && nulls[i] != 'n')
			pfree(DatumGetPointer(values[i]));
	}
	pfree(values);
	pfree(nulls);

	return(ht);
}

HeapTuple
HeapTuple_FromRelationOidAndIterable(Oid rid, PyObj so)
{
	Relation rel;
	TupleDesc td;
	HeapTuple ht;

	if (PyPgHeapTuple_Check(so) && PyPgHeapTuple_FetchTableOid(so) == rid)
	{
		ht = heap_copytuple(PyPgHeapTuple_FetchHeapTuple(so));
		return(ht);
	}

	rel = RelationIdGetRelation(rid);
	td = RelationGetDescr(rel);
	ht = HeapTuple_FromTupleDescAndIterable(td, so);
	RelationClose(rel);

	if (ht)
		HeapTuple_FetchTableOid(ht) = rid;

	return(ht);
}

ArrayType *
Array_FromPySequence
(
	PyObj seq,
	Oid elmtoid,
	bool typbyval,
	int2 typlen,
	char typalign
)
{
	int i, nelems;
	Datum *des;
	bool *frees = NULL;
	ArrayType *ra = NULL;
	HeapTuple tt;

	nelems = PySequence_Length(seq);
	des = palloc(sizeof(Datum) * nelems);
	if (typbyval == false)
		frees = palloc(sizeof(bool) * nelems);

	tt = SearchSysCacheCopy(TYPEOID, ObjectIdGetDatum(elmtoid), 0, 0, 0);

	for (i = 0; i < nelems; ++i)
	{
		PyObj ob;
		ob = PySequence_GetItem(seq, i);
		if (frees) frees[i] = true;

		Py_DECREF(ob);
		if (PyPgObject_Check(ob))
		{
			Oid otoid = PyPgObject_FetchTypeOid(ob);
			if (otoid == elmtoid)
			{
				des[i] = PyPgObject_FetchDatum(ob);
				if (frees) frees[i] = false;
			}
			else
				des[i] = Datum_Cast(PyPgObject_FetchDatum(ob), otoid, elmtoid);
		}
		else
		{
			des[i] = Datum_FromTypeTupleAndPyObject(tt, ob);
		}
	}
	if (i == nelems)
		ra = construct_array(des, nelems, elmtoid, typlen, typbyval, typalign);

	if (frees)
	{
		for (i = 0; i < nelems; ++i)
		{
			if (frees[i])
				pfree(DatumGetPointer(des[i]));
		}
		pfree(frees);
	}
	pfree(des);
	heap_freetuple(tt);

	return(ra);
}

ArrayType *
Array_FromTypeTupleAndPySequence(HeapTuple ttup, PyObj seq)
{
	Form_pg_type t = TYPESTRUCT(ttup);
	Oid to = HeapTuple_FetchOid(ttup);
	return(Array_FromPySequence(seq, to, t->typbyval, t->typlen, t->typalign));
}

ArrayType *
Array_FromTypeOidAndPySequence(Oid typoid, PyObj seq)
{
	ArrayType *ar;
	HeapTuple tt;

	tt = SearchSysCache(TYPEOID, ObjectIdGetDatum(typoid), 0, 0, 0);
	ar = Array_FromTypeTupleAndPySequence(tt, seq);
	ReleaseSysCache(tt);

	return(ar);
}

Datum
Datum_FromTypeTupleAndPyString(HeapTuple tt, PyObj str)
{
	register Datum rd;
	Assert(PyString_Check(str));

	rd = OidFunctionCall1(TYPESTRUCT(tt)->typinput,
			PointerGetDatum(PyString_AS_STRING(str)));

	return(rd);
}

Datum
Datum_FromTypeTupleAndPyPgObject(HeapTuple tt, PyObj ob)
{
	Form_pg_type ts = TYPESTRUCT(tt);
	Oid obtypoid = PyPgObject_FetchTypeOid(ob);
	Oid ttoid = HeapTuple_FetchOid(tt);
	Datum od = PyPgObject_FetchDatum(ob);
	Datum rd;
	Assert(PyPgObject_Check(ob));

	if (ttoid == obtypoid)
		rd = datumCopy(od, ts->typbyval, ts->typlen);
	else
		rd = Datum_Cast(od, obtypoid, ttoid);

	return(rd);
}

Datum
Datum_FromTypeTupleAndPyObject(HeapTuple tt, PyObj ob)
{
	Form_pg_type ts = TYPESTRUCT(tt);
	Datum rd = 0;

	Assert(tt != NULL && ob != NULL);

	if (PyString_Check(ob))
	{
		rd = Datum_FromTypeTupleAndPyString(tt, ob);
	}
	else if (PyPgObject_Check(ob))
	{
		rd = Datum_FromTypeTupleAndPyPgObject(tt, ob);
	}
	else if (ts->typelem != InvalidOid)
	{
		ArrayType *ar;
		ar = Array_FromTypeOidAndPySequence(ts->typelem, ob);
		rd = PointerGetDatum(ar);
	}
	else if (ts->typrelid != InvalidOid)
	{
		HeapTuple ht;
		ht = HeapTuple_FromRelationOidAndIterable(ts->typrelid, ob);
		rd = PointerGetDatum(ht);
	}
	else
	{
		PyObj obstr = NULL;

		obstr = PyObject_Str(ob);
		PG_TRY();
		{
			rd = Datum_FromTypeTupleAndPyString(tt, obstr);
		}
		PG_CATCH();
		{
			DECREF(obstr);
			PG_RE_THROW();
		}
		PG_END_TRY();

		DECREF(obstr);
	}

	return(rd);
}

Datum
Datum_FromTypeOidAndPyObject(Oid typoid, PyObj ob)
{
	HeapTuple tt;
	Datum rd;

	tt = SearchSysCache(TYPEOID, ObjectIdGetDatum(typoid),0,0,0);
	rd = Datum_FromTypeTupleAndPyObject(tt, ob);
	ReleaseSysCache(tt);

	return(rd);
}

Oid
Oid_FromPyObject(PyObj ob)
{
	Oid ro = InvalidOid;

	if (PyPgType_Check(ob))
	{
		ro = PyPgObject_FetchOid(ob);
	}
	else if (PyPgObject_Check(ob))
	{
		Oid typeoid = PyPgObject_FetchTypeOid(ob);
		switch (typeoid)
		{
			case REGPROCOID:
			case REGTYPEOID:
			case OIDOID:
				ro = DatumGetObjectId(PyPgObject_FetchDatum(ob));
			break;
		}
	}
	else
	{
		PyObj io = PyNumber_Int(ob);
		if (io == NULL)
		{
			PyErr_Clear();
			return(InvalidOid);
		}

		if (PyInt_Check(io))
			ro = (Oid) PyInt_AsUnsignedLongMask(io);
		else
			PyErr_SetString(PyExc_OverflowError,
					"object overflowed integer for Oid");

		DECREF(io);
	}

	return(ro);
}
/*
 * vim: ts=3:sw=3:noet:
 */
