#if GEGL_CHANT_PROPERTIES

gegl_chant_int (radius, 1, 80, 4, "Size of block taken into account when computing the complexity")

#else
#define GEGL_CHANT_FILTER
#define GEGL_CHANT_NAME         complexity
#define GEGL_CHANT_DESCRIPTION  "foo"
#define GEGL_CHANT_SELF         "complexity.c"
#define GEGL_CHANT_CATEGORIES   "blur"
#define GEGL_CHANT_CLASS_INIT
#include "gegl-chant.h"

static double
pointcloud_complexity (int            width,
                       int            height,
                       int            row_stride,
                       unsigned char *pixels,
                       int            x0,
                       int            y0,
                       int            x1,
                       int            y1);

static GeglRectangle get_source_rect (GeglOperation *self,
                                      gpointer       context_id);

static gboolean
process (GeglOperation *operation,
         gpointer       context_id)
{
  GeglOperationFilter *filter;
  GeglChantOperation  *self;
  GeglBuffer          *input;
  GeglBuffer          *output;
  GeglRectangle        need   = get_source_rect (operation, context_id);

  guchar              *in_pixels;
  gfloat              *out_pixels;

  filter = GEGL_OPERATION_FILTER (operation);
  self   = GEGL_CHANT_OPERATION (operation);

  input = GEGL_BUFFER (gegl_operation_get_data (operation, context_id, "input"));

  in_pixels = g_malloc (need.w * need.h * 3 * sizeof (guchar));
  out_pixels = g_malloc (need.w * need.h * 1 * sizeof (gfloat));

  output = g_object_new (GEGL_TYPE_BUFFER,
                         "format", babl_format ("Y float"),
                         "x",      need.x,
                         "y",      need.y,
                         "width",  need.w,
                         "height", need.h,
                         NULL);

  gegl_buffer_get (input, &need, in_pixels, babl_format ("R'G'B' u8"), 1.0);

  {
    gint x,y;
    gint radius = self->radius;
    for (y=0;y<need.h;y++)
      for (x=0;x<need.w;x++)
        out_pixels[(y*need.w+x)] = 
          pointcloud_complexity (need.w, need.h, need.w * 3,
                                 in_pixels,
                                 x-radius,y-radius,x+radius,y+radius);
  }
  gegl_buffer_set (output, &need, out_pixels, babl_format ("Y float"));

  g_free (in_pixels);
  g_free (out_pixels);

  gegl_operation_set_data (operation, context_id, "output", G_OBJECT (output));
  return FALSE;
}

static GeglRectangle
get_source_rect (GeglOperation *operation,
                 gpointer       context_id)
{
  GeglRectangle        rect;
  GeglChantOperation  *self = GEGL_CHANT_OPERATION (operation);

  rect  = *gegl_operation_get_requested_region (operation, context_id);
  if (rect.w != 0 &&
      rect.h != 0)
    {
      rect.x -= self->radius;
      rect.y -= self->radius;
      rect.w += self->radius*2;
      rect.h += self->radius*2;
    }
  return rect;
}

static gboolean
calc_source_regions (GeglOperation *self,
                     gpointer       context_id)
{
  GeglRectangle need = get_source_rect (self, context_id);

  gegl_operation_set_source_region (self, context_id, "input", &need);

  return TRUE;
}

static GeglRectangle
get_affected_region (GeglOperation *operation,
                     const gchar   *input_pad,
                     GeglRectangle  region)
{
  GeglChantOperation  *self = GEGL_CHANT_OPERATION (operation);

  region.x -= self->radius;
  region.y -= self->radius;
  region.w += self->radius*2;
  region.h += self->radius*2;
  return region;
}

static void class_init (GeglOperationClass *operation_class)
{
  operation_class->get_affected_region = get_affected_region;
  operation_class->calc_source_regions = calc_source_regions;
}

#ifdef COMMENT
#include <stdlib.h>
                  ___________.
                 /101  /111 / |
                /_____/____/  |
               /     /    /| /|
              /_____/____/ |/ |
 010         |001  |011 |  |110
   \_________|. .  |    | /| /
             |_____|____|/ |/
             |000  |010 |  /
             |     |    | /
             |_____|____|/

#endif

typedef struct oct_t
{
  unsigned int  count;
  struct oct_t *child[8];
  unsigned int  sum[4];
  unsigned char min[4];
  unsigned char max[4];
} oct_t;

/* have a mean to address the parent?
 * bookeeping variable for variables for a filling pass?
 *
 */

oct_t *
oct_new (void)
{
  oct_t *self = calloc (sizeof (oct_t), 1);
  self->min[0]=255;
  self->min[1]=255;
  self->min[2]=255;
  return self;
}

void
oct_free (oct_t *self)
{
  int i;
  for (i=0;i<8;i++)
    if (self->child[i])
      oct_free (self->child[i]);
  free (self);
}


/* add a point to our octree volume,
 * implemented sequentially on a bit encoded adress 
 * instead of recursivly * and operating directly at a pointer to a 8bit
 * rgb triplet to allow traversing an in memory-image
 *
 * 
 *
 */
void
oct_increment (oct_t *root,
               unsigned char value[3])
{
   oct_t *iter=root;  /* our iter is our pointer to the location in the
                         data structure, we start pointing at the root
                         of the tree */
   int bit_no;        /* Which bit_no we are analyzing, */

   for(bit_no=7; bit_no>=0; bit_no--)
     {
       int i;

       iter->count ++;

       for (i=0; i<3 ; i++)
         {
           iter->sum[i] += value[i];
           if (value[i] < iter->min[i])
             iter->min[i]=value[i];
           if (value[i] > iter->max[i])
             iter->max[i]=value[i];
         }

       if (bit_no>=0)
         {
           int r,g,b, child_id;

           /* calculate which of the eight children we should visit */
           r = (value[0] & (1 << bit_no)) ? 1 : 0;
           g = (value[1] & (1 << bit_no)) ? 2 : 0;
           b = (value[2] & (1 << bit_no)) ? 4 : 0;

           child_id = r + g + b;

           /* add new child if adress was unvisited */

           if (!iter->child[child_id])
             iter->child[child_id] = oct_new ();

           /* go to new child */
           iter = iter->child[child_id];
         }
     }
}


static long pow_8[] = {
  1,
  8,
  64,
  512,
  4096,
  32768,
  262144,
  2097152,
  16777216,
  134217728,
  1073741824
};

static double
oct_complexity (oct_t *item,
                int    depth)
{
  long sum=0;
  int i;

  sum += (item->max[0]-item->min[0]) *
         (item->max[1]-item->min[1]) *
         (item->max[2]-item->min[2]);

  for (i=0; i<8; i++)
    if (item->child[i])
      sum += oct_complexity (item->child[i], depth+1);
  
  return 1.0 * sum / pow_8[depth+1];
}

static double
pointcloud_complexity (int            width,
                       int            height,
                       int            row_stride,
                       unsigned char *pixels,
                       int            x0,
                       int            y0,
                       int            x1,
                       int            y1)
{
  int x,y;
  double complexity;

  oct_t *root = oct_new (); 

  for (y=y0; y<y1; y++)
    {
      unsigned char *pix = &(pixels[y*row_stride+x0*3]);
      for (x=x0; x<x1; x++)
        {
           oct_increment (root, pix);
           pix+=3;
        }
    }

  complexity = oct_complexity (root, 0);
  oct_free (root);
  return complexity / (256.0*256.0*256.0);  /* 8 is the depth reached for
                                                   8bit data */
}

#endif
