/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 *
 * $Id: RDFSReasonerImpl.java,v 1.4 2007/04/03 10:13:59 lzhang Exp $
 */
package org.eclipse.eodm.rdf.reasoner.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.eodm.rdf.rdfbase.RDFGraph;
import org.eclipse.eodm.rdf.rdfbase.RDFProperty;
import org.eclipse.eodm.rdf.rdfbase.RDFSResource;
import org.eclipse.eodm.rdf.rdfs.RDFSClass;
import org.eclipse.eodm.rdf.reasoner.RDFSReasoner;
import org.eclipse.eodm.rdf.reasoner.RDFSReasonerException;
import org.eclipse.eodm.util.Triple;




/**
 * 
 * RDFSReasonerImpl : The implementation of RDFReasoner, It provide functions to
 * <li>1. Initialize a reasoner from given rdf model</li>
 * <li>2. Validate a given ontology model</li>
 * <li>3. Get all entailed sub/super classes of a RDFSClass</li>
 * <li>4. Get all entailed sub/super properties of a RDFProperty</li>
 * <li>5. Get all entailed instances that are rdf:type of a RDFSClass</li>
 * <li>6. Get all entailed triples of a RDFProperty</li>
 * <li>7. Test whether two RDFSClass have subClassOf/superClassOf relationship
 * </li>
 * <li>8. Test whether two RDFProperty have subPropertyOf/superPropertyOf
 * relationship</li>
 * <br/><br/>The basic idea of the implementation is to maintain a memory
 * model of RDF graph, using DAG Navigating Algorithms to do runtime
 * entailments, insteading of creating & inserting new entailed triples to
 * original models. Due to the nature of not creating new triples, all anonymous
 * node related RDFS entailments rules & RDF Axioms related rules will not be
 * supported. But we maintain the most of useful & practical RDFS inference.
 * <br/><br/>Currently, the semantics of entailment rules are captured by
 * Inference:
 * 
 * <li>rdfs2 aaa rdfs:domain xxx & uuu aaa yyy -> uuu rdf:type xxx</li>
 * <li>rdfs3 aaa rdfs:range xxx & uuu aaa vvv -> vvv rdf:type xxx</li>
 * <li>rdfs5 uuu rdfs:subPropertyOf vvv & vvv rdfs:subPropertyOf xxx -> uuu
 * rdfs:subPropertyOf xxx</li>
 * <li>rdfs6 uuu rdf:type rdf:Property -> uuu rdfs:subPropertyOf uuu</li>
 * <li>rdfs7 aaa rdfs:subPropertyOf bbb & uuu aaa yyy -> uuu bbb yyy</li>
 * <li>rdfs9 uuu rdfs:subClassOf xxx & vvv rdf:type uuu -> vvv rdf:type xxx
 * </li>
 * <li>rdfs10 uuu rdf:type rdfs:Class -> uuu rdfs:subClassOf uuu</li>
 * <li>rdfs11 uuu rdfs:subClassOf vvv & vvv rdfs:subClassOf xxx -> uuu
 * rdfs:subClassOf xxx</li>
 * <br/><br/>The semantic of below entailment rules are capture when parser
 * building up the RDF Model
 * <li>rdf1 uuu aaa yyy -> aaa rdf:type rdf:Property</li>
 * <li>rdfs4a uuu aaa xxx -> uuu rdf:type rdfs:Resource .</li>
 * <li>rdfs4b uuu aaa vvv -> vvv rdf:type rdfs:Resource .</li>
 * <li>xsd1a uuu aaa "sss" -> uuu aaa "sss"^^xsd:string</li>
 * <br/><br/>The semantic of below entailment rules is implied in the
 * structure of model
 * <li>rdfs8 uuu rdf:type rdfs:Class -> uuu rdfs:subClassOf rdfs:Resource</li>
 * <li>rdfs13 uuu rdf:type rdfs:Datatype -> uuu rdfs:subClassOf rdfs:Literal .
 * </li>
 *  
 */

public class RDFSReasonerImpl implements RDFSReasoner {
    /**
     * Setting up reasoner to initialize some necessary information for graph
     * algorithm. At the same time, validating the given RDFS model.
     * 
     * @param graph
     *            rdf ontology
     * @throws
     *            <p>
     *            RDFSModelInconsistencyException
     *            </p>
     *            if there exist some facts that will cause contraditions in the
     *            context of RDF/RDFS Semantic. Currently the situations are:
     *            <li>Assert subClassOf relationship among xsd:datatype which
     *            are contradictory with xml specification</li>
     *            <li>Statement object that is xml literal is incompatible with
     *            declaring ranges of the predicate</li>
     *            <br/><br/>
     * @throws RDFSReasonerException
     *             if there exist some facts in the model that are valid, but
     *             may possibly cause problems. Currently the situations are:
     *             <li>There exist cycles between RDFSClasses or RDFProperties
     *             when defining subClassOf or subPropertyOf relationship</li>
     *             <li>There exist multiple domain or range of a RDFProperty
     *             </li>
     */
    public void initialize(RDFGraph graph) throws RDFSReasonerException {

    	try {
    		navigateModel(graph);
    	} catch (Exception e){
    		
    	}
    	
    	gModel = graph;
    	
        Report report = new Report();
        
        //validateModel(classes, properties, report);

        if (!report.errorMsg.isEmpty()) {

        }

        if (!report.warningMsg.isEmpty()) {

        }
    }

    /**
     * Setting up reasoner without validating the model, initializing some
     * necessary information for graph algorithm.
     * <ol>
     * <li>Setting up visiting tag</li>
     * <li>Organizing the original model: append every individual to the
     * RDFSClass it directly belong to, every triple to the RDFProperty
     * according to its predicate</li>
     * </ol>
     * 
     * @param model
     *            rdf ontology
     */
    public void initializeWithoutValidation(RDFGraph model) {
        //model.eAdapters().add(this);
        try {
        	navigateModel(model);
        } catch (Exception e){
        	
        }
    }

    /**
     * Get all entailed sub classes of a RDFSClass by navigating the DAG
     * 
     * @param c
     *            the RDFSClass to be queried
     * @return all entailed sub classes of the given RDFSClass
     */
    public List getDescendantClasses(RDFSClass c) {
        ArrayList result = new ArrayList();

        if (c != null) {
            long tag = applyAccessTag(); // get unique access tag

            ArrayList queue = new ArrayList();
            queue.add(c);

            while (!queue.isEmpty()) {
                RDFSClass node = (RDFSClass) queue.remove(0);
                AccessInfo info = accessMap.getAccessInfo(node);
                
                if (info.visitTag != tag) {
                    info.visitTag = tag;
                    result.add(node);
                    queue.addAll( node.getSubClass() );
                }
            }
        }

        return result;
    }

    /**
     * Get all entailed super classes of a RDFSClass by navigating the DAG
     * 
     * @param c
     *            the RDFSClass to be queried
     * @return all entailed super classes of the given RDFSClass c
     */
    public List getAncestorClasses(RDFSClass c) {
    	ArrayList result = new ArrayList();

        if (c != null) {
            long tag = applyAccessTag();
            ArrayList queue = new ArrayList();
            queue.add(c);

            while (!queue.isEmpty()) {
                RDFSClass node = (RDFSClass) queue.remove(0);
                AccessInfo info = accessMap.getAccessInfo(node);
                if (info.visitTag != tag) {
                    info.visitTag = tag;
                    result.add(node);
                    queue.addAll( node.getRDFSsubClassOf() );

                }
            }
        }

        return result;
    }

    /**
     * Get all entailed sub properties of a RDFProperty by navigating the DAG
     * 
     * @param property
     *            the RDFProperty to be queried
     * @return all entailed sub properties of the given property
     */
    public List getDescendantProperties(RDFProperty property) {
        long tag = applyAccessTag();

        ArrayList result = new ArrayList();

        if (property != null) {
        	ArrayList queue = new ArrayList();
            queue.add(property);

            while (!queue.isEmpty()) {
                RDFProperty node = (RDFProperty) queue.remove(0);
                AccessInfo info = accessMap.getAccessInfo(node);
                if (info.visitTag != tag) {
                    info.visitTag = tag;
                    result.add(node);
                    queue.addAll( node.getSubProperty() );
                }
            }
        }

        return result;
    }

    /**
     * Get all entailed super properties of a RDFProperty by navigating the DAG
     * 
     * @param property
     *            the RDFProperty to be queried
     * @return all entailed sub properties of the given property
     */
    public List getAncestorProperties(RDFProperty property) {
        ArrayList result = new ArrayList();

        if (property != null) {
            ArrayList queue = new ArrayList();
            queue.add(property);
            long tag = applyAccessTag();

            while (!queue.isEmpty()) {
                RDFProperty node = (RDFProperty) queue.remove(0);
                AccessInfo info = accessMap.getAccessInfo(node);
                if (info.visitTag != tag) {
                    info.visitTag = tag;
                    result.add(node);
                    queue.addAll( node.getRDFSsubPropertyOf() );
                }
            }
        }

        return result;
    }

    /**
     * Test whether two RDFSClass have subClassOf relationship
     * 
     * @param c1
     *            RDFSClass
     * @param c2
     *            RDFSClass
     * @return true if c1 is subclass of c2
     */
    public boolean isDescendantClassOf(RDFSClass c1, RDFSClass c2) {
        if (c1 != null && c2 != null) {

            ArrayList queue = new ArrayList();
            queue.add(c2);
            long tag = applyAccessTag();

            while (!queue.isEmpty()) {
                RDFSClass node = (RDFSClass) queue.remove(0);
                if (node == c1)
                    return true;

                AccessInfo info = accessMap.getAccessInfo(node);
                if (info.visitTag != tag) {
                    info.visitTag = tag;
                    queue.addAll( node.getSubClass() );
                }
            }
        }

        return false;
    }

    /**
     * Test whether two RDFSClass have superClassOf relationship
     * 
     * @param c1
     *            RDFSClass
     * @param c2
     *            RDFSClass
     * @return true if c1 is super class of c2
     */
    public boolean isAncestorClassOf(RDFSClass c1, RDFSClass c2) {
        return !isDescendantClassOf(c2, c1);
    }

    /**
     * Test whether two RDFProperty have subPropertyOf relationship
     * 
     * @param p1
     *            RDFProperty
     * @param p2
     *            RDFProperty
     * @return true if p1 is subPropertyOf of p2
     */
    public boolean isDescendantPropertyOf(RDFProperty p1, RDFProperty p2) {
        if (p1 != null && p2 != null) {
            ArrayList queue = new ArrayList();
            queue.add(p2);
            long tag = applyAccessTag();
            while (!queue.isEmpty()) {
                RDFProperty node = (RDFProperty) queue.remove(0);
                if (node == p1)
                    return true;

                AccessInfo info = accessMap.getAccessInfo(node);
                if (info.visitTag != tag) {
                    info.visitTag = tag;
                    queue.addAll( node.getSubProperty() );
                }
            }
        }
        return false;
    }

    /**
     * Test whether two RDFProperty have superPropertyOf relationship
     * 
     * @param p1
     *            RDFProperty
     * @param p2
     *            RDFProperty
     * @return true if p1 is superPropertyOf of p2
     */
    public boolean isAncestorPropertyOf(RDFProperty p1, RDFProperty p2) {
        return !isDescendantPropertyOf(p2, p1);
    }


    /**
     * Navigating the whole model to
     * <li>place all classes to list - classes</li>
     * <li>place all properties to list - properties</li>
     * 
     * @param graph
     *            Given rdf ontology
     */
    private void navigateModel(RDFGraph graph) throws Exception {
        classes = graph.getTypeResources(RDFS_CLASS);
        properties = graph.getTypeResources(RDF_PROPERTY);
        
        // navigating the whole resource list to build up some cache
        for(Iterator it = graph.exportTriples().iterator();it.hasNext();){
        	Triple t = (Triple)it.next();
        	
        	String predicate = t.getPredicate();
        	if(predicate.equals(RDF_TYPE) ) {
        		RDFSResource type =  graph.getRDFSResource(t.getObjectURI());
        		if(type != null) {
        			RDFSResource subj = graph.getRDFSResource( t.getSubjectURI());
        			if(subj!=null) {
        				accessMap.getAccessInfo(type).cache.add(subj);
        			}
        		}
        		
        	} else {
	        	// normal triple
        		RDFSResource p = graph.getRDFSResource( predicate );
        		if( p != null ){
        			accessMap.getAccessInfo(p).cache.add(t);
        		}
        	}
        	
        }
    }

    /**
     * Validating given rdf/rdfs model
     * 
     * 
     * @param model
     *            the given rdf/rdfs model
     * @throws InconsistentOntologyException
     *             if there exist some facts that will cause contraditions in
     *             the context of RDF/RDFS Semantic. Currently the situations
     *             are:
     *             <li>there exist improper subClassOf declaration among
     *             xsd:datatypes, which are contradictory with xml specification
     *             </li>
     *             <li>XML literals that appear as predicate objects are
     *             incompatible with declared ranges</li>
     *             <br/><br/>
     * @throws RDFSReasonerException
     *             if there exist some facts in the model that are valid, but
     *             may possibly cause problems. Currently the situations are:
     *             <li>There exist cycles between subClassOf or subPropertyOf
     *             definitions</li>
     *             <li>There are multiple domains or ranges for a RDFProperty
     *             </li>
     *  
     */
    public void validateOntology(RDFGraph model) {

    	try {
    		navigateModel(model);
    	} catch (Exception e) {
    		
    	}
        // Report report = new Report();

        // validation
        //validateModel(classes, properties, report);

//        if (!report.errorMsg.isEmpty()) {
//            throw new InconsistentOntologyException(report.errorMsg.toString());
//        }
//
//        if (!report.warningMsg.isEmpty()) {
//            throw new RDFSReasonerException(report.warningMsg.toString());
//        }
    }

    /**
     * Validate the rdf model and return a report containing validation
     * messages, including Error & Warning Messages <br/><br/>Algorithm of
     * checking subclass/subproper cycle is to use topo sorting to find out
     * cycles. It starts with a nodes, and then topo-sort all its descendants in
     * descending manner, until the connective-subGraph can't be expanded. <br/>
     * <br/>The topoIndex denotes the topo-order of current expanded nodes.
     * startTag denotes the topo-value of the starting point of current
     * connective-subGraph. If it is find out that a node's topo-order is bigger
     * than one of its parent & smaller than the startingTag, it means there
     * exists a cycle between this node and its parent. <br/><br/>
     * 
     * For efficiency consideration, we will doing all other validation when
     * navigating all classes & properties
     * 
     * @param classes
     *            a list of all RDFSClasses in the model
     * @param properties
     *            a list of all RDFProperties in the model
     * @param report
     *            the return result after checking
     *  
     */
/*    protected void validateModel(ArrayList classes, ArrayList properties,
            Report report) {
        ArrayList topoQueue = new ArrayList();
        ArrayList expansionQueue = new ArrayList();

        int topoIndex = classes.size() + 100; // assign a large enough int as
        // starting tag
        for (Iterator iter = classes.iterator(); iter.hasNext();) {
            RDFSClass clazz = (RDFSClass) iter.next();
            // if clazz is a xsd datatype, checking whether exist improper
            // hierarchy declaration
            Namespace ns;
            if ((ns = clazz.getNamespace()) != null
                && ns.getURI() == XSD_NAMESPACE) {
                CheckXSDDatatypeHierarchy(clazz, report);
            }

            ResourceInfo classInfo = accessMap.getInfo(clazz);
            if (classInfo.visitTag == 0) {
                classInfo.visitTag = topoIndex;
                topoQueue.add(clazz);

                // using topo sorting to find out definition cycle
                while (!topoQueue.isEmpty()) {
                    int startTag = topoIndex--;
                    for (Iterator iterTopoQueue = topoQueue.iterator(); iterTopoQueue
                            .hasNext();) {
                        RDFSClass nextExpand = (RDFSClass) iterTopoQueue.next();
                        for (Iterator iterExp = nextExpand.getSubClass(false)
                                .iterator(); iterExp.hasNext();) {
                            RDFSClass expanded = (RDFSClass) iterExp.next();
                            if (nextExpand == expanded) {
                                continue; // skip self
                            }

                            ResourceInfo info = accessMap.getInfo(expanded);
                            if (info.visitTag == 0) {
                                info.visitTag = topoIndex; // the class has not
                                // been visited yet
                                expansionQueue.add(expanded);
                            } else if (info.visitTag > topoIndex
                                       && info.visitTag <= startTag) {
                                report.errorMsg
                                        .add("Error: There exists a subClassOf cycle between "
                                             + nextExpand.getURI()
                                             + " and "
                                             + expanded.getURI() + "\n");
                            }
                        }
                    }

                    ArrayList temp = topoQueue;
                    topoQueue = expansionQueue;
                    expansionQueue = temp;
                    expansionQueue.clear();
                }
            }
        }

        topoIndex = properties.size() + 100;// assign a large enough num as
        // starting tag
        topoQueue.clear();
        expansionQueue.clear();

        for (Iterator iter = properties.iterator(); iter.hasNext();) {
            RDFProperty property = (RDFProperty) iter.next();
            // checking whether there exist multiple domains or ranges for a
            // property, adding Warning
            checkMultiDomainRange(property, report);

            checkRangeValues(property, report);

            // using topo sorting to find out definition cycle
            ResourceInfo propertyInfo = accessMap.getInfo(property);
            if (propertyInfo.visitTag == 0) {
                propertyInfo.visitTag = topoIndex;
                topoQueue.add(property);

                while (!topoQueue.isEmpty()) {
                    int startTag = topoIndex--;
                    for (Iterator iterTopoQueue = topoQueue.iterator(); iterTopoQueue
                            .hasNext();) {
                        RDFProperty nextExpand = (RDFProperty) iterTopoQueue
                                .next();
                        for (Iterator iterExp = nextExpand
                                .getSubProperty(false).iterator(); iterExp
                                .hasNext();) {
                            RDFProperty expanded = (RDFProperty) iterExp.next();
                            if (nextExpand == expanded) {
                                continue; // skip self
                            }

                            ResourceInfo info = accessMap.getInfo(expanded);
                            if (info.visitTag == 0) {
                                info.visitTag = topoIndex; // the property has
                                // not been visited
                                // yet
                                expansionQueue.add(expanded);
                            } else if (info.visitTag > topoIndex
                                       && info.visitTag <= startTag) {
                                report.errorMsg
                                        .add("Error: There exists a subPropertyOf cycle between "
                                             + nextExpand.getURI()
                                             + " and "
                                             + expanded.getURI() + "\n");
                            }
                        }
                    }

                    ArrayList temp = topoQueue;
                    topoQueue = expansionQueue;
                    expansionQueue = temp;
                    expansionQueue.clear();
                }
            }
        }
    }
*/
    
    /**
     * checking whether there exist improper hierarchy declaration about
     * xsd:datatype
     * 
     * @param type
     *            the instances of the XSDDatatype
     * @param report
     *            the return result after checking
     */
/*    
    protected void CheckXSDDatatypeHierarchy(RDFSClass type, Report report) {
        String typeName = type.getLocalName();
        for (Iterator iter = type.getSubClass(false).iterator(); iter.hasNext();) {
            RDFSClass subclass = (RDFSClass) iter.next();
            if (subclass.getNamespace().getURI() == XSD_NAMESPACE) {
                String subclassType = subclass.getLocalName();
                if (!isDatatypeSubClassOf(subclassType, typeName)) {
                    report.errorMsg.add("Error: Invalid assertion - "
                                        + subclass.getURI() + " is subClassOf "
                                        + type.getURI());
                }
            }
        }
    }
*/
    
    /**
     * checking whether there exist multiple domains or ranges for a property
     * 
     * @param property
     *            the property needs to be checked
     * @param report
     *            the return result after checking
     */
/*    
    protected void checkMultiDomainRange(RDFProperty property, Report report) {
        int size = property.getRDFSDomain().size();
        if (size > 1) {
            report.warningMsg.add("Warning:Property "
                                  + property.getURI() + " have " + size
                                  + " domains.");
        }

        size = property.getRDFSRange().size();
        if (size > 1) {
            report.warningMsg.add("Warning:Property "
                                  + property.getURI() + " have " + size
                                  + " ranges.");
        }
    }
*/
    
    /**
     * Get a tag for access graph
     */
    protected long applyAccessTag() {
        return lastTag++;
    }

    protected long lastTag = 1000000;

    /**
     * Checkout whether two xsd:datatype is subsume according to XML
     * specfication the idea is to use a ties structure to capture the
     * subsumption relationship of all xsd:datatype. To achieve this, we assign
     * predefined key for every datatype, if a dataype T1 subsume another type
     * T2 in XML Specification, the key of T1 will be a prefix of the key of T2.
     * 
     * @param type1
     *            the local name of xsd:datatype
     * @param type2
     *            the local name of xsd:datatype
     * @return true if type1 is subClassOf type2 in XML specification
     */
    protected boolean isDatatypeSubClassOf(String type1, String type2) {
        String key1 = (String) DATATYPE_HIER.get(type1);
        String key2 = (String) DATATYPE_HIER.get(type2);
        return key1.startsWith(key2);
    }

    /**
     * the hash map datatypeHier contain all xsd:datatype and their predefined
     * keys
     */
    protected static Map DATATYPE_HIER = buildDataHierarchy();

    /**
     * building up the hash map all xsd:datatype
     * 
     * @return a hashmap of all xsd:datatype with their predefined keys
     */
    protected static Map buildDataHierarchy() {
        Map datatypeHier = new HashMap();
        datatypeHier.put("string", "a");
        datatypeHier.put("normalizedString", "aa");
        datatypeHier.put("token", "aaa");
        datatypeHier.put("language", "aaaa");
        datatypeHier.put("Name", "aaab");
        datatypeHier.put("NMTOKEN", "aaac");
        datatypeHier.put("NCName", "aaaba");
        datatypeHier.put("NMTOKENS", "aaaca");
        datatypeHier.put("ID", "aaabaa");
        datatypeHier.put("IDREF", "aaabab");
        datatypeHier.put("ENTITY", "aaabac");
        datatypeHier.put("IDREFS", "aaababa");
        datatypeHier.put("ENTITIES", "aaabaca");
        datatypeHier.put("boolean", "b");
        datatypeHier.put("base64Binary", "c");
        datatypeHier.put("hexBinary", "d");
        datatypeHier.put("float", "e");
        datatypeHier.put("decimal", "f");
        datatypeHier.put("integer", "fa");
        datatypeHier.put("nonPositiveInteger", "faa");
        datatypeHier.put("long", "fab");
        datatypeHier.put("nonNegativeInteger", "fac");
        datatypeHier.put("negativeInteger", "faaa");
        datatypeHier.put("int", "faba");
        datatypeHier.put("unsignedLong", "faca");
        datatypeHier.put("positiveInteger", "facb");
        datatypeHier.put("short", "fabaa");
        datatypeHier.put("unsignedInt", "facaa");
        datatypeHier.put("byte", "fabaaa");
        datatypeHier.put("unsignedShort", "facaaa");
        datatypeHier.put("unsignedByte", "facaaaa");
        datatypeHier.put("double", "g");
        datatypeHier.put("anyURI", "h");
        datatypeHier.put("QName", "i");
        datatypeHier.put("NOTATION", "j");
        datatypeHier.put("duration", "k");
        datatypeHier.put("dateTime", "l");
        datatypeHier.put("time", "m");
        datatypeHier.put("date", "n");
        datatypeHier.put("gYearMonth", "o");
        datatypeHier.put("gYear", "p");
        datatypeHier.put("gMonthDay", "q");
        datatypeHier.put("gDay", "r");
        datatypeHier.put("gMonth", "s");
        return datatypeHier;
    }

    class AccessInfoMap {
        private Map map = new HashMap();

        public AccessInfo getAccessInfo(Object key) {
        	AccessInfo info = (AccessInfo) map.get(key);
        	
            if (info == null) {
                info = new AccessInfo();
                map.put(key, info);
            }
            
            return info;
        }

        public long getAccessTag(Object key) {
            return getAccessInfo(key).visitTag;
        }

        public void setAccessTag(Object key, long tag) {
        	getAccessInfo(key).visitTag = tag;
        }

    }

    class AccessInfo {
        // visiting tag for navigating the RDF Graph
        long visitTag = 0;
        
        // if the key is a property, the cache will store triples
        // if the key is a rdfclass, the cache will store instances whose type is the key
        HashSet cache = new HashSet();
    }

    protected class Report {
        // List of Fatal Error Messages
        public ArrayList errorMsg = new ArrayList();

        // List of Fatal Warning Messages
        public ArrayList warningMsg = new ArrayList();
    }

    /**
     * Get all entailed instances of a RDFSClass. The algorithm is to
     * <li>get out all instances of the given RDFSClass c's descendants as set
     * I1</li>
     * <li>find out all c's domain-related properties and get out all subjects
     * of inferred triples of them as set I2</li>
     * find out all c's range-related properties and get out all objects of
     * inferred triples of them as set I3</li>
     * The final result is I1+I2+I3 <br/><br/>
     * 
     * @param c
     *            the RDFSClass to be queried
     * @return all entailed individuals that are rdf:type of the given class c
     */
    public List getInstances(RDFSClass c) {
        ArrayList result = new ArrayList();
        long tag = applyAccessTag();

        List children = getDescendantClasses(c);
        
        long domainRelatedTag = applyAccessTag();
        for (Iterator it = children.iterator(); it.hasNext();) {
            RDFSClass subClass = (RDFSClass) it.next();
            
            // get all subclasses' instance
            for (Iterator itInst = accessMap.getAccessInfo(subClass).cache.iterator(); itInst.hasNext();) {
                RDFSResource instance = (RDFSResource) itInst.next();
                AccessInfo info = accessMap.getAccessInfo(instance);
                if (info.visitTag != tag) {
                    info.visitTag = tag;
                    result.add(instance);
                }
            }

            // get subjects of properties whose domain = subClass
            for (Iterator itP = subClass.getPropertyForDomain().iterator(); itP.hasNext();) {
                RDFProperty property = (RDFProperty) itP.next();

                for (Iterator itSubP = getDescendantPropertiesByTag(property,domainRelatedTag).iterator(); itSubP.hasNext();) {
                    RDFProperty subProperty = (RDFProperty) itSubP.next();
                    for (Iterator itS = accessMap.getAccessInfo(subProperty).cache.iterator();itS.hasNext();) {
                    	Triple t = (Triple)itS.next();
                        RDFSResource instance = gModel.getRDFSResource( t.getSubjectURI() );
                        AccessInfo info = accessMap.getAccessInfo(instance);
                        if (info.visitTag != tag) {
                            info.visitTag = tag;
                            result.add(instance);
                        }
                    }
                }
            }
        }

        long rangeRelatedTag = applyAccessTag();
        for (Iterator i = children.iterator(); i.hasNext();) {
            RDFSClass subClass = (RDFSClass) i.next();

            // get objects of properties whose range = subClass
            for (Iterator itP = subClass.getPropertyForRange().iterator(); itP.hasNext();) {
                RDFProperty property = (RDFProperty) itP.next();

                for (Iterator itSubP = getDescendantPropertiesByTag(property,rangeRelatedTag).iterator(); itSubP.hasNext();) {
                    RDFProperty subProperty = (RDFProperty) itSubP.next();
                    for (Iterator itS = accessMap.getAccessInfo(subProperty).cache.iterator();itS.hasNext();) {
                    	Triple t = (Triple)itS.next();
                        RDFSResource instance = gModel.getRDFSResource(t.getObjectURI() );

                        AccessInfo info = accessMap.getAccessInfo(instance);
                        if (info.visitTag != tag) {
                            info.visitTag = tag;
                            result.add(instance);
                        }
                    }
                }
            }
        }
        return result;
    }

    /**
     * Get all entailed triples whose predicate is the given property by
     * navigating the DAG <br/><br/>What should be noticed is that: According
     * to rdfs7, if &lt;aaa subPropertyOf bbb&gt; & &lt;uuu aaa yyy&gt;, a
     * triple <uuu bbb yyy> will be return. Here we just return <uuu aaa yyy> to
     * capture the rdfs semantic. Users should explain the <uuu aaa yyy> as <uuu
     * bbb yyy>
     * 
     * @param property
     *            the RDFProperty to be queried
     * @return all entailed triples
     *  
     */
    public List getTriples(RDFProperty property) {
        ArrayList result = new ArrayList();
       
        // get all subproperies' statement
        for (Iterator it = getDescendantProperties(property).iterator(); it.hasNext();) {
        	RDFProperty p = (RDFProperty)it.next();
        	
        	for(Iterator iter=accessMap.getAccessInfo(p).cache.iterator();iter.hasNext();){
        		Triple s = (Triple)iter.next();
        		Triple t = new Triple();
        		// copy s to t
        		t.setObjectLiteral(s.getObjectLiteral());
        		t.setObjectNodeID(s.getObjectNodeID());
        		t.setObjectURI(s.getObjectURI());
        		
        		t.setPredicate(property.getURI());

        		t.setSubjectURI(s.getSubjectURI());
        		t.setSubjectNodeID(s.getSubjectNodeID());
        		
        		result.add(t);
        	}

        }

        return result;
    }

    /*
     * Checking Wether a Statement's object that is xml literal is incompatible
     * with the declaring ranges of the a property. record the report
     * 
     * @param property the prediccate of the statements @param report the return
     * result after checking
     */
/*    
    protected void checkRangeValues(RDFProperty property, Report report) {
        ArrayList list = new ArrayList();
        for (Iterator iter = property.getRDFSRange().iterator(); iter.hasNext();) {
            RDFSClass range = (RDFSClass) iter.next();
            Namespace ns;
            if ((ns = range.getNamespace()) != null
                && ns.getURI() == XSD_NAMESPACE) {
                // the range is a xsd datatype
                list.add(range);
            }
        }

        if (!list.isEmpty()) {
            // one of more ranges of the property are xsd:datatype,
            // checking every statement whether there exists violation
            for (Iterator iter = property.getPredicateStatement().iterator(); iter
                    .hasNext();) {
                RDFStatement statement = (RDFStatement) iter.next();
                RDFSResource resource = statement.getRDFObject();
                if (resource instanceof TypedLiteral
                    && ((TypedLiteral) resource).getDatatype().getNamespace()
                            .getURI() == XSD_NAMESPACE) {
                    boolean compatible = false;
                    for (Iterator iterType = list.iterator(); iterType
                            .hasNext();) {
                        // checking whether the object is compatible with any
                        // range
                        String objectType = ((TypedLiteral) resource)
                                .getDatatype().getLocalName();
                        String rangeType = ((RDFSClass) iterType.next())
                                .getLocalName();
                        if (isDatatypeSubClassOf(objectType, rangeType)) {
                            compatible = true;
                            break;
                        }
                    }

                    if (!compatible) {
                        report.errorMsg
                                .add("Error: <"
                                     + statement.getRDFSubject().getURI()
                                     + "> <"
                                     + property.getURI()
                                     + "> <"
                                     + ((TypedLiteral) resource)
                                             .getLexicalForm()
                                     + "^^"
                                     + ((TypedLiteral) resource).getDatatype()
                                             .getURI()
                                     + "> is incompatible with any ranges of "
                                     + property.getURI());
                    }
                }
            }
        }

    }
*/
    
    /**
     * Get all entailed sub properties of a RDFProperty by navigating the DAG
     * using given tag
     * 
     * @param property
     *            the RDFProperty to be queried
     * @return all entailed sub properties of the given property
     */
    protected List getDescendantPropertiesByTag(RDFProperty property, long tag) {
        ArrayList result = new ArrayList();

        if (property != null) {
            ArrayList queue = new ArrayList();
            queue.add(property);

            while (!queue.isEmpty()) {
                RDFProperty node = (RDFProperty) queue.remove(0);
                AccessInfo info = accessMap.getAccessInfo(node);
                if (info.visitTag != tag) {
                    info.visitTag = tag;
                    result.add(node);
                    queue.addAll(node.getSubProperty());
                }
            }
        }

        return result;
    }

    private RDFGraph gModel = null;
    
    // private static String XSD_NAMESPACE = "http://www.w3.org/2001/XMLSchema#";
    private static String RDF_TYPE = "http://www.w3.org/1999/02/22-rdf-syntax-ns#type";
    private static String RDFS_CLASS = "http://www.w3.org/2000/01/rdf-schema#Class";
    
    private static String RDF_PROPERTY = "http://www.w3.org/1999/02/22-rdf-syntax-ns#Property";


    protected AccessInfoMap accessMap = new AccessInfoMap();

    protected List classes = new ArrayList();

    protected List properties = new ArrayList();
    
	public boolean isValid(RDFGraph graph) {
		return false;
	}
    
}
