/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident	"@(#)mlib_c_ImageColorYCC2RGB.c	9.2	07/10/09 SMI"

/*
 * FUNCTION
 *      mlib_ImageColorYCC2RGB - color conversion from YCC to RGB space
 *
 * SYNOPSIS
 *      mlib_status mlib_ImageColorYCC2RGB(mlib_image       *dst,
 *                                         const mlib_image *src);
 *
 * ARGUMENT
 *      dst     pointer to output image
 *      src     pointer to input image
 *
 * RESTRICTION
 *      src and dst must be the same type, the same size and the same
 *      number of channels.
 *      They must be 3-channel images.
 *      They can be in MLIB_BYTE, MLIB_SHORT, MLIB_USHORT or MLIB_INT data type.
 *
 * DESCRIPTION
 *
 *        |R|   |fmat[0] fmat[1] fmat[2]|   |Y |   |offset[0]|
 *        |G| = |fmat[3] fmat[4] fmat[5]| * |Cb| + |offset[1]|
 *        |B|   |fmat[6] fmat[7] fmat[8]|   |Cr|   |offset[2]|
 *
 *   where
 *
 *      fmat[] = { 298.082/256,  0,            408.583/256,
 *                 298.082/256, -100.291/256, -208.120/256,
 *                 298.082/256,  516.411/256,  0};
 *    offset[] = {-222.953, 135.575, -276.836};
 */

#include <mlib_image.h>
#include <mlib_ImageCheck.h>
#include <mlib_c_ImageColorTables.h>

/* *********************************************************** */

#if ! defined(__MEDIALIB_OLD_NAMES)
#if defined(__SUNPRO_C)

#pragma weak mlib_ImageColorYCC2RGB = __mlib_ImageColorYCC2RGB

#elif defined(__GNUC__)	/* defined(__SUNPRO_C) */
__typeof__(__mlib_ImageColorYCC2RGB) mlib_ImageColorYCC2RGB
    __attribute__((weak, alias("__mlib_ImageColorYCC2RGB")));

#else /* defined(__SUNPRO_C) */

#error  "unknown platform"

#endif /* defined(__SUNPRO_C) */
#endif /* ! defined(__MEDIALIB_OLD_NAMES) */

/* *********************************************************** */

static const mlib_s32 t_Ycc2rgb[256] = {
	33792, 1083393, 2132994, 3183619, 5281797, 6331398, 7380999, 8430600,
	9480201, 10530826, 12629004, 13678605, 14728206, 15777807, 16827408,
	    17878033,
	19976211, 21025812, 22075413, 23125014, 24174615, 25224216, 27323418,
	    28373019,
	29422620, 30472221, 31521822, 32571423, 34670625, 35720226, 36769827,
	    37819428,
	38869029, 39918630, 42017832, 43067433, 44117034, 45166635, 46216236,
	    47265837,
	49365039, 50414640, 51464241, 52513842, 53563443, 54613044, 56712246,
	    57761847,
	58811448, 59861049, 60910650, 61960251, 64059453, 65109054, 66158655,
	    67208256,
	68257857, 69307458, 71406660, 72456261, 73505862, 74555463, 75605064,
	    76654665,
	78753867, 79803468, 80853069, 81902670, 82952271, 84001872, 86101074,
	    87150675,
	88200276, 89249877, 90299478, 91349079, 92399704, 94497882, 95547483,
	    96597084,
	97646685, 98696286, 99746911, 101845089, 102894690, 103944291,
	    104993892, 106043493,
	107094118, 109192296, 110241897, 111291498, 112341099, 113390700,
	    114440301, 116539503,
	117589104, 118638705, 119688306, 120737907, 121787508, 123886710,
	    124936311, 125985912,
	127035513, 128085114, 129134715, 131233917, 132283518, 133333119,
	    134382720, 135432321,
	136481922, 138581124, 139630725, 140680326, 141729927, 142779528,
	    143829129, 145928331,
	146977932, 148027533, 149077134, 150126735, 151176336, 153275538,
	    154325139, 155374740,
	156424341, 157473942, 158523543, 160622745, 161672346, 162721947,
	    163771548, 164821149,
	165870750, 167969952, 169019553, 170069154, 171118755, 172168356,
	    173217957, 175317159,
	176366760, 177416361, 178465962, 179515563, 180565164, 181615789,
	    183713967, 184763568,
	185813169, 186862770, 187912371, 188962996, 191061174, 192110775,
	    193160376, 194209977,
	195259578, 196310203, 198408381, 199457982, 200507583, 201557184,
	    202606785, 203656386,
	205755588, 206805189, 207854790, 208904391, 209953992, 211003593,
	    213102795, 214152396,
	215201997, 216251598, 217301199, 218350800, 220450002, 221499603,
	    222549204, 223598805,
	224648406, 225698007, 227797209, 228846810, 229896411, 230946012,
	    231995613, 233045214,
	235144416, 236194017, 237243618, 238293219, 239342820, 240392421,
	    242491623, 243541224,
	244590825, 245640426, 246690027, 247739628, 249838830, 250888431,
	    251938032, 252987633,
	254037234, 255086835, 257186037, 258235638, 259285239, 260334840,
	    261384441, 262434042,
	264533244, 265582845, 266632446, 267682047, 268731648, 269781249,
	    270831874, 272930052,
	273979653, 275029254, 276078855, 277128456, 278179081, 280277259,
	    281326860, 282376461,
	283426062, 284475663, 285526288, 287624466, 288674067, 289723668,
	    290773269, 291822870,
	292872471, 294971673, 296021274, 297070875, 298120476, 299170077,
	    300219678, 302318880,
	303368481, 304418082, 305467683, 306517284, 307566885, 309666087,
	    310715688, 311765289
};

/* *********************************************************** */

static const mlib_s32 t_yCc2rgb[256] = {
	-290455416, -288358265, -286261113, -284163962, -282066810, -279969658,
	    -277872507, -275775355,
	-273678204, -271581052, -269483900, -267386749, -265289597, -263192446,
	    -261095294, -258998142,
	-256900991, -254803839, -252706687, -250609536, -247463808, -245366657,
	    -243269505, -241172353,
	-239075202, -236978050, -234880899, -232783747, -230686595, -228589444,
	    -226492292, -224395141,
	-222297989, -220200837, -218103686, -216006534, -213909383, -211812231,
	    -209715079, -207617928,
	-205520776, -203423624, -201326473, -199229321, -197132170, -195035018,
	    -192937866, -190840715,
	-188743563, -186646412, -184549260, -182452108, -180354957, -178257805,
	    -176160654, -174063502,
	-171966350, -169869199, -167772047, -165674896, -163577744, -161480592,
	    -159383441, -157286289,
	-155189137, -153091986, -150994834, -148897683, -146800531, -144703379,
	    -142606228, -140509076,
	-138411925, -136314773, -134217621, -132120470, -130023318, -127926167,
	    -124780439, -122683287,
	-120586136, -118488984, -116391833, -114294681, -112197529, -110100378,
	    -108003226, -105906075,
	-103808923, -101711771, -99614620, -97517468, -95420316, -93323165,
	    -91226013, -89128862,
	-87031710, -84934558, -82837407, -80740255, -78643104, -76545952,
	    -74448800, -72351649,
	-70254497, -68157346, -66060194, -63963042, -61865891, -59768739,
	    -57671588, -55574436,
	-53477284, -51380133, -49282981, -47185829, -45088678, -42991526,
	    -40894375, -38797223,
	-36700071, -34602920, -32505768, -30408617, -28311465, -26214313,
	    -24117162, -22020010,
	-19922859, -17825707, -15728555, -13631404, -11534252, -9437101,
	    -7339949, -5242797,
	-2097070, 82, 2097234, 4194385, 6291537, 8388688, 10485840, 12582992,
	14680143, 16777295, 18874446, 20971598, 23068750, 25165901, 27263053,
	    29360204,
	31457356, 33554508, 35651659, 37748811, 39845962, 41943114, 44040266,
	    46137417,
	48234569, 50331721, 52428872, 54526024, 56623175, 58720327, 60817479,
	    62914630,
	65011782, 67108933, 69206085, 71303237, 73400388, 75497540, 77594691,
	    79691843,
	81788995, 83886146, 85983298, 88080449, 90177601, 92274753, 94371904,
	    96469056,
	98566207, 100663359, 102760511, 104857662, 106954814, 109051966,
	    111149117, 113246269,
	115343420, 117440572, 120586300, 122683451, 124780603, 126877754,
	    128974906, 131072058,
	133169209, 135266361, 137363512, 139460664, 141557816, 143654967,
	    145752119, 147849270,
	149946422, 152043574, 154140725, 156237877, 158335029, 160432180,
	    162529332, 164626483,
	166723635, 168820787, 170917938, 173015090, 175112241, 177209393,
	    179306545, 181403696,
	183500848, 185597999, 187695151, 189792303, 191889454, 193986606,
	    196083757, 198180909,
	200278061, 202375212, 204472364, 206569516, 208666667, 210763819,
	    212860970, 214958122,
	217055274, 219152425, 221249577, 223346728, 225443880, 227541032,
	    229638183, 231735335,
	233832486, 235929638, 238026790, 240123941, 243269669, 245366820,
	    247463972, 249561124
};

/* *********************************************************** */

static const mlib_s32 t_ycC2rgb[256] = {
	256, 2303, 3326, 5374, 6397, 8444, 10491, 11514,
	13561, 14585, 16632, 18679, 19702, 21749, 22773, 24820,
	26867, 27890, 29937, 30961, 33008, 35055, 36078, 38125,
	39148, 41196, 42219, 44266, 46313, 47336, 49384, 50407,
	52454, 54501, 55524, 57572, 58595, 60642, 62689, 63712,
	65759, 66783, 68830, 70877, 71900, 73947, 74971, 77018,
	79065, 80088, 82135, 83159, 85206, 87253, 88276, 90323,
	91346, 93394, 95441, 96464, 98511, 99534, 101582, 103629,
	104652, 106699, 107722, 109770, 111817, 112840, 114887, 115910,
	117957, 120005, 121028, 123075, 124098, 126145, 127169, 129216,
	131263, 132286, 134333, 135357, 137404, 139451, 140474, 142521,
	143544, 145592, 147639, 148662, 150709, 151732, 153780, 155827,
	156850, 158897, 159920, 161968, 164015, 165038, 167085, 168108,
	170155, 172203, 173226, 175273, 176296, 178343, 180391, 181414,
	183461, 184484, 186531, 188579, 189602, 191649, 192672, 194719,
	196766, 197790, 199837, 200860, 202907, 204954, 205978, 208025,
	209048, 211095, 212118, 214166, 216213, 217236, 219283, 220306,
	222353, 224401, 225424, 227471, 228494, 230541, 232589, 233612,
	235659, 236682, 238729, 240776, 241800, 243847, 244870, 246917,
	248964, 249988, 252035, 253058, 255105, 257152, 258176, 260223,
	261246, 263293, 265340, 266363, 268411, 269434, 271481, 273528,
	274551, 276599, 277622, 279669, 281716, 282739, 284787, 285810,
	287857, 288880, 290927, 292974, 293998, 296045, 297068, 299115,
	301162, 302186, 304233, 305256, 307303, 309350, 310374, 312421,
	313444, 315491, 317538, 318561, 320609, 321632, 323679, 325726,
	326749, 328797, 329820, 331867, 333914, 334937, 336985, 338008,
	340055, 342102, 343125, 345172, 346196, 348243, 350290, 351313,
	353360, 354384, 356431, 358478, 359501, 361548, 362572, 364619,
	366666, 367689, 369736, 370759, 372807, 373830, 375877, 377924,
	378947, 380995, 382018, 384065, 386112, 387135, 389183, 390206,
	392253, 394300, 395323, 397370, 398394, 400441, 402488, 403511,
	405558, 406582, 408629, 410676, 411699, 413746, 414770, 416817
};

/* *********************************************************** */

#define	TCLAMP_U8(dst, s)	dst = mlib_pClipOff256[s]

/* *********************************************************** */

#define	SCLAMP_U8(dst, s)                                       \
	{                                                       \
	    mlib_s32 v = s, mask = (v - 0xff00000) >> 31;       \
	                                                        \
	    dst = (((mlib_u32)v >> 20) | ~mask) & ~(v >> 31);   \
	}

/* *********************************************************** */

#ifdef _LITTLE_ENDIAN

#define	SPLIT_S32_U8(x, a0, a1, a2, a3)                         \
	a3 = (x >> 24);                                         \
	a2 = (x & 0xff0000) >> 16;                              \
	a1 = (x & 0x00ff00) >> 8;                               \
	a0 = (x & 0x0000ff)

#else /* _LITTLE_ENDIAN */

#define	SPLIT_S32_U8(x, a0, a1, a2, a3)                         \
	a0 = (x >> 24);                                         \
	a1 = (x & 0xff0000) >> 16;                              \
	a2 = (x & 0x00ff00) >> 8;                               \
	a3 = (x & 0x0000ff)

#endif /* _LITTLE_ENDIAN */

/* *********************************************************** */

#define	MLIB_CONVERT_U8_1(pd, ps)                                 \
	{                                                         \
	    mlib_u32 x0, y0, z0;                                  \
	    mlib_u32 s0;                                          \
	                                                          \
	    x0 = ps[0];                                           \
	    y0 = ps[1];                                           \
	    z0 = ps[2];                                           \
	    s0 = t_Ycc2rgb[x0] + t_yCc2rgb[y0] + t_ycC2rgb[z0];   \
	    SCLAMP_U8(pd[2], s0);                                 \
	    TCLAMP_U8(pd[0], (s0 >> 10) & 0x3ff);                 \
	    TCLAMP_U8(pd[1], s0 & 0x3ff);                         \
	}

/* *********************************************************** */

mlib_status
__mlib_ImageColorYCC2RGB(
    mlib_image *dst,
    const mlib_image *src)
{
/* Rec601 Y'CbCr to Computer R'G'B' */
	mlib_d64 fmat[9] = { 298.082 / 256, 0, 408.583 / 256,
		298.082 / 256, -100.291 / 256, -208.120 / 256,
		298.082 / 256, 516.411 / 256, 0
	};
	mlib_d64 offset[3] = { -222.953, 135.575, -276.836 };
	mlib_s32 slb, dlb, xsize, ysize;
	mlib_type dtype;
	mlib_u8 *psrc, *pdst;
	mlib_s32 j;

	MLIB_IMAGE_CHECK(dst);
	MLIB_IMAGE_CHECK(src);
	MLIB_IMAGE_FULL_EQUAL(dst, src);

	dtype = mlib_ImageGetType(dst);
	xsize = mlib_ImageGetWidth(dst);
	ysize = mlib_ImageGetHeight(dst);
	dlb = mlib_ImageGetStride(dst);
	pdst = (void *)mlib_ImageGetData(dst);

	slb = mlib_ImageGetStride(src);
	psrc = mlib_ImageGetData(src);

	if (dtype == MLIB_BYTE) {

		for (j = 0; j < ysize; j++) {
			mlib_u8 *ps = psrc,
			    *pd = pdst, *pend = pdst + 3 * xsize;

#ifdef __SUNPRO_C
#pragma pipeloop(0)
#endif /* __SUNPRO_C */
			for (; pd < pend; pd += 3) {
				MLIB_CONVERT_U8_1(pd, ps);
				ps += 3;
			}

			psrc += slb;
			pdst += dlb;
		}

		return (MLIB_SUCCESS);

	} else {

		return (__mlib_ImageColorConvert2(dst, src, fmat, offset));
	}
}

/* *********************************************************** */
