/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2005 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident	"@(#)mlib_ImageBlendMulti.c	9.2	07/10/09 SMI"

/*
 * FUNCTIONS
 *      mlib_ImageBlendMulti - multi-source alpha blending
 *      mlib_ImageBlendMulti_Fp - multi-source alpha blending
 *
 * SYNOPSIS
 *      mlib_status mlib_ImageBlendMulti(mlib_image       *dst,
 *                                       const mlib_image **srcs,
 *                                       const mlib_image **alphas,
 *                                       const mlib_s32   *c,
 *                                       mlib_s32         n)
 *      mlib_status mlib_ImageBlendMulti_Fp(mlib_image       *dst,
 *                                          const mlib_image **srcs,
 *                                          const mlib_image **alphas,
 *                                          const mlib_d64   *c,
 *                                          mlib_s32         n)
 *
 * ARGUMENTS
 *      dst     Pointer to the destination image.
 *      srcs    Pointer to an array of source images.
 *      alphas  Pointer to an array of alpha images.
 *      c       The background color.
 *      n       The number of source images to be blended.
 *
 * DESCRIPTION
 *      Blend multiple images into a single image.
 *
 *      S(k) = s(i, j,k)    i = 0, 1, ..., w - 1; j = 0, 1, ..., h - 1;
 *                           k = 0, 1, ..., n - 1
 *      A(k) = a(i, j,k)    i = 0, 1, ..., w - 1; j = 0, 1, ..., h - 1;
 *                           k = 0, 1, ..., n - 1
 *      D    = d(i, j)      i = 0, 1, ..., w - 1; j = 0, 1, ..., h - 1;
 *
 *                         n-1
 *             +- c     if SUM{a(i, j,k)} = 0,
 *             |           k = 0
 *           = |
 *             |  n-1                      n-1
 *             +- SUM{a(i, j,k)*s(i, j,k)} / SUM{a(i, j,k)}  otherwise
 *                k = 0                      k = 0
 *
 *      where S(k) and A(k) are the kth source and alpha image respectively.
 *
 * RESTRICTIONS
 *      Each of the images can have 1 to 4 channels.
 *      All images involved should have the same data type and same size
 *      and the source and destination images should have the same number of
 *      channels.  The alpha images should have either 1 channel or the same
 *      number of channels as the sources and destination.  A single-channel
 *      alpha image would be applied to all bands of the corresponding
 *      source image.  Single and multi-channel alpha images should be able
 *      to be mixed in the same invocation.
 *
 *      For mlib_ImageBlendMulti, the images can be of type MLIB_BYTE,
 *      MLIB_SHORT, MLIB_USHORT, or MLIB_INT.
 *
 *      For mlib_ImageBlendMulti_Fp, the images can be of type MLIB_FLOAT
 *      or MLIB_DOUBLE.
 */

#include <mlib_image.h>

#ifdef MLIB_VIS_EXPERIMENT
#include <vis_proto.h>
#endif /* MLIB_VIS_EXPERIMENT */

#include <mlib_SysMath.h>
#include <mlib_ImageCheck.h>
#include <mlib_ImageDivTables.h>
#include <mlib_Utils.h>

/* *********************************************************** */

#if ! defined(__MEDIALIB_OLD_NAMES)
#if defined(__SUNPRO_C)

#pragma weak mlib_ImageBlendMulti = __mlib_ImageBlendMulti
#pragma weak mlib_ImageBlendMulti_Fp = __mlib_ImageBlendMulti_Fp

#elif defined(__GNUC__)	/* defined(__SUNPRO_C) */
__typeof__(__mlib_ImageBlendMulti) mlib_ImageBlendMulti
    __attribute__((weak, alias("__mlib_ImageBlendMulti")));
__typeof__(__mlib_ImageBlendMulti_Fp) mlib_ImageBlendMulti_Fp
    __attribute__((weak, alias("__mlib_ImageBlendMulti_Fp")));

#else /* defined(__SUNPRO_C) */

#error  "unknown platform"

#endif /* defined(__SUNPRO_C) */
#endif /* ! defined(__MEDIALIB_OLD_NAMES) */

/* *********************************************************** */

#define	FTYPE	mlib_d64

#define	ATYPE	mlib_s32

/* *********************************************************** */

typedef struct
{
	void *sl;
	void *al;
	mlib_s32 slb;
	mlib_s32 alb;
	mlib_s32 achan;
} src_t;

/* *********************************************************** */

#define	FUNC_ARG                                                    \
	void *dl, void *sl, void *al, void *_buffs, void *_buffa,   \
	    const void *_bcolor, mlib_s32 shift, mlib_s32 wid

typedef void (*func_type) (FUNC_ARG);

/* *********************************************************** */

static const FTYPE mlib_div_u10[] = {
	0, 1.00000000000000000000, 0.50000000000000000000,
	    0.33333333333333331483,
	0.25000000000000000000, 0.20000000000000001110, 0.16666666666666665741,
	    0.14285714285714284921,
	0.12500000000000000000, 0.11111111111111110494, 0.10000000000000000555,
	    0.09090909090909091161,
	0.08333333333333332871, 0.07692307692307692735, 0.07142857142857142461,
	    0.06666666666666666574,
	0.06250000000000000000, 0.05882352941176470507, 0.05555555555555555247,
	    0.05263157894736841813,
	0.05000000000000000278, 0.04761904761904761640, 0.04545454545454545581,
	    0.04347826086956521618,
	0.04166666666666666435, 0.04000000000000000083, 0.03846153846153846367,
	    0.03703703703703703498,
	0.03571428571428571230, 0.03448275862068965469, 0.03333333333333333287,
	    0.03225806451612903136,
	0.03125000000000000000, 0.03030303030303030387, 0.02941176470588235253,
	    0.02857142857142857054,
	0.02777777777777777624, 0.02702702702702702853, 0.02631578947368420907,
	    0.02564102564102564014,
	0.02500000000000000139, 0.02439024390243902524, 0.02380952380952380820,
	    0.02325581395348837177,
	0.02272727272727272790, 0.02222222222222222307, 0.02173913043478260809,
	    0.02127659574468085055,
	0.02083333333333333218, 0.02040816326530612082, 0.02000000000000000042,
	    0.01960784313725490169,
	0.01923076923076923184, 0.01886792452830188607, 0.01851851851851851749,
	    0.01818181818181818094,
	0.01785714285714285615, 0.01754385964912280604, 0.01724137931034482735,
	    0.01694915254237288130,
	0.01666666666666666644, 0.01639344262295082053, 0.01612903225806451568,
	    0.01587301587301587213,
	0.01562500000000000000, 0.01538461538461538547, 0.01515151515151515194,
	    0.01492537313432835792,
	0.01470588235294117627, 0.01449275362318840597, 0.01428571428571428527,
	    0.01408450704225352144,
	0.01388888888888888812, 0.01369863013698630061, 0.01351351351351351426,
	    0.01333333333333333419,
	0.01315789473684210453, 0.01298701298701298787, 0.01282051282051282007,
	    0.01265822784810126563,
	0.01250000000000000069, 0.01234567901234567833, 0.01219512195121951262,
	    0.01204819277108433798,
	0.01190476190476190410, 0.01176470588235294101, 0.01162790697674418589,
	    0.01149425287356321823,
	0.01136363636363636395, 0.01123595505617977497, 0.01111111111111111154,
	    0.01098901098901098987,
	0.01086956521739130405, 0.01075268817204301161, 0.01063829787234042527,
	    0.01052631578947368397,
	0.01041666666666666609, 0.01030927835051546372, 0.01020408163265306041,
	    0.01010101010101010187,
	0.01000000000000000021, 0.00990099009900990111, 0.00980392156862745084,
	    0.00970873786407766906,
	0.00961538461538461592, 0.00952380952380952467, 0.00943396226415094304,
	    0.00934579439252336379,
	0.00925925925925925875, 0.00917431192660550510, 0.00909090909090909047,
	    0.00900900900900900893,
	0.00892857142857142808, 0.00884955752212389368, 0.00877192982456140302,
	    0.00869565217391304358,
	0.00862068965517241367, 0.00854700854700854787, 0.00847457627118644065,
	    0.00840336134453781476,
	0.00833333333333333322, 0.00826446280991735560, 0.00819672131147541026,
	    0.00813008130081300899,
	0.00806451612903225784, 0.00800000000000000017, 0.00793650793650793607,
	    0.00787401574803149595,
	0.00781250000000000000, 0.00775193798449612392, 0.00769230769230769273,
	    0.00763358778625954169,
	0.00757575757575757597, 0.00751879699248120259, 0.00746268656716417896,
	    0.00740740740740740769,
	0.00735294117647058813, 0.00729927007299270049, 0.00724637681159420299,
	    0.00719424460431654714,
	0.00714285714285714263, 0.00709219858156028352, 0.00704225352112676072,
	    0.00699300699300699300,
	0.00694444444444444406, 0.00689655172413793094, 0.00684931506849315030,
	    0.00680272108843537390,
	0.00675675675675675713, 0.00671140939597315422, 0.00666666666666666709,
	    0.00662251655629139072,
	0.00657894736842105227, 0.00653594771241830085, 0.00649350649350649393,
	    0.00645161290322580645,
	0.00641025641025641003, 0.00636942675159235701, 0.00632911392405063281,
	    0.00628930817610062927,
	0.00625000000000000035, 0.00621118012422360206, 0.00617283950617283916,
	    0.00613496932515337438,
	0.00609756097560975631, 0.00606060606060606060, 0.00602409638554216899,
	    0.00598802395209580875,
	0.00595238095238095205, 0.00591715976331360933, 0.00588235294117647051,
	    0.00584795321637426868,
	0.00581395348837209294, 0.00578034682080924827, 0.00574712643678160912,
	    0.00571428571428571428,
	0.00568181818181818198, 0.00564971751412429377, 0.00561797752808988748,
	    0.00558659217877094990,
	0.00555555555555555577, 0.00552486187845303844, 0.00549450549450549493,
	    0.00546448087431693989,
	0.00543478260869565202, 0.00540540540540540571, 0.00537634408602150581,
	    0.00534759358288770022,
	0.00531914893617021264, 0.00529100529100529071, 0.00526315789473684199,
	    0.00523560209424083801,
	0.00520833333333333304, 0.00518134715025906755, 0.00515463917525773186,
	    0.00512820512820512820,
	0.00510204081632653021, 0.00507614213197969504, 0.00505050505050505093,
	    0.00502512562814070359,
	0.00500000000000000010, 0.00497512437810945264, 0.00495049504950495056,
	    0.00492610837438423651,
	0.00490196078431372542, 0.00487804878048780487, 0.00485436893203883453,
	    0.00483091787439613504,
	0.00480769230769230796, 0.00478468899521531082, 0.00476190476190476233,
	    0.00473933649289099562,
	0.00471698113207547152, 0.00469483568075117381, 0.00467289719626168189,
	    0.00465116279069767435,
	0.00462962962962962937, 0.00460829493087557603, 0.00458715596330275255,
	    0.00456621004566210020,
	0.00454545454545454523, 0.00452488687782805470, 0.00450450450450450447,
	    0.00448430493273542594,
	0.00446428571428571404, 0.00444444444444444444, 0.00442477876106194684,
	    0.00440528634361233521,
	0.00438596491228070151, 0.00436681222707423558, 0.00434782608695652179,
	    0.00432900432900432900,
	0.00431034482758620684, 0.00429184549356223174, 0.00427350427350427393,
	    0.00425531914893617028,
	0.00423728813559322032, 0.00421940928270042159, 0.00420168067226890738,
	    0.00418410041841004148,
	0.00416666666666666661, 0.00414937759336099585, 0.00413223140495867780,
	    0.00411522633744856002,
	0.00409836065573770513, 0.00408163265306122486, 0.00406504065040650450,
	    0.00404858299595141705,
	0.00403225806451612892, 0.00401606425702811208, 0.00400000000000000008,
	    0.00398406374501992025,
	0.00396825396825396803, 0.00395256916996047404, 0.00393700787401574798,
	    0.00392156862745098034,
	0.00390625000000000000, 0.00389105058365758760, 0.00387596899224806196,
	    0.00386100386100386109,
	0.00384615384615384637, 0.00383141762452107260, 0.00381679389312977084,
	    0.00380228136882129275,
	0.00378787878787878798, 0.00377358490566037739, 0.00375939849624060130,
	    0.00374531835205992513,
	0.00373134328358208948, 0.00371747211895910763, 0.00370370370370370385,
	    0.00369003690036900358,
	0.00367647058823529407, 0.00366300366300366300, 0.00364963503649635024,
	    0.00363636363636363636,
	0.00362318840579710149, 0.00361010830324909760, 0.00359712230215827357,
	    0.00358422939068100358,
	0.00357142857142857132, 0.00355871886120996423, 0.00354609929078014176,
	    0.00353356890459363953,
	0.00352112676056338036, 0.00350877192982456147, 0.00349650349650349650,
	    0.00348432055749128920,
	0.00347222222222222203, 0.00346020761245674751, 0.00344827586206896547,
	    0.00343642611683848791,
	0.00342465753424657515, 0.00341296928327645055, 0.00340136054421768695,
	    0.00338983050847457617,
	0.00337837837837837857, 0.00336700336700336686, 0.00335570469798657711,
	    0.00334448160535117051,
	0.00333333333333333355, 0.00332225913621262466, 0.00331125827814569536,
	    0.00330033003300330037,
	0.00328947368421052613, 0.00327868852459016393, 0.00326797385620915043,
	    0.00325732899022801317,
	0.00324675324675324697, 0.00323624595469255679, 0.00322580645161290322,
	    0.00321543408360128614,
	0.00320512820512820502, 0.00319488817891373789, 0.00318471337579617850,
	    0.00317460317460317460,
	0.00316455696202531641, 0.00315457413249211347, 0.00314465408805031463,
	    0.00313479623824451398,
	0.00312500000000000017, 0.00311526479750778807, 0.00310559006211180103,
	    0.00309597523219814260,
	0.00308641975308641958, 0.00307692307692307692, 0.00306748466257668719,
	    0.00305810397553516822,
	0.00304878048780487815, 0.00303951367781155014, 0.00303030303030303030,
	    0.00302114803625377643,
	0.00301204819277108449, 0.00300300300300300298, 0.00299401197604790437,
	    0.00298507462686567167,
	0.00297619047619047603, 0.00296735905044510397, 0.00295857988165680466,
	    0.00294985250737463123,
	0.00294117647058823525, 0.00293255131964809384, 0.00292397660818713434,
	    0.00291545189504373173,
	0.00290697674418604647, 0.00289855072463768119, 0.00289017341040462413,
	    0.00288184438040345802,
	0.00287356321839080456, 0.00286532951289398272, 0.00285714285714285714,
	    0.00284900284900284914,
	0.00284090909090909099, 0.00283286118980169985, 0.00282485875706214688,
	    0.00281690140845070438,
	0.00280898876404494374, 0.00280112044817927173, 0.00279329608938547495,
	    0.00278551532033426185,
	0.00277777777777777788, 0.00277008310249307480, 0.00276243093922651922,
	    0.00275482093663911853,
	0.00274725274725274747, 0.00273972602739726030, 0.00273224043715846994,
	    0.00272479564032697538,
	0.00271739130434782601, 0.00271002710027100271, 0.00270270270270270285,
	    0.00269541778975741254,
	0.00268817204301075290, 0.00268096514745308316, 0.00267379679144385011,
	    0.00266666666666666658,
	0.00265957446808510632, 0.00265251989389920411, 0.00264550264550264536,
	    0.00263852242744063315,
	0.00263157894736842099, 0.00262467191601049865, 0.00261780104712041901,
	    0.00261096605744125330,
	0.00260416666666666652, 0.00259740259740259740, 0.00259067357512953378,
	    0.00258397932816537479,
	0.00257731958762886593, 0.00257069408740359879, 0.00256410256410256410,
	    0.00255754475703324829,
	0.00255102040816326510, 0.00254452926208651418, 0.00253807106598984752,
	    0.00253164556962025321,
	0.00252525252525252547, 0.00251889168765743066, 0.00251256281407035180,
	    0.00250626566416040086,
	0.00250000000000000005, 0.00249376558603491266, 0.00248756218905472632,
	    0.00248138957816377171,
	0.00247524752475247528, 0.00246913580246913575, 0.00246305418719211825,
	    0.00245700245700245694,
	0.00245098039215686271, 0.00244498777506112468, 0.00243902439024390244,
	    0.00243309002433090031,
	0.00242718446601941727, 0.00242130750605326888, 0.00241545893719806752,
	    0.00240963855421686768,
	0.00240384615384615398, 0.00239808153477218209, 0.00239234449760765541,
	    0.00238663484486873519,
	0.00238095238095238117, 0.00237529691211401436, 0.00236966824644549781,
	    0.00236406619385342784,
	0.00235849056603773576, 0.00235294117647058803, 0.00234741784037558691,
	    0.00234192037470725995,
	0.00233644859813084095, 0.00233100233100233100, 0.00232558139534883718,
	    0.00232018561484918784,
	0.00231481481481481469, 0.00230946882217090066, 0.00230414746543778802,
	    0.00229885057471264365,
	0.00229357798165137627, 0.00228832951945080092, 0.00228310502283105010,
	    0.00227790432801822313,
	0.00227272727272727262, 0.00226757369614512478, 0.00226244343891402735,
	    0.00225733634311512396,
	0.00225225225225225223, 0.00224719101123595525, 0.00224215246636771297,
	    0.00223713646532438474,
	0.00223214285714285702, 0.00222717149220489968, 0.00222222222222222222,
	    0.00221729490022172949,
	0.00221238938053097342, 0.00220750551876379691, 0.00220264317180616761,
	    0.00219780219780219780,
	0.00219298245614035076, 0.00218818380743982487, 0.00218340611353711779,
	    0.00217864923747276710,
	0.00217391304347826090, 0.00216919739696312371, 0.00216450216450216450,
	    0.00215982721382289430,
	0.00215517241379310342, 0.00215053763440860215, 0.00214592274678111587,
	    0.00214132762312633845,
	0.00213675213675213697, 0.00213219616204690827, 0.00212765957446808514,
	    0.00212314225053078552,
	0.00211864406779661016, 0.00211416490486257937, 0.00210970464135021079,
	    0.00210526315789473679,
	0.00210084033613445369, 0.00209643605870020976, 0.00209205020920502074,
	    0.00208768267223382025,
	0.00208333333333333330, 0.00207900207900207912, 0.00207468879668049793,
	    0.00207039337474120098,
	0.00206611570247933890, 0.00206185567010309283, 0.00205761316872428001,
	    0.00205338809034907605,
	0.00204918032786885257, 0.00204498977505112494, 0.00204081632653061243,
	    0.00203665987780040714,
	0.00203252032520325225, 0.00202839756592292086, 0.00202429149797570852,
	    0.00202020202020202020,
	0.00201612903225806446, 0.00201207243460764604, 0.00200803212851405604,
	    0.00200400801603206396,
	0.00200000000000000004, 0.00199600798403193596, 0.00199203187250996012,
	    0.00198807157057654055,
	0.00198412698412698402, 0.00198019801980198022, 0.00197628458498023702,
	    0.00197238658777120325,
	0.00196850393700787399, 0.00196463654223968552, 0.00196078431372549017,
	    0.00195694716242661437,
	0.00195312500000000000, 0.00194931773879142289, 0.00194552529182879380,
	    0.00194174757281553390,
	0.00193798449612403098, 0.00193423597678916829, 0.00193050193050193055,
	    0.00192678227360308283,
	0.00192307692307692318, 0.00191938579654510550, 0.00191570881226053630,
	    0.00191204588910133836,
	0.00190839694656488542, 0.00190476190476190476, 0.00190114068441064638,
	    0.00189753320683111958,
	0.00189393939393939399, 0.00189035916824196602, 0.00188679245283018869,
	    0.00188323917137476452,
	0.00187969924812030065, 0.00187617260787992495, 0.00187265917602996257,
	    0.00186915887850467297,
	0.00186567164179104474, 0.00186219739292364989, 0.00185873605947955382,
	    0.00185528756957328389,
	0.00185185185185185192, 0.00184842883548983362, 0.00184501845018450179,
	    0.00184162062615101296,
	0.00183823529411764703, 0.00183486238532110102, 0.00183150183150183150,
	    0.00182815356489945155,
	0.00182481751824817512, 0.00182149362477231330, 0.00181818181818181818,
	    0.00181488203266787652,
	0.00181159420289855075, 0.00180831826401446649, 0.00180505415162454880,
	    0.00180180180180180183,
	0.00179856115107913678, 0.00179533213644524244, 0.00179211469534050179,
	    0.00178890876565295171,
	0.00178571428571428566, 0.00178253119429590007, 0.00177935943060498212,
	    0.00177619893428063950,
	0.00177304964539007088, 0.00176991150442477874, 0.00176678445229681976,
	    0.00176366843033509690,
	0.00176056338028169018, 0.00175746924428822485, 0.00175438596491228073,
	    0.00175131348511383539,
	0.00174825174825174825, 0.00174520069808027927, 0.00174216027874564460,
	    0.00173913043478260876,
	0.00173611111111111101, 0.00173310225303292885, 0.00173010380622837375,
	    0.00172711571675302237,
	0.00172413793103448273, 0.00172117039586919111, 0.00171821305841924395,
	    0.00171526586620926241,
	0.00171232876712328758, 0.00170940170940170940, 0.00170648464163822527,
	    0.00170357751277683137,
	0.00170068027210884347, 0.00169779286926994904, 0.00169491525423728809,
	    0.00169204737732656508,
	0.00168918918918918928, 0.00168634064080944342, 0.00168350168350168343,
	    0.00168067226890756313,
	0.00167785234899328855, 0.00167504187604690120, 0.00167224080267558525,
	    0.00166944908180300506,
	0.00166666666666666677, 0.00166389351081530786, 0.00166112956810631233,
	    0.00165837479270315095,
	0.00165562913907284768, 0.00165289256198347103, 0.00165016501650165019,
	    0.00164744645799011534,
	0.00164473684210526307, 0.00164203612479474543, 0.00163934426229508197,
	    0.00163666121112929631,
	0.00163398692810457521, 0.00163132137030995114, 0.00162866449511400659,
	    0.00162601626016260162,
	0.00162337662337662348, 0.00162074554294975681, 0.00161812297734627839,
	    0.00161550888529886924,
	0.00161290322580645161, 0.00161030595813204508, 0.00160771704180064307,
	    0.00160513643659711074,
	0.00160256410256410251, 0.00160000000000000008, 0.00159744408945686894,
	    0.00159489633173843701,
	0.00159235668789808925, 0.00158982511923688404, 0.00158730158730158730,
	    0.00158478605388272589,
	0.00158227848101265820, 0.00157977883096366506, 0.00157728706624605673,
	    0.00157480314960629919,
	0.00157232704402515732, 0.00156985871271585566, 0.00156739811912225699,
	    0.00156494522691705794,
	0.00156250000000000009, 0.00156006240249609990, 0.00155763239875389404,
	    0.00155520995334370140,
	0.00155279503105590052, 0.00155038759689922478, 0.00154798761609907130,
	    0.00154559505409582686,
	0.00154320987654320979, 0.00154083204930662563, 0.00153846153846153846,
	    0.00153609831029185868,
	0.00153374233128834359, 0.00153139356814701384, 0.00152905198776758411,
	    0.00152671755725190838,
	0.00152439024390243908, 0.00152207001522070007, 0.00151975683890577507,
	    0.00151745068285280733,
	0.00151515151515151515, 0.00151285930408472016, 0.00151057401812688822,
	    0.00150829562594268483,
	0.00150602409638554225, 0.00150375939849624069, 0.00150150150150150149,
	    0.00149925037481259365,
	0.00149700598802395219, 0.00149476831091180872, 0.00149253731343283584,
	    0.00149031296572280179,
	0.00148809523809523801, 0.00148588410104011880, 0.00148367952522255198,
	    0.00148148148148148141,
	0.00147928994082840233, 0.00147710487444608577, 0.00147492625368731561,
	    0.00147275405007363767,
	0.00147058823529411763, 0.00146842878120411152, 0.00146627565982404692,
	    0.00146412884333821380,
	0.00146198830409356717, 0.00145985401459854005, 0.00145772594752186587,
	    0.00145560407569141200,
	0.00145348837209302324, 0.00145137880986937590, 0.00144927536231884060,
	    0.00144717800289435590,
	0.00144508670520231207, 0.00144300144300144300, 0.00144092219020172901,
	    0.00143884892086330938,
	0.00143678160919540228, 0.00143472022955523680, 0.00143266475644699136,
	    0.00143061516452074391,
	0.00142857142857142857, 0.00142653352353780318, 0.00142450142450142457,
	    0.00142247510668563307,
	0.00142045454545454549, 0.00141843971631205683, 0.00141643059490084993,
	    0.00141442715700141448,
	0.00141242937853107344, 0.00141043723554301831, 0.00140845070422535219,
	    0.00140646976090014067,
	0.00140449438202247187, 0.00140252454417952310, 0.00140056022408963587,
	    0.00139860139860139860,
	0.00139664804469273747, 0.00139470013947001390, 0.00139275766016713092,
	    0.00139082058414464528,
	0.00138888888888888894, 0.00138696255201109574, 0.00138504155124653740,
	    0.00138312586445366536,
	0.00138121546961325961, 0.00137931034482758610, 0.00137741046831955927,
	    0.00137551581843191198,
	0.00137362637362637373, 0.00137174211248285312, 0.00136986301369863015,
	    0.00136798905608755128,
	0.00136612021857923497, 0.00136425648021828104, 0.00136239782016348769,
	    0.00136054421768707474,
	0.00135869565217391301, 0.00135685210312075973, 0.00135501355013550135,
	    0.00135317997293640064,
	0.00135135135135135143, 0.00134952766531713894, 0.00134770889487870627,
	    0.00134589502018842531,
	0.00134408602150537645, 0.00134228187919463080, 0.00134048257372654158,
	    0.00133868808567603743,
	0.00133689839572192506, 0.00133511348464619489, 0.00133333333333333329,
	    0.00133155792276964057,
	0.00132978723404255316, 0.00132802124833997334, 0.00132625994694960205,
	    0.00132450331125827814,
	0.00132275132275132268, 0.00132100396301188905, 0.00131926121372031658,
	    0.00131752305665349149,
	0.00131578947368421050, 0.00131406044678055193, 0.00131233595800524933,
	    0.00131061598951507209,
	0.00130890052356020950, 0.00130718954248366004, 0.00130548302872062665,
	    0.00130378096479791391,
	0.00130208333333333326, 0.00130039011703511056, 0.00129870129870129870,
	    0.00129701686121919580,
	0.00129533678756476689, 0.00129366106080206996, 0.00129198966408268739,
	    0.00129032258064516129,
	0.00128865979381443297, 0.00128700128700128696, 0.00128534704370179939,
	    0.00128369704749679066,
	0.00128205128205128205, 0.00128040973111395642, 0.00127877237851662415,
	    0.00127713920817369101,
	0.00127551020408163255, 0.00127388535031847127, 0.00127226463104325709,
	    0.00127064803049555279,
	0.00126903553299492376, 0.00126742712294043085, 0.00126582278481012661,
	    0.00126422250316055636,
	0.00126262626262626273, 0.00126103404791929382, 0.00125944584382871533,
	    0.00125786163522012572,
	0.00125628140703517590, 0.00125470514429109150, 0.00125313283208020043,
	    0.00125156445556946186,
	0.00125000000000000003, 0.00124843945068664171, 0.00124688279301745633,
	    0.00124533001245330011,
	0.00124378109452736316, 0.00124223602484472054, 0.00124069478908188586,
	    0.00123915737298636928,
	0.00123762376237623764, 0.00123609394313967851, 0.00123456790123456788,
	    0.00123304562268803947,
	0.00123152709359605913, 0.00123001230012300127, 0.00122850122850122847,
	    0.00122699386503067492,
	0.00122549019607843136, 0.00122399020807833527, 0.00122249388753056234,
	    0.00122100122100122100,
	0.00121951219512195122, 0.00121802679658952490, 0.00121654501216545015,
	    0.00121506682867557705,
	0.00121359223300970863, 0.00121212121212121212, 0.00121065375302663444,
	    0.00120918984280532038,
	0.00120772946859903376, 0.00120627261761158014, 0.00120481927710843384,
	    0.00120336943441636587,
	0.00120192307692307699, 0.00120048019207683065, 0.00119904076738609104,
	    0.00119760479041916166,
	0.00119617224880382770, 0.00119474313022700112, 0.00119331742243436760,
	    0.00119189511323003570,
	0.00119047619047619058, 0.00118906064209274662, 0.00118764845605700718,
	    0.00118623962040332155,
	0.00118483412322274891, 0.00118343195266272191, 0.00118203309692671392,
	    0.00118063754427390785,
	0.00117924528301886788, 0.00117785630153121310, 0.00117647058823529401,
	    0.00117508813160987070,
	0.00117370892018779345, 0.00117233294255568573, 0.00117096018735362998,
	    0.00116959064327485382,
	0.00116822429906542047, 0.00116686114352392055, 0.00116550116550116550,
	    0.00116414435389988356,
	0.00116279069767441859, 0.00116144018583042973, 0.00116009280742459392,
	    0.00115874855156431053,
	0.00115740740740740734, 0.00115606936416184978, 0.00115473441108545033,
	    0.00115340253748558250,
	0.00115207373271889401, 0.00115074798619102417, 0.00114942528735632182,
	    0.00114810562571756604,
	0.00114678899082568814, 0.00114547537227949604, 0.00114416475972540046,
	    0.00114285714285714294,
	0.00114155251141552505, 0.00114025085518814143, 0.00113895216400911156,
	    0.00113765642775881678,
	0.00113636363636363631, 0.00113507377979568669, 0.00113378684807256239,
	    0.00113250283125707822,
	0.00113122171945701368, 0.00112994350282485880, 0.00112866817155756198,
	    0.00112739571589627954,
	0.00112612612612612612, 0.00112485939257592812, 0.00112359550561797763,
	    0.00112233445566778910,
	0.00112107623318385649, 0.00111982082866741322, 0.00111856823266219237,
	    0.00111731843575418985,
	0.00111607142857142851, 0.00111482720178372350, 0.00111358574610244984,
	    0.00111234705228031145,
	0.00111111111111111111, 0.00110987791342952277, 0.00110864745011086474,
	    0.00110741971207087482,
	0.00110619469026548671, 0.00110497237569060782, 0.00110375275938189845,
	    0.00110253583241455349,
	0.00110132158590308380, 0.00110011001100110005, 0.00109890109890109890,
	    0.00109769484083424812,
	0.00109649122807017538, 0.00109529025191675792, 0.00109409190371991243,
	    0.00109289617486338798,
	0.00109170305676855890, 0.00109051254089422033, 0.00108932461873638355,
	    0.00108813928182807402,
	0.00108695652173913045, 0.00108577633007600439, 0.00108459869848156185,
	    0.00108342361863488618,
	0.00108225108225108225, 0.00108108108108108110, 0.00107991360691144715,
	    0.00107874865156418545,
	0.00107758620689655171, 0.00107642626480086104, 0.00107526881720430107,
	    0.00107411385606874326,
	0.00107296137339055794, 0.00107181136120042871, 0.00107066381156316922,
	    0.00106951871657754013,
	0.00106837606837606848, 0.00106723585912486666, 0.00106609808102345414,
	    0.00106496272630457944,
	0.00106382978723404257, 0.00106269925611052066, 0.00106157112526539276,
	    0.00106044538706256638,
	0.00105932203389830508, 0.00105820105820105827, 0.00105708245243128969,
	    0.00105596620908130932,
	0.00105485232067510540, 0.00105374077976817695, 0.00105263157894736840,
	    0.00105152471083070449,
	0.00105042016806722685, 0.00104931794333683109, 0.00104821802935010488,
	    0.00104712041884816765,
	0.00104602510460251037, 0.00104493207941483814, 0.00104384133611691013,
	    0.00104275286757038585,
	0.00104166666666666665, 0.00104058272632674307, 0.00103950103950103956,
	    0.00103842159916926269,
	0.00103734439834024896, 0.00103626943005181342, 0.00103519668737060049,
	    0.00103412616339193389,
	0.00103305785123966945, 0.00103199174406604739, 0.00103092783505154642,
	    0.00102986611740473735,
	0.00102880658436214001, 0.00102774922918807813, 0.00102669404517453803,
	    0.00102564102564102564,
	0.00102459016393442628, 0.00102354145342886389, 0.00102249488752556247,
	    0.00102145045965270687,
	0.00102040816326530621, 0.00101936799184505615, 0.00101832993890020357,
	    0.00101729399796541197,
	0.00101626016260162612, 0.00101522842639593914, 0.00101419878296146043,
	    0.00101317122593718345,
	0.00101214574898785426, 0.00101112234580384231, 0.00101010101010101010,
	    0.00100908173562058528,
	0.00100806451612903223, 0.00100704934541792548, 0.00100603621730382302,
	    0.00100502512562814081,
	0.00100401606425702802, 0.00100300902708124373, 0.00100200400801603198,
	    0.00100100100100100099,
	0.00100000000000000002, 0.00099900099900099900, 0.00099800399201596798,
	    0.00099700897308075765,
	0.00099601593625498006, 0.00099502487562189048, 0.00099403578528827028,
	    0.00099304865938430980,
	0.00099206349206349201, 0.00099108027750247768, 0.00099009900990099011,
	    0.00098911968348170125,
	0.00098814229249011851, 0.00098716683119447180, 0.00098619329388560163,
	    0.00098522167487684722,
	0.00098425196850393699, 0.00098328416912487715, 0.00098231827111984276,
	    0.00098135426889106960,
	0.00098039215686274508, 0.00097943192948090111, 0.00097847358121330719,
	    0.00097751710654936461,
};

/* *********************************************************** */

#define	LOAD(res, sp)	res = *(sp)

/* *********************************************************** */

#ifdef MLIB_VIS_EXPERIMENT

/* *********************************************************** */

/*
 * vis_ld_u8/u16 can be generated automatically by Mercury (Sun C 5.6)
 * and later.
 */

/* The macro causes a slowdown in most cases, so let's turn it off. */
#define	LOAD_u8(res, sp)                                        \
	{                                                       \
	    d64_2x32 ss;                                        \
	                                                        \
	    ss.d64 = vis_ld_u8(sp);                             \
	    res = ss.i32s.f1;                                   \
	}

#else /* MLIB_VIS_EXPERIMENT */

#define	LOAD_u8	LOAD

#endif /* MLIB_VIS_EXPERIMENT */

/* *********************************************************** */

#define	LOAD_x8	LOAD_u8
#define	mlib_x8	mlib_u8

/* *********************************************************** */

#define	LOAD_s16	LOAD
#define	LOAD_u16	LOAD
#define	LOAD_s32	LOAD
#define	LOAD_f32	LOAD
#define	LOAD_d64	LOAD

/* *********************************************************** */

#define	ADD_1s(DTYPE)                                           \
	ATYPE alp = *ap;                                        \
	                                                        \
	*buffa += alp;                                          \
	buffs[0] += s0 * alp;                                   \
	ap += 1;                                                \
	sp += 1;                                                \
	buffa += 1;                                             \
	buffs += 1

/* *********************************************************** */

#define	ADD_2s(DTYPE)                                           \
	ATYPE alp = *ap;                                        \
	                                                        \
	*buffa += alp;                                          \
	buffs[0] += s0 * alp;                                   \
	buffs[1] += s1 * alp;                                   \
	ap += 1;                                                \
	sp += 2;                                                \
	buffa += 1;                                             \
	buffs += 2

/* *********************************************************** */

#define	ADD_3s(DTYPE)                                           \
	ATYPE alp = *ap;                                        \
	                                                        \
	*buffa += alp;                                          \
	buffs[0] += s0 * alp;                                   \
	buffs[1] += s1 * alp;                                   \
	buffs[2] += s2 * alp;                                   \
	ap += 1;                                                \
	sp += 3;                                                \
	buffa += 1;                                             \
	buffs += 3

/* *********************************************************** */

#define	ADD_4s(DTYPE)                                           \
	ATYPE alp = *ap;                                        \
	                                                        \
	*buffa += alp;                                          \
	buffs[0] += s0 * alp;                                   \
	buffs[1] += s1 * alp;                                   \
	buffs[2] += s2 * alp;                                   \
	buffs[3] += s3 * alp;                                   \
	ap += 1;                                                \
	sp += 4;                                                \
	buffa += 1;                                             \
	buffs += 4

/* *********************************************************** */

#define	ADD_2m(DTYPE)                                           \
	ATYPE alp0, alp1;                                       \
	                                                        \
	alp0 = ap[0];                                           \
	buffa[0] += alp0;                                       \
	buffs[0] += s0 * alp0;                                  \
	alp1 = ap[1];                                           \
	buffa[1] += alp1;                                       \
	buffs[1] += s1 * alp1;                                  \
	ap += 2;                                                \
	sp += 2;                                                \
	buffa += 2;                                             \
	buffs += 2

/* *********************************************************** */

#define	ADD_3m(DTYPE)                                           \
	ATYPE alp0, alp1, alp2;                                 \
	                                                        \
	alp0 = ap[0];                                           \
	buffa[0] += alp0;                                       \
	buffs[0] += s0 * alp0;                                  \
	alp1 = ap[1];                                           \
	buffa[1] += alp1;                                       \
	buffs[1] += s1 * alp1;                                  \
	alp2 = ap[2];                                           \
	buffa[2] += alp2;                                       \
	buffs[2] += s2 * alp2;                                  \
	ap += 3;                                                \
	sp += 3;                                                \
	buffa += 3;                                             \
	buffs += 3

/* *********************************************************** */

#define	ADD_4m(DTYPE)                                           \
	ATYPE alp0, alp1, alp2, alp3;                           \
	                                                        \
	alp0 = ap[0];                                           \
	buffa[0] += alp0;                                       \
	buffs[0] += s0 * alp0;                                  \
	alp1 = ap[1];                                           \
	buffa[1] += alp1;                                       \
	buffs[1] += s1 * alp1;                                  \
	alp2 = ap[2];                                           \
	buffa[2] += alp2;                                       \
	buffs[2] += s2 * alp2;                                  \
	alp3 = ap[3];                                           \
	buffa[3] += alp3;                                       \
	buffs[3] += s3 * alp3;                                  \
	ap += 4;                                                \
	sp += 4;                                                \
	buffa += 4;                                             \
	buffs += 4

/* *********************************************************** */

#define	ADD_2x(DTYPE)                                           \
	ATYPE alp = *ap;                                        \
	                                                        \
	buffa[0] += alp;                                        \
	buffs[0] += s0 * alp;                                   \
	buffa[1] += alp;                                        \
	buffs[1] += s1 * alp;                                   \
	ap += 1;                                                \
	sp += 2;                                                \
	buffa += 2;                                             \
	buffs += 2

/* *********************************************************** */

#define	ADD_3x(DTYPE)                                           \
	ATYPE alp = *ap;                                        \
	                                                        \
	buffa[0] += alp;                                        \
	buffs[0] += s0 * alp;                                   \
	buffa[1] += alp;                                        \
	buffs[1] += s1 * alp;                                   \
	buffa[2] += alp;                                        \
	buffs[2] += s2 * alp;                                   \
	ap += 1;                                                \
	sp += 3;                                                \
	buffa += 3;                                             \
	buffs += 3

/* *********************************************************** */

#define	ADD_4x(DTYPE)                                           \
	ATYPE alp = *ap;                                        \
	                                                        \
	buffa[0] += alp;                                        \
	buffs[0] += s0 * alp;                                   \
	buffa[1] += alp;                                        \
	buffs[1] += s1 * alp;                                   \
	buffa[2] += alp;                                        \
	buffs[2] += s2 * alp;                                   \
	buffa[3] += alp;                                        \
	buffs[3] += s3 * alp;                                   \
	ap += 1;                                                \
	sp += 4;                                                \
	buffa += 4;                                             \
	buffs += 4

/* *********************************************************** */

#define	ADD_1sl(DTYPE)                                          \
	ATYPE alp, alp_s;                                       \
	                                                        \
	LOAD_##DTYPE(alp, ap);                                  \
	alp_s = alp + *buffa;                                   \
	DIV_##DTYPE(0, buffs[0] + s0 * alp, alp_s);             \
	buffs[0] = 0;                                           \
	buffa[0] = 0;                                           \
	ap += 1;                                                \
	sp += 1;                                                \
	buffa += 1;                                             \
	buffs += 1;                                             \
	dp += 1

/* *********************************************************** */

#define	ADD_2sl(DTYPE)                                          \
	ATYPE alp, alp_s;                                       \
	                                                        \
	LOAD_##DTYPE(alp, ap);                                  \
	alp_s = alp + *buffa;                                   \
	DIV_##DTYPE(0, buffs[0] + s0 * alp, alp_s);             \
	DIV_##DTYPE(1, buffs[1] + s1 * alp, alp_s);             \
	buffs[0] = 0;                                           \
	buffa[0] = 0;                                           \
	buffs[1] = 0;                                           \
	ap += 1;                                                \
	sp += 2;                                                \
	buffa += 1;                                             \
	buffs += 2;                                             \
	dp += 2

/* *********************************************************** */

#define	ADD_3sl(DTYPE)                                          \
	ATYPE alp, alp_s;                                       \
	                                                        \
	LOAD_##DTYPE(alp, ap);                                  \
	alp_s = alp + *buffa;                                   \
	DIV_##DTYPE(0, buffs[0] + s0 * alp, alp_s);             \
	DIV_##DTYPE(1, buffs[1] + s1 * alp, alp_s);             \
	DIV_##DTYPE(2, buffs[2] + s2 * alp, alp_s);             \
	buffs[0] = 0;                                           \
	buffa[0] = 0;                                           \
	buffs[1] = 0;                                           \
	buffs[2] = 0;                                           \
	ap += 1;                                                \
	sp += 3;                                                \
	buffa += 1;                                             \
	buffs += 3;                                             \
	dp += 3

/* *********************************************************** */

#define	ADD_4sl(DTYPE)                                          \
	ATYPE alp, alp_s;                                       \
	                                                        \
	LOAD_##DTYPE(alp, ap);                                  \
	alp_s = alp + *buffa;                                   \
	DIV_##DTYPE(0, buffs[0] + s0 * alp, alp_s);             \
	DIV_##DTYPE(1, buffs[1] + s1 * alp, alp_s);             \
	DIV_##DTYPE(2, buffs[2] + s2 * alp, alp_s);             \
	DIV_##DTYPE(3, buffs[3] + s3 * alp, alp_s);             \
	buffs[0] = 0;                                           \
	buffa[0] = 0;                                           \
	buffs[1] = 0;                                           \
	buffs[2] = 0;                                           \
	buffs[3] = 0;                                           \
	ap += 1;                                                \
	sp += 4;                                                \
	buffa += 1;                                             \
	buffs += 4;                                             \
	dp += 4

/* *********************************************************** */

#define	ADD_2ml(DTYPE)                                           \
	ATYPE alp0, alp1;                                        \
	                                                         \
	alp0 = ap[0];                                            \
	alp1 = ap[1];                                            \
	DIV_##DTYPE(0, buffs[0] + s0 * alp0, buffa[0] + alp0);   \
	DIV_##DTYPE(1, buffs[1] + s1 * alp1, buffa[1] + alp1);   \
	buffs[0] = 0;                                            \
	buffa[0] = 0;                                            \
	buffs[1] = 0;                                            \
	buffa[1] = 0;                                            \
	ap += 2;                                                 \
	sp += 2;                                                 \
	buffa += 2;                                              \
	buffs += 2;                                              \
	dp += 2

/* *********************************************************** */

#define	ADD_3ml(DTYPE)                                           \
	ATYPE alp0, alp1, alp2;                                  \
	                                                         \
	alp0 = ap[0];                                            \
	alp1 = ap[1];                                            \
	alp2 = ap[2];                                            \
	DIV_##DTYPE(0, buffs[0] + s0 * alp0, buffa[0] + alp0);   \
	DIV_##DTYPE(1, buffs[1] + s1 * alp1, buffa[1] + alp1);   \
	DIV_##DTYPE(2, buffs[2] + s2 * alp2, buffa[2] + alp2);   \
	buffs[0] = 0;                                            \
	buffa[0] = 0;                                            \
	buffs[1] = 0;                                            \
	buffa[1] = 0;                                            \
	buffs[2] = 0;                                            \
	buffa[2] = 0;                                            \
	ap += 3;                                                 \
	sp += 3;                                                 \
	buffa += 3;                                              \
	buffs += 3;                                              \
	dp += 3

/* *********************************************************** */

#define	ADD_4ml(DTYPE)                                           \
	ATYPE alp0, alp1, alp2, alp3;                            \
	                                                         \
	alp0 = ap[0];                                            \
	alp1 = ap[1];                                            \
	alp2 = ap[2];                                            \
	alp3 = ap[3];                                            \
	DIV_##DTYPE(0, buffs[0] + s0 * alp0, buffa[0] + alp0);   \
	DIV_##DTYPE(1, buffs[1] + s1 * alp1, buffa[1] + alp1);   \
	DIV_##DTYPE(2, buffs[2] + s2 * alp2, buffa[2] + alp2);   \
	DIV_##DTYPE(3, buffs[3] + s3 * alp3, buffa[3] + alp3);   \
	buffs[0] = 0;                                            \
	buffa[0] = 0;                                            \
	buffs[1] = 0;                                            \
	buffa[1] = 0;                                            \
	buffs[2] = 0;                                            \
	buffa[2] = 0;                                            \
	buffs[3] = 0;                                            \
	buffa[3] = 0;                                            \
	ap += 4;                                                 \
	sp += 4;                                                 \
	buffa += 4;                                              \
	buffs += 4;                                              \
	dp += 4

/* *********************************************************** */

#define	ADD_2xl(DTYPE)                                          \
	ATYPE alp = *ap;                                        \
	                                                        \
	DIV_##DTYPE(0, buffs[0] + s0 * alp, buffa[0] + alp);    \
	DIV_##DTYPE(1, buffs[1] + s1 * alp, buffa[1] + alp);    \
	buffs[0] = 0;                                           \
	buffa[0] = 0;                                           \
	buffs[1] = 0;                                           \
	buffa[1] = 0;                                           \
	ap += 1;                                                \
	sp += 2;                                                \
	buffa += 2;                                             \
	buffs += 2;                                             \
	dp += 2

/* *********************************************************** */

#define	ADD_3xl(DTYPE)                                          \
	ATYPE alp = *ap;                                        \
	                                                        \
	DIV_##DTYPE(0, buffs[0] + s0 * alp, buffa[0] + alp);    \
	DIV_##DTYPE(1, buffs[1] + s1 * alp, buffa[1] + alp);    \
	DIV_##DTYPE(2, buffs[2] + s2 * alp, buffa[2] + alp);    \
	buffs[0] = 0;                                           \
	buffa[0] = 0;                                           \
	buffs[1] = 0;                                           \
	buffa[1] = 0;                                           \
	buffs[2] = 0;                                           \
	buffa[2] = 0;                                           \
	ap += 1;                                                \
	sp += 3;                                                \
	buffa += 3;                                             \
	buffs += 3;                                             \
	dp += 3

/* *********************************************************** */

#define	ADD_4xl(DTYPE)                                          \
	ATYPE alp = *ap;                                        \
	                                                        \
	DIV_##DTYPE(0, buffs[0] + s0 * alp, buffa[0] + alp);    \
	DIV_##DTYPE(1, buffs[1] + s1 * alp, buffa[1] + alp);    \
	DIV_##DTYPE(2, buffs[2] + s2 * alp, buffa[2] + alp);    \
	DIV_##DTYPE(3, buffs[3] + s3 * alp, buffa[3] + alp);    \
	buffs[0] = 0;                                           \
	buffa[0] = 0;                                           \
	buffs[1] = 0;                                           \
	buffa[1] = 0;                                           \
	buffs[2] = 0;                                           \
	buffa[2] = 0;                                           \
	buffs[3] = 0;                                           \
	buffa[3] = 0;                                           \
	ap += 1;                                                \
	sp += 4;                                                \
	buffa += 4;                                             \
	buffs += 4;                                             \
	dp += 4

/* *********************************************************** */

#define	DEF_u8                                                  \
	const mlib_s32 *bcolor = _bcolor;                       \
	FTYPE dscale = mlib_div_u10[1 << shift];                \
	mlib_s32 asum, res, mask

/* *********************************************************** */

#define	DIV_u8(CH, x, y)                                             \
	asum = y;                                                    \
	SATUR_U8(res, (x) * dscale * mlib_div_u10[asum >> shift]);   \
	mask = (asum | (-asum)) >> 31;                               \
	dp[CH] = (res & mask) | (bcolor[CH] & ~mask)

/* *********************************************************** */

#define	DEF_x8                                                  \
	const mlib_s32 *bcolor = _bcolor;                       \
	mlib_s32 asum, res, mask

/* *********************************************************** */

#define	DIV_x8(CH, x, y)                                        \
	asum = y;                                               \
	SATUR_U8(res, (x) / asum);                              \
	mask = (asum | (-asum)) >> 31;                          \
	dp[CH] = (res & mask) | (bcolor[CH] & ~mask)

/* *********************************************************** */

#define	DEF_s16                                                 \
	const mlib_s32 *bcolor = _bcolor;                       \
	mlib_s32 asum, res, mask

/* *********************************************************** */

#ifndef	MLIB_VIS_EXPERIMENT

#define	DIV_s16(CH, x, y)                                       \
	asum = y;                                               \
	SATUR_S16(res, (x) / asum);                             \
/*                                                              \
 * res = (mlib_s32)((x)/asum);                                  \
 */                                                             \
	mask = (asum | (-asum)) >> 31;                          \
	dp[CH] = (res & mask) | (bcolor[CH] & ~mask)

#elif MLIB_VIS_EXPERIMENT >= 0x200

/* This is slower than the one using SATUR_S16(). */
#define	DIV_s16(CH, x, y)                                       \
	{                                                       \
	    d64_2x32 ss;                                        \
	                                                        \
	    asum = y;                                           \
	    ss.i32s.f0 = 0;                                     \
	    ss.i32s.f1 = (x) / asum;                            \
	    ss.f32s.f1 = vis_fpackfix(ss.d64);                  \
	    res = ss.i32s.f1;                                   \
	    mask = (asum | (-asum)) >> 31;                      \
	    dp[CH] = (res & mask) | (bcolor[CH] & ~mask);       \
	}

#else

/*
 * vis_fpadd32 could cause an overflow and wraparound, so the
 * following macro might not do s16 saturation in some cases.
 */
#define	DIV_s16(CH, x, y)                                       \
	{                                                       \
	    d64_2x32 ss;                                        \
	                                                        \
	    asum = y;                                           \
	    ss.i32s.f0 = 0;                                     \
	    ss.i32s.f1 = (x) / asum;                            \
	    ss.d64 = vis_fpadd32(ss.d64, ss.d64);               \
	    ss.f32s.f1 = vis_fpackfix(ss.d64);                  \
	    res = ss.i32s.f1;                                   \
	    mask = (asum | (-asum)) >> 31;                      \
	    dp[CH] = (res & mask) | (bcolor[CH] & ~mask);       \
	}

#endif /* MLIB_VIS_EXPERIMENT */

/* *********************************************************** */

#define	DEF_u16                                                 \
	const mlib_s32 *bcolor = _bcolor;                       \
	mlib_s32 asum, res, mask

/* *********************************************************** */

#define	DIV_u16(CH, x, y)                                       \
	asum = y;                                               \
	SATUR_U16(res, (x) / asum);                             \
	mask = (asum | (-asum)) >> 31;                          \
	dp[CH] = (res & mask) | (bcolor[CH] & ~mask)

/* *********************************************************** */

#define	DEF_s32                                                 \
	const mlib_s32 *bcolor = _bcolor;                       \
	mlib_s32 isum, res, mask;                               \
	FTYPE asum

/* *********************************************************** */

#define	DIV_s32(CH, x, y)                                       \
	asum = y;                                               \
	SATUR_S32(isum, asum);                                  \
	mask = (isum | (-isum)) >> 31;                          \
	SATUR_S32(res, (x) / asum);                             \
	dp[CH] = (res & mask) | (bcolor[CH] & ~mask)

/* *********************************************************** */

#define	DEF_f32                                                 \
	                                                        \
/*                                                              \
 * mlib_d64 *bcolor = (void*)_bcolor;                           \
 */                                                             \
	ATYPE *buffa2 = (void *)_bcolor;                        \
	FTYPE asum, res

/* *********************************************************** */

#define	DIV_f32(CH, x, y)                                       \
	asum = y;                                               \
	(*buffa2++) = asum;                                     \
	res = (x) / asum;                                       \
/*                                                              \
 * if (asum == 0) res = bcolor[CH];                             \
 */                                                             \
	dp[CH] = res

/* *********************************************************** */

#define	DEF_d64	DEF_f32

/* *********************************************************** */

#define	DIV_d64(CH, x, y)	DIV_f32(CH, x, y)

/* *********************************************************** */

#define	LOAD_SRC_1(TYPE)	FTYPE s0; LOAD_##TYPE(s0, sp)

/* *********************************************************** */

#define	LOAD_SRC_2(TYPE)                                        \
	FTYPE s0, s1;                                           \
	                                                        \
	LOAD_##TYPE(s0, sp);                                    \
	LOAD_##TYPE(s1, sp + 1)

/* *********************************************************** */

#define	LOAD_SRC_3(TYPE)                                        \
	FTYPE s0, s1, s2;                                       \
	                                                        \
	LOAD_##TYPE(s0, sp);                                    \
	LOAD_##TYPE(s1, sp + 1);                                \
	LOAD_##TYPE(s2, sp + 2)

/* *********************************************************** */

#define	LOAD_SRC_4(TYPE)                                        \
	FTYPE s0, s1, s2, s3;                                   \
	                                                        \
	LOAD_##TYPE(s0, sp);                                    \
	LOAD_##TYPE(s1, sp + 1);                                \
	LOAD_##TYPE(s2, sp + 2);                                \
	LOAD_##TYPE(s3, sp + 3)

/* *********************************************************** */

#define	LOAD_BCOLOR(N, DTYPE)	FTYPE c##N = bcolor[N]

/* *********************************************************** */

#define	CHECK_ZERO(N, DTYPE)                                    \
	if (buffa[N] == 0)                                      \
	    dp[N] = (mlib_##DTYPE) c##N

/* *********************************************************** */

#define	MACRO_1(MACRO, DTYPE)	MACRO(0, DTYPE)

/* *********************************************************** */

#define	MACRO_2(MACRO, DTYPE)	MACRO(0, DTYPE); MACRO(1, DTYPE)

/* *********************************************************** */

#define	MACRO_3(MACRO, DTYPE)                                   \
	MACRO(0, DTYPE);                                        \
	MACRO(1, DTYPE);                                        \
	MACRO(2, DTYPE)

/* *********************************************************** */

#define	MACRO_4(MACRO, DTYPE)                                   \
	MACRO(0, DTYPE);                                        \
	MACRO(1, DTYPE);                                        \
	MACRO(2, DTYPE);                                        \
	MACRO(3, DTYPE)

/* *********************************************************** */

#define	ZERO_FUNC(DTYPE, CH)                                    \
	static void                                             \
	mlib_ImageBlendMulti_Zero_##DTYPE##_##CH(FUNC_ARG)      \
	{                                                       \
	    mlib_##DTYPE *dp = dl;                              \
	    mlib_##DTYPE *buffa = _buffa;                       \
	    const mlib_d64 *bcolor = _bcolor;                   \
	                                                        \
	    MACRO_##CH(LOAD_BCOLOR, DTYPE);                     \
	    mlib_s32 i;                                         \
	                                                        \
	    for (i = 0; i < wid; i++) {                         \
		MACRO_##CH(CHECK_ZERO, DTYPE);                  \
		buffa += CH;                                    \
		dp += CH;                                       \
	    }                                                   \
	}

/* *********************************************************** */

ZERO_FUNC(f32, 1)
ZERO_FUNC(f32, 2)
ZERO_FUNC(f32, 3)
ZERO_FUNC(f32, 4)
ZERO_FUNC(d64, 1)
ZERO_FUNC(d64, 2)
ZERO_FUNC(d64, 3)
ZERO_FUNC(d64, 4)

func_type mlib_func_arr_zero[] = {
	mlib_ImageBlendMulti_Zero_f32_1,
	mlib_ImageBlendMulti_Zero_f32_2,
	mlib_ImageBlendMulti_Zero_f32_3,
	mlib_ImageBlendMulti_Zero_f32_4,
	mlib_ImageBlendMulti_Zero_d64_1,
	mlib_ImageBlendMulti_Zero_d64_2,
	mlib_ImageBlendMulti_Zero_d64_3,
	mlib_ImageBlendMulti_Zero_d64_4
};

/* *********************************************************** */

#define	LINE_FUNC_A(DTYPE, CH, AA)                              \
	static void                                             \
	mlib_ImageBlendMulti_##DTYPE##_##CH##AA                 \
	    (FUNC_ARG)                                          \
	{                                                       \
	    mlib_##DTYPE *sp = sl;                              \
	    mlib_##DTYPE *ap = al;                              \
	    FTYPE *buffs = _buffs;                              \
	    ATYPE *buffa = _buffa;                              \
	    mlib_s32 i;                                         \
	                                                        \
	    for (i = 0; i < wid; i++) {                         \
		LOAD_SRC_##CH(DTYPE); {                         \
		    ADD_##CH##AA(DTYPE);                        \
		}                                               \
	    }                                                   \
	}

/* *********************************************************** */

#ifndef	MLIB_VIS_EXPERIMENT

#define	LINE_FUNC_L(DTYPE, CH, AA)                              \
	static void                                             \
	mlib_ImageBlendMulti_##DTYPE##_##CH##AA                 \
	    (FUNC_ARG)                                          \
	{                                                       \
	    mlib_##DTYPE *sp = sl;                              \
	    mlib_##DTYPE *ap = al;                              \
	    FTYPE *buffs = _buffs;                              \
	    ATYPE *buffa = _buffa;                              \
	    mlib_s32 i;                                         \
	    mlib_##DTYPE *dp = dl;                              \
	                                                        \
	    DEF_##DTYPE;                                        \
	    for (i = 0; i < wid; i++) {                         \
		LOAD_SRC_##CH(DTYPE); {                         \
		    ADD_##CH##AA(DTYPE);                        \
		}                                               \
	    }                                                   \
	}

#elif MLIB_VIS_EXPERIMENT >= 0x200

#define	LINE_FUNC_L(DTYPE, CH, AA)                              \
	static void                                             \
	mlib_ImageBlendMulti_##DTYPE##_##CH##AA                 \
	    (FUNC_ARG)                                          \
	{                                                       \
	    mlib_##DTYPE *sp = sl;                              \
	    mlib_##DTYPE *ap = al;                              \
	    FTYPE *buffs = _buffs;                              \
	    ATYPE *buffa = _buffa;                              \
	    mlib_s32 i;                                         \
	    mlib_##DTYPE *dp = dl;                              \
	                                                        \
	    DEF_##DTYPE;                                        \
	    vis_write_gsr32(0x10 << 3);                         \
	    for (i = 0; i < wid; i++) {                         \
		LOAD_SRC_##CH(DTYPE); {                         \
		    ADD_##CH##AA(DTYPE);                        \
		}                                               \
	    }                                                   \
	}

#else

#define	LINE_FUNC_L(DTYPE, CH, AA)                              \
	static void                                             \
	mlib_ImageBlendMulti_##DTYPE##_##CH##AA                 \
	    (FUNC_ARG)                                          \
	{                                                       \
	    mlib_##DTYPE *sp = sl;                              \
	    mlib_##DTYPE *ap = al;                              \
	    FTYPE *buffs = _buffs;                              \
	    ATYPE *buffa = _buffa;                              \
	    mlib_s32 i;                                         \
	    mlib_##DTYPE *dp = dl;                              \
	                                                        \
	    DEF_##DTYPE;                                        \
	    vis_write_gsr32(0x0f << 3);                         \
	    for (i = 0; i < wid; i++) {                         \
		LOAD_SRC_##CH(DTYPE); {                         \
		    ADD_##CH##AA(DTYPE);                        \
		}                                               \
	    }                                                   \
	}

#endif /* MLIB_VIS_EXPERIMENT */

/* *********************************************************** */

#define	LINE_FUNC_CH(DTYPE, CH)                                         \
	LINE_FUNC_A(DTYPE, CH, s) LINE_FUNC_A(DTYPE, CH, m)             \
	    LINE_FUNC_A(DTYPE, CH, x) LINE_FUNC_L(DTYPE, CH, sl)        \
	    LINE_FUNC_L(DTYPE, CH, ml) LINE_FUNC_L(DTYPE, CH, xl)

/* *********************************************************** */

#define	LINE_FUNC_DTYPE(DTYPE)                                  \
	LINE_FUNC_A(DTYPE, 1, s) LINE_FUNC_L(DTYPE, 1, sl)      \
	    LINE_FUNC_CH(DTYPE, 2) LINE_FUNC_CH(DTYPE, 3)       \
	    LINE_FUNC_CH(DTYPE, 4)                              \
	    func_type                                           \
		mlib_func_arr_##DTYPE[] = {                     \
		    mlib_ImageBlendMulti_##DTYPE##_1s,          \
		    mlib_ImageBlendMulti_##DTYPE##_2s,          \
		    mlib_ImageBlendMulti_##DTYPE##_3s,          \
		    mlib_ImageBlendMulti_##DTYPE##_4s, NULL,    \
		    mlib_ImageBlendMulti_##DTYPE##_2x,          \
		    mlib_ImageBlendMulti_##DTYPE##_3x,          \
		    mlib_ImageBlendMulti_##DTYPE##_4x, NULL,    \
		    mlib_ImageBlendMulti_##DTYPE##_2m,          \
		    mlib_ImageBlendMulti_##DTYPE##_3m,          \
		    mlib_ImageBlendMulti_##DTYPE##_4m,          \
		    mlib_ImageBlendMulti_##DTYPE##_1sl,         \
		    mlib_ImageBlendMulti_##DTYPE##_2sl,         \
		    mlib_ImageBlendMulti_##DTYPE##_3sl,         \
		    mlib_ImageBlendMulti_##DTYPE##_4sl, NULL,   \
		    mlib_ImageBlendMulti_##DTYPE##_2xl,         \
		    mlib_ImageBlendMulti_##DTYPE##_3xl,         \
		    mlib_ImageBlendMulti_##DTYPE##_4xl, NULL,   \
		    mlib_ImageBlendMulti_##DTYPE##_2ml,         \
		    mlib_ImageBlendMulti_##DTYPE##_3ml,         \
		    mlib_ImageBlendMulti_##DTYPE##_4ml,         \
	    }

/* *********************************************************** */

#undef  FTYPE
#define	FTYPE	mlib_f32
#undef  FP_SAT_TYPE
#define	FP_SAT_TYPE	mlib_f32

/* *********************************************************** */

#undef  FTYPE
#define	FTYPE	mlib_d64
#undef  FP_SAT_TYPE
#define	FP_SAT_TYPE	mlib_d64

LINE_FUNC_DTYPE(u8);
LINE_FUNC_DTYPE(x8);
LINE_FUNC_DTYPE(s16);
LINE_FUNC_DTYPE(u16);

/* *********************************************************** */

#undef  ATYPE
#define	ATYPE	FTYPE
LINE_FUNC_DTYPE(s32);
LINE_FUNC_DTYPE(d64);

/* *********************************************************** */

#undef  FTYPE
#define	FTYPE	mlib_f32
LINE_FUNC_DTYPE(f32);

/* *********************************************************** */

#define	MAX_N	16

/* *********************************************************** */

mlib_status
mlib_ImageBlendMulti_alltypes(
    mlib_image *dst,
    const mlib_image **srcs,
    const mlib_image **alphas,
    const void *bcolor,
    mlib_s32 n,
    mlib_s32 fp_flag)
{
	src_t src_arr[MAX_N], *src_a = src_arr;
	mlib_s32 dlb, xsize, ysize, nchan, bsize;
	mlib_s32 i, j, k, ind, shift;
	mlib_u8 *dl;
	mlib_type dtype;
	mlib_d64 *buffs, *buffa;
	void *c = (void *)bcolor;
	func_type *p_func, func_zero = NULL;

	MLIB_IMAGE_CHECK(dst);
	MLIB_IMAGE_GET_ALL_PARAMS(dst, dtype, nchan, xsize, ysize, dlb, dl);

	if (srcs == NULL || alphas == NULL) {
		return (MLIB_NULLPOINTER);
	}

	if (n < 1) {
		return (MLIB_FAILURE);
	}

	switch (dtype) {
	case MLIB_BYTE:
		p_func = (n < 5) ? mlib_func_arr_u8 : mlib_func_arr_x8;

		if (fp_flag)
			return (MLIB_FAILURE);
		break;
	case MLIB_SHORT:
		p_func = mlib_func_arr_s16;
		if (fp_flag)
			return (MLIB_FAILURE);
		break;
	case MLIB_USHORT:
		p_func = mlib_func_arr_u16;
		if (fp_flag)
			return (MLIB_FAILURE);
		break;
	case MLIB_INT:
		p_func = mlib_func_arr_s32;
		if (fp_flag)
			return (MLIB_FAILURE);
		break;
	case MLIB_FLOAT:
		p_func = mlib_func_arr_f32;
		if (!fp_flag)
			return (MLIB_FAILURE);
		break;
	case MLIB_DOUBLE:
		p_func = mlib_func_arr_d64;
		if (!fp_flag)
			return (MLIB_FAILURE);
		break;
	default:
		return (MLIB_FAILURE);
	}

	if (n > MAX_N) {
		src_a = __mlib_malloc(n * sizeof (src_t));
		if (src_a == NULL)
			return (MLIB_FAILURE);
	}

	for (k = 0; k < n; k++) {
		const mlib_image *src = srcs[k];
		const mlib_image *alp = alphas[k];

		MLIB_IMAGE_CHECK(src);
		MLIB_IMAGE_CHECK(alp);
		MLIB_IMAGE_FULL_EQUAL(dst, src);
		MLIB_IMAGE_TYPE_EQUAL(dst, alp);
		MLIB_IMAGE_SIZE_EQUAL(dst, alp);
		src_a[k].sl = mlib_ImageGetData(src);
		src_a[k].al = mlib_ImageGetData(alp);
		src_a[k].slb = mlib_ImageGetStride(src);
		src_a[k].alb = mlib_ImageGetStride(alp);
		src_a[k].achan = mlib_ImageGetChannels(alp);

		if (src_a[k].achan != 1 && src_a[k].achan != nchan)
			return (MLIB_FAILURE);
	}

	bsize = nchan * (xsize + 4);

	buffs = __mlib_malloc(2 * bsize * sizeof (mlib_d64));
	if (buffs == NULL)
		return (MLIB_FAILURE);
	buffa = buffs + bsize;

	for (i = 0; i < 2 * bsize; i++) {
		buffs[i] = 0;
	}

	ind = 0;
	for (k = 0; k < n; k++)
		ind |= src_a[k].achan;
	if (ind != 1)
		p_func += 4;

	p_func += (nchan - 1);

	if (dtype == MLIB_FLOAT || dtype == MLIB_DOUBLE) {
		func_zero =
		    mlib_func_arr_zero[(nchan - 1) + ((dtype ==
		    MLIB_DOUBLE) ? 4 : 0)];
		c = __mlib_malloc(bsize * sizeof (mlib_d64));
		if (c == NULL)
			return (MLIB_FAILURE);
	}

	for (shift = 0; (4 << shift) < n; shift++);

	for (j = 0; j < ysize; j++) {
		for (k = 0; k < n - 1; k++) {
			ind = (src_a[k].achan != 1) ? 4 : 0;
			p_func[ind] (NULL, src_a[k].sl, src_a[k].al, buffs,
			    buffa, c, shift, xsize);
		}

		ind = (src_a[k].achan != 1) ? 4 : 0;
		p_func[ind + 12] (dl, src_a[k].sl, src_a[k].al, buffs, buffa, c,
		    shift, xsize);

		if (func_zero != NULL) {
			func_zero(dl, NULL, NULL, NULL, c, bcolor, shift,
			    xsize);
		}

		dl += dlb;
		for (k = 0; k < n; k++) {
			src_a[k].sl = (mlib_u8 *)src_a[k].sl + src_a[k].slb;
			src_a[k].al = (mlib_u8 *)src_a[k].al + src_a[k].alb;
		}
	}

	if (c != bcolor)
		__mlib_free(c);
	__mlib_free(buffs);

	if (src_a != src_arr)
		__mlib_free(src_a);

	return (MLIB_SUCCESS);
}

/* *********************************************************** */

mlib_status
__mlib_ImageBlendMulti(
    mlib_image *dst,
    const mlib_image **srcs,
    const mlib_image **alphas,
    const mlib_s32 *c,
    mlib_s32 n)
{
	return (mlib_ImageBlendMulti_alltypes(dst, srcs, alphas, c, n, 0));
}

/* *********************************************************** */

mlib_status
__mlib_ImageBlendMulti_Fp(
    mlib_image *dst,
    const mlib_image **srcs,
    const mlib_image **alphas,
    const mlib_d64 *c,
    mlib_s32 n)
{
	return (mlib_ImageBlendMulti_alltypes(dst, srcs, alphas, c, n, 1));
}

/* *********************************************************** */
