/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident	"@(#)k_cexpl.c	1.4	06/01/31 SMI"

/* INDENT OFF */
/*
 * long double __k_cexpl(long double x, int *n);
 * Returns the exponential of x in the form of 2**n * y, y=__k_cexpl(x,&n).
 *
 *	1. Argument Reduction: given the input x, find r and integer k
 *	   and j such that
 *	             x = (32k+j)*ln2 + r,  |r| <= (1/64)*ln2 .
 *
 *	2. expl(x) = 2^k * (2^(j/32) + 2^(j/32)*expm1(r))
 *	   Note:
 *	   a. expm1(r) = (2r)/(2-R), R = r - r^2*(t1 + t2*r^2)
 *	   b. 2^(j/32) is represented as
 *			exp2_32_hi[j]+exp2_32_lo[j]
 *         where
 *              exp2_32_hi[j] = 2^(j/32) rounded
 *		exp2_32_lo[j] = 2^(j/32) - exp2_32_hi[j].
 *
 * Special cases:
 *	expl(INF) is INF, expl(NaN) is NaN;
 *	expl(-INF)=  0;
 *	for finite argument, only expl(0)=1 is exact.
 *
 * Accuracy:
 *	according to an error analysis, the error is always less than
 *	an ulp (unit in the last place).
 *
 * Misc. info.
 *      When |x| is really big, say |x| > 1000000, the accuracy
 *      is not important because the ultimate result will over or under
 *      flow. So we will simply replace n = 1000000 and r = 0.0. For
 *      moderate size x, according to an error analysis, the error is
 *      always less than 1 ulp (unit in the last place).
 *
 * Constants:
 * Only decimal values are given. We assume that the compiler will convert
 * from decimal to binary accurately enough to produce the correct
 * hexadecimal values.
 */
/* INDENT ON */

#include "libm.h"		/* __k_cexpl */
#include "complex_wrapper.h"	/* HI_XWORD */

/* INDENT OFF */
/* ln2/32 = 0.0216608493924982909192885037955680177523593791987579766912713 */
#if defined(__i386)
static const long double
			/* 43 significant bits, 21 trailing zeros */
ln2_32hi = 2.166084939249657281834515742957592010498046875e-2L,
ln2_32lo = 1.7181009433463659920976473789104487579766912713e-15L;
static const long double exp2_32_hi[] = {  	/* exp2_32[j] = 2^(j/32) */
	1.0000000000000000000000000e+00L,
	1.0218971486541166782081522e+00L,
	1.0442737824274138402382006e+00L,
	1.0671404006768236181297224e+00L,
	1.0905077326652576591003302e+00L,
	1.1143867425958925362894369e+00L,
	1.1387886347566916536971221e+00L,
	1.1637248587775775137938619e+00L,
	1.1892071150027210666875674e+00L,
	1.2152473599804688780476325e+00L,
	1.2418578120734840485256747e+00L,
	1.2690509571917332224885722e+00L,
	1.2968395546510096659215822e+00L,
	1.3252366431597412945939118e+00L,
	1.3542555469368927282668852e+00L,
	1.3839098819638319548151403e+00L,
	1.4142135623730950487637881e+00L,
	1.4451808069770466200253470e+00L,
	1.4768261459394993113155431e+00L,
	1.5091644275934227397133885e+00L,
	1.5422108254079408235859630e+00L,
	1.5759808451078864864006862e+00L,
	1.6104903319492543080837174e+00L,
	1.6457554781539648445110730e+00L,
	1.6817928305074290860378350e+00L,
	1.7186192981224779156032914e+00L,
	1.7562521603732994831094730e+00L,
	1.7947090750031071864148413e+00L,
	1.8340080864093424633989166e+00L,
	1.8741676341102999013002103e+00L,
	1.9152065613971472938202589e+00L,
	1.9571441241754002689657438e+00L,
};
static const long double exp2_32_lo[] = {
	0.0000000000000000000000000e+00L,
	2.6327965667180882569382524e-20L,
	8.3765863521895191129661899e-20L,
	3.9798705777454504249209575e-20L,
	1.0668046596651558640993042e-19L,
	1.9376009847285360448117114e-20L,
	6.7081819456112953751277576e-21L,
	1.9711680502629186462729727e-20L,
	2.9932584438449523689104569e-20L,
	6.8887754153039109411061914e-20L,
	6.8002718741225378942847820e-20L,
	6.5846917376975403439742349e-20L,
	1.2171958727511372194876001e-20L,
	3.5625253228704087115438260e-20L,
	3.1129551559077560956309179e-20L,
	5.7519192396164779846216492e-20L,
	3.7900651177865141593101239e-20L,
	1.1659262405698741798080115e-20L,
	7.1364385105284695967172478e-20L,
	5.2631003710812203588788949e-20L,
	2.6328853788732632868460580e-20L,
	5.4583950085438242788190141e-20L,
	9.5803254376938269960718656e-20L,
	7.6837733983874245823512279e-21L,
	2.4415965910835093824202087e-20L,
	2.6052966871016580981769728e-20L,
	2.6876456344632553875309579e-21L,
	1.2861930155613700201703279e-20L,
	8.8166633394037485606572294e-20L,
	2.9788615389580190940837037e-20L,
	5.2352341619805098677422139e-20L,
	5.2578463064010463732242363e-20L,
};
#else	/* sparc */
static const long double
			/* 0x3FF962E4 2FEFA39E F35793C7 00000000 */
ln2_32hi = 2.166084939249829091928849858592451515688e-2L,
ln2_32lo = 5.209643502595475652782654157501186731779e-27L;
static const long double exp2_32_hi[] = {  	/* exp2_32[j] = 2^(j/32) */
	1.000000000000000000000000000000000000000e+0000L,
	1.021897148654116678234480134783299439782e+0000L,
	1.044273782427413840321966478739929008785e+0000L,
	1.067140400676823618169521120992809162607e+0000L,
	1.090507732665257659207010655760707978993e+0000L,
	1.114386742595892536308812956919603067800e+0000L,
	1.138788634756691653703830283841511254720e+0000L,
	1.163724858777577513813573599092185312343e+0000L,
	1.189207115002721066717499970560475915293e+0000L,
	1.215247359980468878116520251338798457624e+0000L,
	1.241857812073484048593677468726595605511e+0000L,
	1.269050957191733222554419081032338004715e+0000L,
	1.296839554651009665933754117792451159835e+0000L,
	1.325236643159741294629537095498721674113e+0000L,
	1.354255546936892728298014740140702804343e+0000L,
	1.383909881963831954872659527265192818002e+0000L,
	1.414213562373095048801688724209698078570e+0000L,
	1.445180806977046620037006241471670905678e+0000L,
	1.476826145939499311386907480374049923924e+0000L,
	1.509164427593422739766019551033193531420e+0000L,
	1.542210825407940823612291862090734841307e+0000L,
	1.575980845107886486455270160181905008906e+0000L,
	1.610490331949254308179520667357400583459e+0000L,
	1.645755478153964844518756724725822445667e+0000L,
	1.681792830507429086062250952466429790080e+0000L,
	1.718619298122477915629344376456312504516e+0000L,
	1.756252160373299483112160619375313221294e+0000L,
	1.794709075003107186427703242127781814354e+0000L,
	1.834008086409342463487083189588288856077e+0000L,
	1.874167634110299901329998949954446534439e+0000L,
	1.915206561397147293872611270295830887850e+0000L,
	1.957144124175400269018322251626871491190e+0000L,
};

static const long double exp2_32_lo[] = {
	+0.000000000000000000000000000000000000000e+0000L,
	+1.805067874203309547455733330545737864651e-0035L,
	-9.374520292280427421957567419730832143843e-0035L,
	-1.596968447292758770712909630231499971233e-0035L,
	+9.112493410125022978511686101672486662119e-0035L,
	-6.504228206978548287230374775259388710985e-0035L,
	-8.148468844525851137325691767488155323605e-0035L,
	-5.066214576721800313372330745142903350963e-0035L,
	-1.359830974688816973749875638245919118924e-0035L,
	+9.497427635563196470307710566433246597109e-0035L,
	-3.283170523176998601615065965333915261932e-0036L,
	-5.017235709387190410290186530458428950862e-0035L,
	-2.391474797689109171622834301602640139258e-0035L,
	-8.350571357633908815298890737944083853080e-0036L,
	+7.036756889073265042421737190671876440729e-0035L,
	-5.182484853064646457536893018566956189817e-0035L,
	+9.422242548621832065692116736394064879758e-0035L,
	-3.967500825398862309167306130216418281103e-0035L,
	+7.143528991563300614523273615092767243521e-0035L,
	+1.159871252867985124246517834100444327747e-0035L,
	+4.696933478358115495309739213201874466685e-0035L,
	-3.386513175995004710799241984999819165197e-0035L,
	-8.587318774298247068868655935103874453522e-0035L,
	-9.605951548749350503185499362246069088835e-0035L,
	+9.609733932128012784507558697141785813655e-0035L,
	+6.378397921440028439244761449780848545957e-0035L,
	+7.792430785695864249456461125169277701177e-0035L,
	+7.361337767588456524131930836633932195088e-0035L,
	-6.472995147913347230035214575612170525266e-0035L,
	+8.587474417953698694278798062295229624207e-0035L,
	+2.371815422825174835691651228302690977951e-0035L,
	-3.026891682096118773004597373421900314256e-0037L,
};
#endif

static const long double
	one = 1.0L,
	two = 2.0L,
	ln2_64 = 1.083042469624914545964425189778400898568e-2L,
	invln2_32 = 4.616624130844682903551758979206054839765e+1L;

/* rational approximation coeffs for [-(ln2)/64,(ln2)/64] */
static const long double
	t1 =  1.666666666666666666666666666660876387437e-1L,
	t2 = -2.777777777777777777777707812093173478756e-3L,
	t3 =  6.613756613756613482074280932874221202424e-5L,
	t4 = -1.653439153392139954169609822742235851120e-6L,
	t5 =  4.175314851769539751387852116610973796053e-8L;
/* INDENT ON */

long double
__k_cexpl(long double x, int *n) {
	int hx, ix, j, k;
	long double t, r;

	*n = 0;
	hx = HI_XWORD(x);
	ix = hx & 0x7fffffff;
	if (hx >= 0x7fff0000)
		return (x + x);	/* NaN of +inf */
	if (((unsigned) hx) >= 0xffff0000)
		return (-one / x);	/* NaN or -inf */
	if (ix < 0x3fc30000)
		return (one + x);	/* |x|<2^-60 */
	if (hx > 0) {
		if (hx > 0x401086a0) {	/* x > 200000 */
			*n = 200000;
			return (one);
		}
		k = (int) (invln2_32 * (x + ln2_64));
	} else {
		if (ix > 0x401086a0) {	/* x < -200000 */
			*n = -200000;
			return (one);
		}
		k = (int) (invln2_32 * (x - ln2_64));
	}
	j = k & 0x1f;
	*n = k >> 5;
	t = (long double) k;
	x = (x - t * ln2_32hi) - t * ln2_32lo;
	t = x * x;
	r = (x - t * (t1 + t * (t2 + t * (t3 + t * (t4 + t * t5))))) - two;
	x = exp2_32_hi[j] - ((exp2_32_hi[j] * (x + x)) / r - exp2_32_lo[j]);
	k >>= 5;
	if (k > 240) {
		XFSCALE(x, 240);
		*n -= 240;
	} else if (k > 0) {
		XFSCALE(x, k);
		*n = 0;
	}
	return (x);
}
