/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident	"@(#)tanl.c	1.7	06/01/31 SMI"

/*
 * tanl(x)
 * Table look-up algorithm by K.C. Ng, November, 1989.
 *
 * kernel function:
 *	__k_tanl	... tangent function on [-pi/4,pi/4]
 *	__rem_pio2l	... argument reduction routine
 *
 * Method.
 *      Let S and C denote the sin and cos respectively on [-PI/4, +PI/4].
 *      1. Assume the argument x is reduced to y1+y2 = x-k*pi/2 in
 *	   [-pi/2 , +pi/2], and let n = k mod 4.
 *	2. Let S=S(y1+y2), C=C(y1+y2). Depending on n, we have
 *
 *          n        sin(x)      cos(x)        tan(x)
 *     ----------------------------------------------------------
 *	    0	       S	   C		 S/C
 *	    1	       C	  -S		-C/S
 *	    2	      -S	  -C		 S/C
 *	    3	      -C	   S		-C/S
 *     ----------------------------------------------------------
 *
 * Special cases:
 *      Let trig be any of sin, cos, or tan.
 *      trig(+-INF)  is NaN, with signals;
 *      trig(NaN)    is that NaN;
 *
 * Accuracy:
 *	computer TRIG(x) returns trig(x) nearly rounded.
 */

#pragma weak tanl = __tanl

#include "libm.h"
#include "longdouble.h"

long double
tanl(long double x) {
	long double y[2], z = 0.0L;
	int n, ix;

	ix = *(int *) &x;		/* High word of x */
	ix &= 0x7fffffff;
	if (ix <= 0x3ffe9220)		/* |x| ~< pi/4 */
		return (__k_tanl(x, z, 0));
	else if (ix >= 0x7fff0000)	/* trig(Inf or NaN) is NaN */
		return (x - x);
	else {				/* argument reduction needed */
		n = __rem_pio2l(x, y);
		return (__k_tanl(y[0], y[1], (n & 1)));
	}
}
