/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.beanutils.MethodUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.text.StrSubstitutor;
import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.impl.RegexProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.impl.SimpleProxyMapping;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.yaml.snakeyaml.TypeDescription;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.BaseConstructor;
import org.yaml.snakeyaml.constructor.Constructor;

/**
 * Utility to load reverse proxy mapping configurations in YAML format.
 * This utility uses <a href="https://code.google.com/p/snakeyaml/">snakeyaml</a> library internally
 * to parse YAML configurations.
 */
public class YamlConfigUtils
{

    private static Logger log = LoggerFactory.getLogger(YamlConfigUtils.class);

    /**
     * Loads YAML objects from the <code>yamlConfig</code>.
     * <p>
     * The <code>yamlConfig</code> can represent any of the following:
     * <ul>
     * <li>a file resource URI starting with "file:",</li>
     * <li>a classpath resource starting with "classpath:",</li>
     * <li>a servlet context relative path resource starting with "/",</li>
     * <li>otherwise, <code>yamlConfig</code> is regarded as a whole YAML configuration string.</li>
     * </ul>
     * </p>
     * <p>
     * <em>Note:</em> if <code>yamlConfig</code> represents an external resource (file resource URI, classpath resource or
     * or servlet context relative path resource), then the resource must be encoded in UTF-8.
     * </p>
     * 
     * @param constructor snakeyaml base constructor to be used when constructing a {@link Yaml}
     * @param yamlConfig YAML configuration resource or string
     * @param servletContextOrPortletContext either a {@link javax.servlet.ServletContext} or {@link javax.portlet.PortletContext} instance.
     * @return an iterable of the parsed YAML objects from the <code>yamlConfig</code>.
     */
    public static Iterable<Object> loadYaml(BaseConstructor constructor, String yamlConfig, Object servletContextOrPortletContext)
    {
        Iterable<Object> allObjects = null;

        String mappingYaml = null;
        InputStream input = null;

        final String mappings = StrSubstitutor.replaceSystemProperties(yamlConfig);
        final String trimmedMappings = StringUtils.trim(mappings);

        if (StringUtils.isNotEmpty(trimmedMappings))
        {
            try
            {
                if (StringUtils.startsWith(trimmedMappings, "file:"))
                {
                    input = new FileInputStream(new File(URI.create(trimmedMappings)));
                }
                else if (StringUtils.startsWith(trimmedMappings, "classpath:"))
                {
                    input = Thread.currentThread().getContextClassLoader().getResourceAsStream(trimmedMappings.substring(10));
                }
                else if (StringUtils.startsWith(trimmedMappings, "/"))
                {
                    input = (InputStream) MethodUtils.invokeMethod(servletContextOrPortletContext, "getResourceAsStream", trimmedMappings);
                }
                else
                {
                    mappingYaml = mappings;
                }

                if (mappingYaml == null && input != null)
                {
                    mappingYaml = IOUtils.toString(input, "UTF-8");
                }

                if (mappingYaml != null)
                {
                    Yaml yaml = new Yaml(constructor);
                    allObjects = yaml.loadAll(mappingYaml);
                }
            }
            catch (Exception e)
            {
                log.error("Failed to load yaml configuration at '{}'. {}", yamlConfig, e);
            }
            finally
            {
                IOUtils.closeQuietly(input);
            }
        }

        return allObjects;
    }

    /**
     * Loads list of {@link ProxyMapping} objects from the <code>yamlConfig</code>.
     * @param yamlConfig YAML configuration resource or string
     * @param servletContextOrPortletContext either a {@link javax.servlet.ServletContext} or {@link javax.portlet.PortletContext} instance.
     * @return a list of {@link ProxyMapping} objects from the <code>yamlConfig</code>.
     */
    public static List<ProxyMapping> loadProxyMappings(String yamlConfig, Object servletContextOrPortletContext)
    {
        List<ProxyMapping> proxyMappings = new ArrayList<ProxyMapping>();

        Constructor constructor = new Constructor();
        constructor.addTypeDescription(new TypeDescription(SimpleProxyMapping.class, "!simple"));
        constructor.addTypeDescription(new TypeDescription(RegexProxyMapping.class, "!regex"));

        Iterable<Object> allObjects = loadYaml(constructor, yamlConfig, servletContextOrPortletContext);

        if (allObjects != null)
        {
            for (Object item : allObjects)
            {
                if (item instanceof Collection)
                {
                    for (Object mapping : (Collection<?>) item)
                    {
                        if (mapping instanceof ProxyMapping)
                        {
                            proxyMappings.add((ProxyMapping) mapping);
                        }
                        else
                        {
                            log.error("Invalid mapping type: {} - {}" + mapping.getClass(), mapping);
                        }
                    }
                }
                else if (item instanceof ProxyMapping)
                {
                    proxyMappings.add((ProxyMapping) item);
                }
                else
                {
                    log.error("Invalid mapping type: {} - {}" + item.getClass(), item);
                }
            }
        }

        return proxyMappings;
    }

    private YamlConfigUtils()
    {
    }
}
