/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.command;

import java.io.IOException;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.chain.Command;
import org.apache.commons.lang.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.protocol.HTTP;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;
import org.apache.portals.applications.webcontent2.proxy.util.RequestUtils;

/**
 * {@link Command} responsible for copying all the original HTTP response headers
 * except of HTTP Cookie related headers
 * from the remote content to the internally created {@link HttpResponse} instance.
 * <p>
 * <em>Note:</em> All the HTTP Cookie related headers are cared by {@link AddCookiesToResponseCommand}.
 * </p>
 * <p>
 * Also, the following HTTP headers are excluded during the copy:
 * <ul>
 * <li><code>Content-Length</code></li>
 * <li><code>Content-Encoding</code></li>
 * </ul>
 * </p>
 * <p>
 * Meanwhile, the following HTTP header(s) are rewritten:
 * <ul>
 * <li><code>Host</code></li>
 * </ul>
 * </p>
 */
public class AddHeadersToResponseCommand extends AbstractProxyCommand
{

    /**
     * Custom specific "Host" HTTP header value.
     * If this is null, then it is resolved at runtime
     * by using {@link ServletRequest#getServerName()} and {@link ServletRequest#getServerPort()}.
     */
    private String hostHeaderValue;

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean executeInternal(final ProxyContext context) throws ReverseProxyException, IOException
    {
        HttpResponse httpResponse = context.getHttpResponse();

        int statusCode = httpResponse.getStatusLine().getStatusCode();

        // Check if the proxy response is a redirect
        if (statusCode >= HttpServletResponse.SC_MULTIPLE_CHOICES /* 300 */
            && statusCode <= HttpServletResponse.SC_NOT_MODIFIED /* 304 */)
        {
            return false;
        }

        // Pass the response code back to the client
        context.getRequestContext().setStatus(statusCode);

        HttpEntity httpEntity = httpResponse.getEntity();

        if (httpEntity != null)
        {
            // Pass response headers back to the client
            Header [] headerArrayResponse = httpResponse.getAllHeaders();

            for (Header header : headerArrayResponse)
            {
                String headerName = header.getName();

                if (StringUtils.startsWithIgnoreCase(headerName, "Set-Cookie"))
                {
                    continue;
                }

                if (StringUtils.equalsIgnoreCase(headerName, HTTP.CONTENT_LEN) || StringUtils.equalsIgnoreCase(headerName, HTTP.CONTENT_ENCODING))
                {
                    if (context.getContentRewriter() != null || RequestUtils.isDispatched(context.getRequestContext()))
                    {
                        continue;
                    }
                }

                String headerValue = header.getValue();

                if (StringUtils.equalsIgnoreCase(headerName, HTTP.TARGET_HOST))
                {
                    String host = getHostHeaderValue();

                    if (host == null)
                    {
                        if (context.getRequestContext().getServerPort() == 80)
                        {
                            host = context.getRequestContext().getServerName();
                        }
                        else
                        {
                            host = context.getRequestContext().getServerName() + ":" + context.getRequestContext().getServerPort();
                        }
                    }

                    context.getRequestContext().setHeader(headerName, host);
                }
                else
                {
                    context.getRequestContext().setHeader(headerName, headerValue);
                }
            }
        }

        return false;
    }

    /**
     * Returns the custom specific "Host" HTTP header value.
     * If this returns null, then it is resolved at runtime
     * by using {@link ServletRequest#getServerName()} and {@link ServletRequest#getServerPort()}.
     * @return
     */
    public String getHostHeaderValue()
    {
        return hostHeaderValue;
    }

    /**
     * Sets the custom specific "Host" HTTP header value.
     * If this is set to a non-null value, then it doesn't try to resolve the "Host" HTTP header value
     * at runtime, but just returns the custom specific header value instead.
     * @param hostHeaderValue
     */
    public void setHostHeaderValue(String hostHeaderValue)
    {
        this.hostHeaderValue = hostHeaderValue;
    }
}
