/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy;

import java.net.URI;
import java.util.Map;

import org.apache.portals.applications.webcontent2.rewriter.ContentRewriter;

/**
 * <p>Reverse proxy mapping abstraction, which is responsible for</p>
 * <ul>
 * <li>checking whether or not a local request path info is matched by this proxy mapping,</li>
 * <li>resolving a local request path info to a remote target URI,</li>
 * <li>checking whether or not a remote target URI is matched by this proxy mapping,</li>
 * <li>resolving a remote target URI to a local request path info,</li>
 * <li>finding a proper {@link ContentRewriter} instance for the MIME type of the response content if any available,</li>
 * <li>storing/retrieving/setting ad hoc attributes.</li>
 * </ul>
 * <p>
 * <em>Note:</em> {@link #resolveRemoteFromLocal(String)} is usually invoked to resolve
 * a remote target URI from a local request path info, whereas {@link #resolveLocalFromRemote(URI)}
 * is usually invoked to resolve a local request path info from a remote target URI location
 * (e.g, URI redirection response from the remote server).
 * </p>
 * <p>
 * <em>Note:</em> It is recommended to invoke {@link #resolveRemoteFromLocal(String)}
 * only when {@link #matchesLocal(String)} returns true.
 * In the same way, it is recommended to invoke {@link #resolveLocalFromRemote(URI)}
 * only when {@link #matchesRemote(URI)} returns true.
 * </p>
 */
public interface ProxyMapping
{

    /**
     * Returns true the <code>localPath</code> can be mapped by this proxy mapping.
     * Otherwise, it should return false.
     * @param localPath
     * @return
     */
    public boolean matchesLocal(String localPath);

    /**
     * Resolves the <code>localPath</code> to a remote target URI.
     * @param localPath
     * @return
     */
    public String resolveRemoteFromLocal(String localPath);

    /**
     * Returns true the <code>remoteURI</code> can be mapped by this proxy mapping.
     * Otherwise, it should return false.
     * @param remoteURI
     * @return
     */
    public boolean matchesRemote(URI remoteURI);

    /**
     * Resolves the <code>remoteURI</code> to a local request path info.
     * @param remoteURI
     * @return
     */
    public String resolveLocalFromRemote(URI remoteURI);

    /**
     * Returns a {@link ContentRewriter} instance to be used for the given <code>mimeType</code>.
     * @param mimeType
     * @return
     */
    public ContentRewriter getContentRewriter(String mimeType);

    /**
     * Returns the value of the named attribute as an <code>Object</code>,
     * or <code>null</code> if no attribute of the given name exists. 
     * @param name
     * @return
     */
    public Object getAttribute(String name);

    /**
     * Stores an attribute in this mapping.
     * @param name
     * @param value
     */
    public void setAttribute(String name, Object value);

    /**
     * Removes an attribute from this mapping.
     * @param name
     */
    public void removeAttribute(String name);

    /**
     * Returns a <code>Map</code> containing the attributes available. 
     * This method returns an empty <code>Map</code>
     * if there's no attributes available to it.
     * @return
     */
    public Map<String, Object> getAttributesMap();

}
