/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.rewriter.htmlcleaner;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.htmlcleaner.CommentNode;
import org.htmlcleaner.HtmlNode;
import org.htmlcleaner.TagNode;
import org.htmlcleaner.TagNodeVisitor;

/**
 * A generic abstract {@link TagNodeVisitor} class
 * which can be extended simply by implementing the link rewriting method only
 * ({@link #rewriteURI(String, String, String)}).
 * <p>
 * This abstract class is provided to help developers implement {@link TagNodeVisitor}s
 * more easily with a simple link rewriting capability which is very common in
 * reverse proxy modules.
 * </p>
 * <p>
 * This class is constructed with a map constructor argument, <code>Map&lt;String,String[]&gt; tagNameAndLinkAttrs</code>,
 * which should contain entries of tag name and link attribute names array of the tag.
 * For example, an HTML &lt;a /&gt; tag typically may contain an <code>href</code> attribute for
 * the link information. e.g, &lt;a href="http://portals.apache.org">Apache Portals&lt;/a&gt;.
 * If you want to rewrite the link string contained in the <code>href</code> attribute of the
 * &lt;a /&gt; tag, then the map constructor argument should contain a key-value pair entry (the tag name
 * : link attribute names array):  { "a" :  [ "href" ] }.
 * Then this class will invoke {@link #rewriteURI(String, String, String)} for each link attribute
 * whenever it visits a &lt;a /&gt; tag.
 * </p>
 */
public abstract class AbstractProxyTagNodeVisitor implements TagNodeVisitor
{

    private Map<String, String[]> tagNameAndLinkAttrs;

    /**
     * Construct a AbstractProxyTagNodeVisitor from the map of
     * the link element tag name (e.g, "a" or "img") and its link attributes (e.g, [ "href" ] or [ "src" ]).
     * @param tagNameAndLinkAttrs
     */
    public AbstractProxyTagNodeVisitor(Map<String, String[]> tagNameAndLinkAttrs)
    {
        this.tagNameAndLinkAttrs = new HashMap<String, String[]>();

        if (tagNameAndLinkAttrs != null)
        {
            this.tagNameAndLinkAttrs.putAll(tagNameAndLinkAttrs);
        }
    }

    /**
     * {@inheritDoc}
     * <p>
     * This method checks the type of the current <code>htmlNode</code>
     * and it invokes {@link #visitTagNode(TagNode, TagNode)} if the current <code>htmlNode</code>
     * is instance of {@link TagNode}, or invokes {@link #visitCommentNode(TagNode, CommentNode)}
     * if the current <code>htmlNode</code> is instance of {@link CommentNode}.
     * </p>
     */
    public boolean visit(TagNode parentNode, HtmlNode htmlNode)
    {
        if (htmlNode instanceof TagNode)
        {
            return visitTagNode(parentNode, (TagNode) htmlNode);
        }
        else if (htmlNode instanceof CommentNode)
        {
            return visitCommentNode(parentNode, (CommentNode) htmlNode);
        }

        return true;
    }

    /**
     * Default implementation of handling visited <code>TagNode</code>s.
     * <p>
     * This method checks if the tag name was registered to be handled to rewrite the link attributes.
     * If it is registered through the constructor, then it retrieves the attribute names array by the
     * tag name. For each attribute, it invokes {@link #rewriteURI(String, String, String)} method
     * to translate the link URI.
     * If {@link #rewriteURI(String, String, String)} returns a different string than the original
     * link URI string, then it updates the link attributes by the newly rewritten link URI string.
     * </p>
     * @param parentNode
     * @param tag
     * @return
     */
    protected boolean visitTagNode(TagNode parentNode, TagNode tag)
    {
        String tagName = tag.getName();
        String[] linkAttrNames = tagNameAndLinkAttrs.get(tagName);

        if (ArrayUtils.isEmpty(linkAttrNames))
        {
            return true;
        }

        String link;
        String rewrittenLink;

        for (String linkAttrName : linkAttrNames)
        {
            if (StringUtils.isNotEmpty(linkAttrName))
            {
                link = tag.getAttributeByName(linkAttrName);

                if (StringUtils.isNotEmpty(link))
                {
                    rewrittenLink = rewriteURI(tagName, linkAttrName, link);

                    if (!StringUtils.equals(link, rewrittenLink))
                    {
                        tag.addAttribute(linkAttrName, rewrittenLink);
                    }
                }
            }
        }

        return true;
    }

    /**
     * Default implementation of handling visited <code>CommentNode</code>s.
     * <p>
     * This does nothing but simply returns true by default.
     * </p>
     * @param parentNode
     * @param comment
     * @return
     */
    protected boolean visitCommentNode(TagNode parentNode, CommentNode comment)
    {
        return true;
    }

    /**
     * The method which should rewrite the original link URI string to a new link URI to return if necessary.
     * @param tagName the tag name holding the link
     * @param attrName the attribute name of the tag node holding the link
     * @param uri the original link URI string
     * @return newly rewritten link URI string if necessary, or the original link URI string if rewriting is unnecessary.
     */
    protected abstract String rewriteURI(String tagName, String attrName, String uri);

}
