/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent.rewriter;

import java.io.Reader;
import java.io.Writer;
import java.net.URL;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * AbstractRewriter
 *
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: AbstractRewriter.java 891414 2009-12-16 20:19:02Z rwatler $
 */
public abstract class AbstractRewriter implements Rewriter
{
    protected final static Logger log = LoggerFactory.getLogger(AbstractRewriter.class);
    
    private String baseUrl = null;
    private boolean useProxy = false; 
        
    public void parse(ParserAdaptor adaptor, Reader reader) throws RewriterException
    {
        adaptor.parse(this, reader);        
    }
    
    public void rewrite(ParserAdaptor adaptor, Reader reader, Writer writer) throws RewriterException
    {
        adaptor.rewrite(this, reader, writer);        
    }
    
    public abstract String rewriteUrl(String url, String tag, String attribute);
    
    public void setBaseUrl(String base)
    {
        this.baseUrl = base;
    }
    
    public String getBaseUrl()
    {
        return baseUrl;
    }
    public String getBaseRelativeUrl(String relativeUrl)
    {        
        try
        {
            String baseUrl = getBaseUrl();
            if (baseUrl != null)
            {
                URL xlate = new URL(new URL(baseUrl), relativeUrl);
                return xlate.toString();
            }
        }
        catch (Exception e)
        {
            log.error("Unable to translate URL relative to base URL", e);
        }
        return relativeUrl;
    }
    
    public boolean getUseProxy()
    {
        return useProxy;
    }
    
    public void setUseProxy(boolean useProxy)
    {
        this.useProxy = useProxy;        
    }
    
    public boolean enterSimpleTagEvent(String tag, MutableAttributes attrs)
    {
        return true;
    }
    
    public String exitSimpleTagEvent(String tag, MutableAttributes attrs)
    {
        return null;
    }
    
    public boolean enterStartTagEvent(String tag, MutableAttributes attrs)
    {
        return true;
    }
    
    public String exitStartTagEvent(String tag, MutableAttributes attrs)
    {
        return null;
    }
    
    public boolean enterEndTagEvent(String tag)
    {
        return true;
    }
    
    public String exitEndTagEvent(String tag)
    {
        return null;
    }
    
    public boolean enterText(char[] values, int param)
    {
        return true;
    }

    public void enterConvertTagEvent(String tag, MutableAttributes attrs)
    {
    }

    public String enterConvertTextEvent(String tag, String text)
    {
        return null;
    }
}
