/******************************************************************************
  (c) Copyright 2002-2006, 1060 Research Ltd                                   

  This Software is licensed to You, the licensee, for use under the terms of   
  the 1060 Public License v1.0. Please read and agree to the 1060 Public       
  License v1.0 [www.1060research.com/license] before using or redistributing   
  this software.                                                               

  In summary the 1060 Public license has the following conditions.             
  A. You may use the Software free of charge provided you agree to the terms   
  laid out in the 1060 Public License v1.0                                     
  B. You are only permitted to use the Software with components or applications
  that provide you with OSI Certified Open Source Code [www.opensource.org], or
  for which licensing has been approved by 1060 Research Limited.              
  You may write your own software for execution by this Software provided any  
  distribution of your software with this Software complies with terms set out 
  in section 2 of the 1060 Public License v1.0                                 
  C. You may redistribute the Software provided you comply with the terms of   
  the 1060 Public License v1.0 and that no warranty is implied or given.       
  D. If you find you are unable to comply with this license you may seek to    
  obtain an alternative license from 1060 Research Limited by contacting       
  license@1060research.com or by visiting www.1060research.com                 

  NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC  
  LICENSE V1.0 FOR DETAILS                                                     

  THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ     
  THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license      

  File:          $RCSfile: PingPongFrame.java,v $
  Version:       $Name:  $ $Revision: 1.12 $
  Last Modified: $Date: 2006/04/07 14:14:40 $
 *****************************************************************************/
package org.ten60.netkernel.pingpong.transport;

import org.ten60.netkernel.pingpong.representation.*;
import org.ten60.netkernel.layer1.nkf.*;
import org.ten60.netkernel.layer1.representation.ByteArrayAspect;

import com.ten60.netkernel.urii.IURRepresentation;
import com.ten60.netkernel.urii.aspect.BooleanAspect;
import com.ten60.netkernel.urii.aspect.IAspectReadableBinaryStream;
import com.ten60.netkernel.util.Utils;
import com.ten60.netkernel.util.Utils1_5;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import javax.swing.*;
import java.io.*;

/**
 * The main frame of the UI
 * @author  tab
 */
public class PingPongFrame extends JFrame implements MouseMotionListener
{
	private BufferStrategy mStrategy;
	private PingPongTransport mUI;
	private IURRepresentation mConstantsRep;
	private IAspectConstants mConstants;
	private int mWidth, mHeight;
	private int mBallRadius;
	private Canvas mCanvas;
	private ControlPanel mCP;
	
	private int mKey;
	private int mMouseX;
	private int mMouseY;
	private int mActualMouseX;
	private int mActualMouseY;
	private int mBatWidth;
	private int mBatHeight;
	private int mMaxBatMovement=16;
	
	private Color[] mColours;
	
	private RunnerThread mThread;
	private float mLastImpactVelocity;
	
	/** sound player */
	private SoundPlayer mSoundPlayer = new SoundPlayer();
	/** the  sound being played */
	private IAspectReadableBinaryStream mSound;
	
	
	/** Creates a new instance of PingPongFrame */
	public PingPongFrame(PingPongTransport aUI, Point aLocation) throws Exception
	{
		mUI=aUI;
		
		updateConstants();
		
		if (aLocation!=null)
		{	setLocation(aLocation);
		}
		
		setTitle("Ping Pong");
		mColours = new Color[32];
		for (int i=0; i<32; i++)
		{	mColours[i] = generateColour();
		}
		
		IAspectReadableBinaryStream s = (IAspectReadableBinaryStream)mUI.getRequestContext().sourceAspect("ffcpl:/pingpong/sound",IAspectReadableBinaryStream.class);
		ByteArrayOutputStream baos=new ByteArrayOutputStream(s.getContentLength());
		Utils.pipe(s.getInputStream(), baos);
		mSound=new ByteArrayAspect(baos);		
		
		final int width=mWidth;
		final int height=mHeight;
		mCanvas = new Canvas()
		{	public Dimension getPreferredSize()
			{	return new Dimension(mWidth, mHeight);
		    }
		};

		mCanvas.addMouseMotionListener(this);
		mCanvas.setIgnoreRepaint(true);

		// listen for size changes and update constants
		mCanvas.addComponentListener(new ComponentAdapter()
		{	public void componentResized(ComponentEvent aEvent)
			{	
				try
				{	mWidth=aEvent.getComponent().getWidth();
					mHeight=aEvent.getComponent().getHeight();
					ConstantsAspect aspect = new ConstantsAspect(mConstants);
					aspect.setValue(IAspectConstants.WALL_WIDTH, new Integer(mWidth));
					aspect.setValue(IAspectConstants.WALL_HEIGHT, new Integer(mHeight));
					getRequestContext().sinkAspect("ffcpl:/pingpong/constants/current",aspect);
					synchronized(this)
					{	mStrategy = null;
					}
					pack();
				} catch (NKFException e)
				{ e.printStackTrace();
				}
			}
		} );
		
		// listen for window closing
		this.addWindowListener(new WindowAdapter()
			{
				public void windowClosing(WindowEvent e)
				{	mThread.setStopped();
					try
					{	mUI.getRequestContext().sinkAspect("ffcpl:/pingpong/isVisible", new BooleanAspect(false));
					}
					catch (NKFException e2)
					{	System.out.println(e2.toString());
					}
				}
			} );
			
		mCP = new ControlPanel(this);
		getContentPane().add("North", mCP);
	    getContentPane().add("Center", mCanvas);
	    pack();		
		
		setVisible(true);
		
		mThread = new RunnerThread();
		mThread.start();
		
		
	}
	
	private static float[] hues = new float[]{ 0.708f, 0.125f, 0.0416f, 0.375f };
	
	private Color generateColour()
	{	int i = (int)(Math.random()*2);
		float r=(float)Math.random()+0.5f;
		float v=r;
		if (v>1.0f) v=1.0f;
		float s=2-r;
		if (s>1.0f) s=1.0f;
		
		Color c=new Color(Color.HSBtoRGB(hues[i],s,v));
		return c;
	}
	
	public INKFConvenienceHelper getRequestContext()
	{	return mUI.getRequestContext();
	}
	
	IAspectConstants getConstants()
	{	return mConstants;
	}
	
	public void stop()
	{	mThread.setStopped();
		setVisible(false);
		dispose();
	}
	
	private void updateConstants() throws NKFException
	{
		if (mConstantsRep==null || mConstantsRep.getMeta().isExpired())
		{
			INKFConvenienceHelper context = mUI.getRequestContext();
			mConstantsRep = context.source("ffcpl:/pingpong/constants/current",IAspectConstants.class);
			mConstants = (IAspectConstants)mConstantsRep.getAspect(IAspectConstants.class);

			int w=mConstants.getValue(IAspectConstants.WALL_WIDTH).intValue();
			int h=mConstants.getValue(IAspectConstants.WALL_HEIGHT).intValue();
			mBallRadius=mConstants.getValue(IAspectConstants.BALL_RADIUS).intValue();
			
			mWidth=w;
			mHeight=h;
			mStrategy = null;
			if (mCanvas!=null) mCanvas.setSize(w,h);
			pack();
			this.setSize(getPreferredSize());
			
			mBatWidth=mConstants.getValue(IAspectConstants.BAT_WIDTH).intValue();
			mBatHeight=mConstants.getValue(IAspectConstants.BAT_HEIGHT).intValue();
		}
	}
	
	public PingPongTransport getUI()
	{	return mUI;
	}
	
	public void mouseDragged(MouseEvent e)
	{
	}
	
	public void mouseMoved(MouseEvent e)
	{	mActualMouseX=e.getX();
		mActualMouseY=e.getY();
	}		
	
	private void updateUI() throws Exception
	{
		int dx = mActualMouseX-mMouseX;
		int dy = mActualMouseY-mMouseY;
		float d = (float)Math.sqrt(dx*dx+dy*dy);
		if (d>mMaxBatMovement)
		{	mMouseX+=dx/d*mMaxBatMovement;
			mMouseY+=dy/d*mMaxBatMovement;
		}
		else
		{	mMouseX=mActualMouseX;
			mMouseY=mActualMouseY;
		}
		// update model
		IAspectPoint bat = new PointAspect(mMouseX,mMouseY);	
		try
		{	getRequestContext().sinkAspect("ffcpl:/pingpong/bat", bat);
		}
		catch (NKFException ex)
		{	System.out.println(ex.toString());
		}
		
		updateConstants();
		
		IAspectPingPong gameState = (IAspectPingPong)getRequestContext().sourceAspect("ffcpl:/pingpong/balls",IAspectPingPong.class);
			
		BufferStrategy bs;
		if (mStrategy==null)
		{	mCanvas.createBufferStrategy(2);
			bs = mCanvas.getBufferStrategy();
			mStrategy=bs;
		}
		else
		{	bs=mStrategy;
		}
		Graphics g = bs.getDrawGraphics();
		
		if (mWidth!=mCanvas.getWidth())
		{	g.setColor(Color.LIGHT_GRAY);
			g.fillRect(mWidth,0, mCanvas.getWidth()-mWidth,mCanvas.getHeight());
		}
		g.setColor(Color.BLACK);
		g.fillRect(0,0, mWidth,mHeight);

		for (int i=gameState.getBallCount()-1; i>=0; i--)
		{	g.setColor(mColours[i%32]);
			g.fillOval(gameState.getBallPosition(i).x-mBallRadius, gameState.getBallPosition(i).y-mBallRadius, mBallRadius*2, mBallRadius*2);
		}
		g.setColor(Color.WHITE);
		g.fillRect(mMouseX-mBatWidth/2,mMouseY-mBatHeight/2, mBatWidth, mBatHeight);
		bs.show();
		Toolkit.getDefaultToolkit().sync(); //keeps the UI updated faster			
		
		
		// sound
		if (mUI.isAudible())
		{	float i=gameState.maxImpactVelocity();
			// make use of Hass effect to limit number of sounds played
			if (i>mLastImpactVelocity && i>0.5f)
			{	float db=-15/i;
				mSoundPlayer.playSound2(mSound,db);
				mLastImpactVelocity=i;
			}
			else
			{	mLastImpactVelocity*=0.95f;
			}
		}

	}
	
	
	private float mFloatTime=20.0f;
	private long mTime2;
	
	/**Thread to drive the model transitions*/
	private class RunnerThread extends Thread
	{	private boolean mStopped=false;
		private static final int PERIOD=33; // 33 millseconds - 30 frames/sec
		public void setStopped()
		{	mStopped=true;
			try
			{	join();
			}
			catch (InterruptedException e)
			{
			}
		}
		
		public void run()
		{	//long t;
			while (!mStopped)
			{	//t=System.currentTimeMillis();				
				try
				{	updateUI();
				} catch (NKFException e)
				{	System.out.println(e.toString());
				}
				catch (Exception e)
				{	e.printStackTrace();
				}				
				synchronizeFrame();
			}
		}
		
		/**Compute the delay needed to achieve a constant framerate
		 * Requires special case for Win32's poor time resolution and also
		 * uses Java1.5 nanosecond timing if present.
		 */
		private void synchronizeFrame()
		{	long t=0;
			
			if (File.separatorChar=='/')
			{	// linux and mac have good frame synchronization
				long t2=System.currentTimeMillis();
				t=t2-mTime2;
				mTime2=t2;
			}
			else
			{	if (Utils1_5.isSuitable())
				{	try
					{	java.lang.reflect.Method m=System.class.getMethod("nanoTime", new Class[0]);
						long t2=((Long)m.invoke(null, new Object[0])).longValue();
						t=(t2-mTime2)/1000000;
						mTime2=t2;
					} catch (Exception e)
					{	
						
					}
				}
				else
				{	// win32 timing without 1.5 is bad so use averaging
					long t2=System.currentTimeMillis();
					float diff=(float)(t2-mTime2);
					mTime2=t2;
					mFloatTime=mFloatTime*0.99f+diff*0.01f;
					t=(long)mFloatTime;
				}
			}
			
			long p=PERIOD-1-t;
			//System.out.println(p);
			if (p>0 && p<=PERIOD)
			{	try
				{	
					Thread.currentThread().sleep(p);
				} catch (Exception e)
				{	// ignored interruped exception
				}
			}			
			
		}
	}
	
}