/******************************************************************************
  (c) Copyright 2002-2006, 1060 Research Ltd                                   

  This Software is licensed to You, the licensee, for use under the terms of   
  the 1060 Public License v1.0. Please read and agree to the 1060 Public       
  License v1.0 [www.1060research.com/license] before using or redistributing   
  this software.                                                               

  In summary the 1060 Public license has the following conditions.             
  A. You may use the Software free of charge provided you agree to the terms   
  laid out in the 1060 Public License v1.0                                     
  B. You are only permitted to use the Software with components or applications
  that provide you with OSI Certified Open Source Code [www.opensource.org], or
  for which licensing has been approved by 1060 Research Limited.              
  You may write your own software for execution by this Software provided any  
  distribution of your software with this Software complies with terms set out 
  in section 2 of the 1060 Public License v1.0                                 
  C. You may redistribute the Software provided you comply with the terms of   
  the 1060 Public License v1.0 and that no warranty is implied or given.       
  D. If you find you are unable to comply with this license you may seek to    
  obtain an alternative license from 1060 Research Limited by contacting       
  license@1060research.com or by visiting www.1060research.com                 

  NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC  
  LICENSE V1.0 FOR DETAILS                                                     

  THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ     
  THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license      

  File:          $RCSfile$
  Version:       $Name$ $Revision$
  Last Modified: $Date$
 *****************************************************************************/
package org.ten60.netkernel.jarboot;

import java.io.*;
import java.net.URL;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.*;
import java.net.URLClassLoader;

import org.netkernel.layer0.boot.BootUtils;

/**
 * All-in-one jar bootloader, creates a classloader based on all nested jar files inside /lib/ directory
 * and then constructs InnerBoot (which should do the rest using the supplied jars)
 * @author  tab
 */
public class BootLoader
{
	private static String EXPAND_PREFIX="netkernel_";
	
	public static void main(String[] args) throws Exception
	{	
		String javaVersion=System.getProperty("java.version");
		String[] versionParts=javaVersion.split("\\.");
		int majorVersion = Integer.parseInt(versionParts[1]);
		if (majorVersion<8)
		{	System.err.println("NetKernel requires java version >= 1.8, currently ["+javaVersion+"]");
			return;
		}
		
		String classpath = System.getProperty("java.class.path");
		boolean inJar = classpath.endsWith(".jar") && classpath.indexOf(';')==-1;
		if (!inJar)
		{	System.err.println(BootLoader.class.getName()+" can only be run from inside jar");
			return;
		}
		int j=classpath.indexOf(File.pathSeparatorChar);
		if (j>0)
		{	classpath=classpath.substring(0,j);
		}		
				
		String libPrefix="lib/";
			
		File expandDir = null;
		String expandDirString = System.getProperty("netkernel.expand.dir");
		if (expandDirString!=null)
		{	expandDir = new File(expandDirString);
		}		
		
		//delete temp files on next instantiation - they are locked when we exit due to classloaders :-(
		if (expandDir==null)
		{	String tempPath = System.getProperty("java.io.tmpdir");
			File tempDir = new File(tempPath);
			File[] files = tempDir.listFiles(new FilenameFilter()
			{	public boolean accept(File aDir, String aName)
				{	return aName.startsWith(EXPAND_PREFIX);
				}
			} );
			for (int i=0; i<files.length; i++)
			{	files[i].delete();
			}
		}
		
		// expand all nested jars into temp dir
		JarFile jarFile = new JarFile(classpath);
	    Enumeration e = jarFile.entries();
		ArrayList urlList = new ArrayList();
	    while (e.hasMoreElements())
	    {
	        JarEntry entry = (JarEntry)e.nextElement();
	        if (entry.isDirectory()) continue;
	        String name = entry.getName();
	        if (!name.startsWith(libPrefix) || !name.endsWith(".jar")) continue;
	        name=name.substring(libPrefix.length(), name.length()-4);
	        InputStream is=jarFile.getInputStream(entry);
	        File expanded=expandJar(is,name,expandDir);
	        urlList.add(expanded);
	    }
		
		urlList.add(new File(classpath));
		URL[] urls = new URL[urlList.size()];
		for (int i=0; i<urlList.size(); i++)
		{	urls[i]=((File)urlList.get(i)).toURL();
		}
		URLClassLoader ucl = new BootClassLoader(urls,BootLoader.class.getClassLoader().getParent());
		Class c=ucl.loadClass("org.ten60.netkernel.jarboot.InnerBoot");
		c.newInstance();
	}
	
	public static File expandJar(InputStream aIs, String aName, File aExpandDir) throws IOException
	{
		String prefix = aName;
		StringBuffer sb = new StringBuffer(prefix.length());
		for (int n = 0; n < prefix.length(); n++)
		{
			char c = prefix.charAt(n);
			if (Character.isLetterOrDigit(c) || c=='-' || c=='.')
			{
				sb.append(c);
			}
			else
			{
				sb.append('_');
			}
		}
		sb.append("_");
		prefix = sb.toString();
		
		File dest;
		if (aExpandDir != null)
		{	dest = new File(aExpandDir, prefix);
		}
		else
		{	dest = File.createTempFile(EXPAND_PREFIX+prefix, ".jar");
			dest.deleteOnExit();
		}
		
		// TODO: need to only copy if dest doesn't exist or is stale
		System.out.println("Expanding " + aName ); //+ " to " + dest.getAbsolutePath());
		FileOutputStream fos = new FileOutputStream(dest);
		pipe(aIs, fos);
		return dest;
	}	
	
	/**
	 * copy an input stream to an outputsteam and close streams when finished
	 * @throws IOException if there are any problems
	 */
	public static void pipe(InputStream aInput, OutputStream aOutput) throws IOException
	{	byte b[] = new byte[256];
		int c;
		try
		{	while ((c = aInput.read(b)) > 0)
			{	aOutput.write(b, 0, c);
			}
		}
		finally
		{	try
			{	aInput.close();
			}
			finally
			{	aOutput.close();
			}
		}
	}
}
