/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: CellRevisionProviderScala.java
 * Written by: Dmitry Nadezhin, Sun Microsystems.
 *
 * Copyright (c) 2005, Oracle and/or its affiliates. All rights reserved.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.sun.electric.database

import com.sun.electric.util.TextUtils
import com.sun.electric.util.collections.ArrayIterator

private[database] object CellRevisionProviderScala {
  val emptyNodeList = new ImmutableNodeInstIterableS(ImmutableNodeInst.NULL_ARRAY)
}

private[database] class CellRevisionProviderScala extends CellRevisionProviderDefault {
  
  override def createCellRevision(c: ImmutableCell) = new CellRevisionS(c)
  
  protected override def createNodeList(elems: Array[ImmutableNodeInst], oldList: ImmutableNodeInst.Iterable) = {
    if (oldList == null) createNodeList(elems, CellRevisionProviderScala.emptyNodeList)
    else if (elems == null) oldList
    else if (elems.isEmpty) CellRevisionProviderScala.emptyNodeList
    else {
      val it = oldList.iterator()
      var changed = false
      var i = 0
      while (it.hasNext && i < elems.length) {
        val on = it.next()
        if (elems(i) == on) {
          i += 1
        } else {
          changed = true
          var cmp = TextUtils.STRING_NUMBER_ORDER.compare(elems(i).name.toString, on.name.toString)
          while (cmp < 0) {
            if (i > 0 && TextUtils.STRING_NUMBER_ORDER.compare(elems(i - 1).name.toString, elems(i).name.toString) >= 0)
              throw new IllegalArgumentException("nodes order")
            i += 1
            cmp = if (i == elems.length) +1 else
              TextUtils.STRING_NUMBER_ORDER.compare(elems(i).name.toString, on.name.toString)
          }
          if (cmp == 0)
            i += 1
        }
      }
      while (i < elems.length) {
        changed = true
        if (i > 0 && TextUtils.STRING_NUMBER_ORDER.compare(elems(i - 1).name.toString, elems(i).name.toString) >= 0)
          throw new IllegalArgumentException("nodes order")
        i += 1
      }
      if (it.hasNext) {
        changed = true
      }
      if (changed) new ImmutableNodeInstIterableS(elems) else oldList
    }
  }
  
  protected override def createArcList(elems: Array[ImmutableArcInst], oldList: ImmutableArcInst.Iterable) =
    super.createArcList(elems, oldList)
  
  protected override def createExportList(elems: Array[ImmutableExport], oldList: ImmutableExport.Iterable) =
    super.createExportList(elems, oldList)
}

/**
 * Immutable list of ImmutableNodeInsts
 */
private[database] class ImmutableNodeInstIterableS(elems: Array[ImmutableNodeInst]) extends ImmutableNodeInst.Iterable {

//        ImmutableNodeInstIterable(ImmutableNodeInst[] elems) {
//            this.elems = elems;
//            ImmutableNodeInst prevN = null;
//            for (ImmutableNodeInst n : this.elems) {
//                if (prevN != null && TextUtils.STRING_NUMBER_ORDER.compare(prevN.name.toString(), n.name.toString()) >= 0) {
//                    throw new IllegalArgumentException("nodes order");
//                }
//                prevN = n;
//            }
//        }

  /**
   * Returns <tt>true</tt> if this list contains no ImmutableNodeInsts.
   * @return <tt>true</tt> if this list contains no ImmutableNodeInsts.
   */
  override def isEmpty(): Boolean = elems.length == 0

  /**
   * Returns the number of ImmutableNodeInsts in this list.
   * @return the number of ImmutableNodeInsts in this list
   */
  override def size(): Int = elems.length

  /**
   * Returns the ImmutableNodeInst at the specified position in this list.
   *
   * @param index index of the element to return
   * @return the element at the specified position in this list
   * @throws IndexOutOfBoundsException if the index is out of range
   *         (<tt>index &lt; 0 || index &gt;= size()</tt>)
   */
  override def get(index: Int): ImmutableNodeInst = elems(index)

  override def iterator(): java.util.Iterator[ImmutableNodeInst] = ArrayIterator.iterator(elems)

  /**
   * Searches the nodes for the specified name using the binary
   * search algorithm.
   * @param name the name to be searched.
   * @return index of the search name, if it is contained in the arcs;
   *	       otherwise, <tt>(-(<i>insertion point</i>) - 1)</tt>.  The
   *	       <i>insertion point</i> is defined as the point at which the
   *	       ImmutableNodeInst would be inserted into the list: the index of the first
   *	       element greater than the name, or <tt>size()</tt>, if all
   *	       elements in the list are less than the specified name.  Note
   *	       that this guarantees that the return value will be &gt;= 0 if
   *	       and only if the ImmutableNodeInst is found.
   */
  override def searchByName(name: String): Int = {
    var low = 0
    var high = elems.length - 1
    var pick = high // initially try the last postition
    while (low <= high) {
      val n = elems(pick)
      val cmp = TextUtils.STRING_NUMBER_ORDER.compare(n.name.toString(), name)
      if (cmp < 0) {
        low = pick + 1
      } else if (cmp > 0) {
        high = pick - 1
      } else {
        return pick // NodeInst found
      }
      pick = (low + high) >> 1 // try in a middle
    }
    -(low + 1) // NodeInst not found.
  }
}

