﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoT {
namespace Model {

/**
 */
class DeleteJobExecutionRequest : public IoTRequest {
 public:
  AWS_IOT_API DeleteJobExecutionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteJobExecution"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  AWS_IOT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The ID of the job whose execution on a particular device will be deleted.</p>
   */
  inline const Aws::String& GetJobId() const { return m_jobId; }
  inline bool JobIdHasBeenSet() const { return m_jobIdHasBeenSet; }
  template <typename JobIdT = Aws::String>
  void SetJobId(JobIdT&& value) {
    m_jobIdHasBeenSet = true;
    m_jobId = std::forward<JobIdT>(value);
  }
  template <typename JobIdT = Aws::String>
  DeleteJobExecutionRequest& WithJobId(JobIdT&& value) {
    SetJobId(std::forward<JobIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the thing whose job execution will be deleted.</p>
   */
  inline const Aws::String& GetThingName() const { return m_thingName; }
  inline bool ThingNameHasBeenSet() const { return m_thingNameHasBeenSet; }
  template <typename ThingNameT = Aws::String>
  void SetThingName(ThingNameT&& value) {
    m_thingNameHasBeenSet = true;
    m_thingName = std::forward<ThingNameT>(value);
  }
  template <typename ThingNameT = Aws::String>
  DeleteJobExecutionRequest& WithThingName(ThingNameT&& value) {
    SetThingName(std::forward<ThingNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the job execution to be deleted. The <code>executionNumber</code>
   * refers to the execution of a particular job on a particular device.</p> <p>Note
   * that once a job execution is deleted, the <code>executionNumber</code> may be
   * reused by IoT, so be sure you get and use the correct value here.</p>
   */
  inline long long GetExecutionNumber() const { return m_executionNumber; }
  inline bool ExecutionNumberHasBeenSet() const { return m_executionNumberHasBeenSet; }
  inline void SetExecutionNumber(long long value) {
    m_executionNumberHasBeenSet = true;
    m_executionNumber = value;
  }
  inline DeleteJobExecutionRequest& WithExecutionNumber(long long value) {
    SetExecutionNumber(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>(Optional) When true, you can delete a job execution which is "IN_PROGRESS".
   * Otherwise, you can only delete a job execution which is in a terminal state
   * ("SUCCEEDED", "FAILED", "REJECTED", "REMOVED" or "CANCELED") or an exception
   * will occur. The default is false.</p>  <p>Deleting a job execution which
   * is "IN_PROGRESS", will cause the device to be unable to access job information
   * or update the job execution status. Use caution and ensure that the device is
   * able to recover to a valid state.</p>
   */
  inline bool GetForce() const { return m_force; }
  inline bool ForceHasBeenSet() const { return m_forceHasBeenSet; }
  inline void SetForce(bool value) {
    m_forceHasBeenSet = true;
    m_force = value;
  }
  inline DeleteJobExecutionRequest& WithForce(bool value) {
    SetForce(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The namespace used to indicate that a job is a customer-managed job.</p>
   * <p>When you specify a value for this parameter, Amazon Web Services IoT Core
   * sends jobs notifications to MQTT topics that contain the value in the following
   * format.</p> <p>
   * <code>$aws/things/<i>THING_NAME</i>/jobs/<i>JOB_ID</i>/notify-namespace-<i>NAMESPACE_ID</i>/</code>
   * </p>  <p>The <code>namespaceId</code> feature is only supported by IoT
   * Greengrass at this time. For more information, see <a
   * href="https://docs.aws.amazon.com/greengrass/v2/developerguide/setting-up.html">Setting
   * up IoT Greengrass core devices.</a> </p>
   */
  inline const Aws::String& GetNamespaceId() const { return m_namespaceId; }
  inline bool NamespaceIdHasBeenSet() const { return m_namespaceIdHasBeenSet; }
  template <typename NamespaceIdT = Aws::String>
  void SetNamespaceId(NamespaceIdT&& value) {
    m_namespaceIdHasBeenSet = true;
    m_namespaceId = std::forward<NamespaceIdT>(value);
  }
  template <typename NamespaceIdT = Aws::String>
  DeleteJobExecutionRequest& WithNamespaceId(NamespaceIdT&& value) {
    SetNamespaceId(std::forward<NamespaceIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobId;

  Aws::String m_thingName;

  long long m_executionNumber{0};

  bool m_force{false};

  Aws::String m_namespaceId;
  bool m_jobIdHasBeenSet = false;
  bool m_thingNameHasBeenSet = false;
  bool m_executionNumberHasBeenSet = false;
  bool m_forceHasBeenSet = false;
  bool m_namespaceIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
