/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.filter;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpOptions;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpTrace;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyService;
import org.apache.portals.applications.webcontent2.proxy.impl.ServletRequestContext;
import org.apache.portals.applications.webcontent2.proxy.util.RequestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Generic reverse proxy servlet filter invoking {@link ReverseProxyService} injected.
 * <p>
 * Unlike servlets, servlet filters cannot know the filter mapping path prefix by itself.
 * So, this filter should be configured with <code>filterPath</code> init parameters
 * in order to get the link rewriting and redirection done properly with the prefix.
 * For example, if this filter is mapped to '/rproxyfilter/*', then you should add an init parameter
 * like the following:
 * <xmp>
 *   <init-param>
 *     <param-name>filterPath</param-name>
 *     <param-value>/rproxyfilter</param-value>
 *   </init-param>
 * </xmp>
 * </p>
 */
public class GenericReverseProxyFilter implements Filter
{

    /**
     * Default allowed HTTP methods.
     */
    private static final Set<String> DEFAULT_ALLOWED_HTTP_METHODS = 
        new HashSet<String>(Arrays.asList(HttpGet.METHOD_NAME, 
                                          HttpHead.METHOD_NAME, 
                                          HttpPost.METHOD_NAME, 
                                          HttpPut.METHOD_NAME, 
                                          HttpDelete.METHOD_NAME,
                                          HttpOptions.METHOD_NAME,
                                          HttpTrace.METHOD_NAME));

    private static Logger log = LoggerFactory.getLogger(GenericReverseProxyFilter.class);

    /**
     * The {@link ServletConfig} instance for this filter.
     */
    private FilterConfig filterConfig;

    /**
     * Internal {@link ReverseProxyService} to invoke while processing reverse proxy requests.
     */
    private ReverseProxyService proxyService;

    /**
     * The filter mapping path prefix.
     * <p>
     * For example, if this filter is mapped to '/rproxyfilter/*',
     * then this property should be "/rproxyfilter".
     * </p>
     */
    private String filterPath;

    /**
     * Zero-argument default constructor.
     */
    public GenericReverseProxyFilter()
    {
    }

    /**
     * Returns the internal {@link ReverseProxyService} instance.
     * @return
     */
    public ReverseProxyService getProxyService()
    {
        return proxyService;
    }

    /**
     * Sets the internal {@link ReverseProxyService} instance.
     * @param proxyService
     */
    public void setProxyService(ReverseProxyService proxyService)
    {
        this.proxyService = proxyService;
    }

    /**
     * Initializes this servlet filter.
     */
    public void init(FilterConfig filterConfig) throws ServletException
    {
        this.filterConfig = filterConfig;

        filterPath = StringUtils.trim(filterConfig.getInitParameter("filterPath"));
    }

    /**
     * Invokes the internal {@link ReverseProxyService} on allowed HTTP requests.
     */
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException, ServletException
    {
        if (!(req instanceof HttpServletRequest) || !(res instanceof HttpServletResponse))
        {
            chain.doFilter(req, res);
            return;
        }

        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) res;

        if (!DEFAULT_ALLOWED_HTTP_METHODS.contains(request.getMethod()))
        {
            chain.doFilter(request, response);
        }
        else
        {
            try
            {
                ServletRequestContext requestContext = new ServletRequestContext(request, response);

                String requestBasePath = getRequestBasePath(request);

                if (requestBasePath == null)
                {
                    requestBasePath = request.getContextPath();
                }

                requestContext.setRequestBasePath(requestBasePath);

                ProxyContext proxyContext = new ProxyContext(requestContext);
                ProxyContext.setCurrentProxyContext(proxyContext);
                proxyService.invoke(proxyContext);
            }
            catch (ReverseProxyException e)
            {
                if (e.getStatusCode() > 0)
                {
                    if (log.isDebugEnabled())
                    {
                        log.error("Response proxy processing exception occurred.", e);
                    }
                    else
                    {
                        log.error("Response proxy processing exception occurred. " + e);
                    }

                    response.sendError(e.getStatusCode(), e.getLocalizedMessage());
                }
                else
                {
                    throw new ServletException(e);
                }
            }
            catch (IOException e)
            {
                throw e;
            }
            catch (Exception e)
            {
                throw new ServletException(e);
            }
            finally
            {
                ProxyContext.removeCurrentProxyContext();
            }
        }
    }

    /**
     * Destroys this servlet filter.
     */
    public void destroy()
    {
    }

    /**
     * Returns the {@link FilterConfig} of this servlet filter.
     * @return
     */
    protected FilterConfig getFilterConfig()
    {
        return filterConfig;
    }

    /**
     * Returns the servlet filter mapping path prefix.
     * For example, if the servlet filter is mapped to '/rproxyfilter/*', then
     * this method is supposed to return '/rproxyfilter'.
     * @return
     */
    protected String getFilterPath()
    {
        return filterPath;
    }

    /**
     * Returns the concatenation of the servlet context path 
     * (see {@link javax.servlet.ServletContext#getContext(String)}
     * and the filterPath (see {@link #getFilterPath()}).
     */
    protected String getRequestBasePath(HttpServletRequest request)
    {
        StringBuilder sb = new StringBuilder();
        sb.append(RequestUtils.getContextPath(request));

        if (filterPath != null)
        {
            sb.append(filterPath);
        }

        return sb.toString();
    }
}
