/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.command;

import java.io.IOException;
import java.net.URI;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.chain.Command;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.URICleaner;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * {@link Command} responsible for resolving the remote target URI
 * from the local request path info and the resolved {@link ProxyMapping}.
 */
public class ResolveRemoteURICommand extends AbstractProxyCommand
{

    private static Logger log = LoggerFactory.getLogger(ResolveRemoteURICommand.class);

    /**
     * Internal {@link URICleaner} which is used to verify
     * the resolved remote target URI and clean it up if necessary.
     */
    private URICleaner uriCleaner;

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean executeInternal(final ProxyContext context) throws ReverseProxyException, IOException
    {
        String localPath = context.getLocalPath();
        String remoteURI = context.getResolvedMapping().resolveRemoteFromLocal(localPath);

        if (remoteURI == null)
        {
            throw new ReverseProxyException(HttpServletResponse.SC_NOT_FOUND, "No mapped URI.");
        }

        // resolve the scheme for scheme-less URI.
        if (remoteURI.startsWith("//"))
        {
            final String defaultScheme = getDefaultScheme(context);
            remoteURI = new StringBuilder(remoteURI.length() + 6).append(defaultScheme).append(':').append(remoteURI).toString();
        }

        String queryString = context.getRequestContext().getQueryString();

        if (queryString != null)
        {
            remoteURI = new StringBuilder(remoteURI.length() + 1 + queryString.length()).append(remoteURI).append('?').append(queryString).toString();
        }

        // Before invoking HTTP Methods, let's clean the remote target URI
        // in order to avoid invalid URI exception (e.g. space in the URI)
        if (uriCleaner != null)
        {
            remoteURI = uriCleaner.clean(remoteURI);
        }

        URI remoteURIObj = null;

        try
        {
            remoteURIObj = URI.create(remoteURI);
        }
        catch (Exception e)
        {
            log.error("Invalid remote target URI: '{}'.", remoteURI);
            return true;
        }

        context.setRemoteURI(remoteURIObj);

        return false;
    }

    /**
     * Returns the internal {@link URICleaner}.
     * @return
     */
    public URICleaner getUriCleaner()
    {
        return uriCleaner;
    }

    /**
     * Sets the internal {@link URICleaner}.
     * @param uriCleaner
     */
    public void setUriCleaner(URICleaner uriCleaner)
    {
        this.uriCleaner = uriCleaner;
    }

    /**
     * Finds the default scheme ("http" or "https") when a scheme-less remote URI was requested.
     * @param context
     * @return
     */
    protected String getDefaultScheme(final ProxyContext context)
    {
        return context.getRequestContext().getScheme();
    }
}
