/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.command;

import java.io.IOException;
import java.util.Enumeration;

import org.apache.commons.chain.Command;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.protocol.HTTP;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;


/**
 * {@link Command} responsible for copying all the original HTTP request headers
 * to the internally created {@link HttpRequestBase} instance.
 * <p>
 * The following HTTP headers are excluded during the copy:
 * <ul>
 * <li><code>Content-Length</code></li>
 * <li><code>Host</code></li>
 * </ul>
 * </p>
 */
public class AddHeaderToHttpRequestCommand extends AbstractProxyCommand
{

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean executeInternal(final ProxyContext context) throws ReverseProxyException, IOException
    {
        HttpRequestBase httpRequest = context.getHttpRequest();

        // pass most headers to remote URI...
        for (@SuppressWarnings("rawtypes")
        Enumeration enumHeaderNames = context.getRequestContext().getHeaderNames(); enumHeaderNames.hasMoreElements();)
        {
            String headerName = (String) enumHeaderNames.nextElement();

            if (StringUtils.equalsIgnoreCase(headerName, HTTP.CONTENT_LEN))
                continue;

            if (StringUtils.equalsIgnoreCase(headerName, HTTP.TARGET_HOST))
                continue;

            for (@SuppressWarnings("rawtypes")
            Enumeration enumHeaderValues = context.getRequestContext().getHeaders(headerName); enumHeaderValues.hasMoreElements();)
            {
                String headerValue = (String) enumHeaderValues.nextElement();
                httpRequest.addHeader(headerName, headerValue);
            }
        }

        return false;
    }

}
