/******************************************************************************
 * (c) Copyright 2002,2003, 1060 Research Ltd
 *
 * This Software is licensed to You, the licensee, for use under the terms of
 * the 1060 Public License v1.0. Please read and agree to the 1060 Public
 * License v1.0 [www.1060research.com/license] before using or redistributing
 * this software.
 *
 * In summary the 1060 Public license has the following conditions.
 * A. You may use the Software free of charge provided you agree to the terms
 * laid out in the 1060 Public License v1.0
 * B. You are only permitted to use the Software with components or applications
 * that provide you with OSI Certified Open Source Code [www.opensource.org], or
 * for which licensing has been approved by 1060 Research Limited.
 * You may write your own software for execution by this Software provided any
 * distribution of your software with this Software complies with terms set out
 * in section 2 of the 1060 Public License v1.0
 * C. You may redistribute the Software provided you comply with the terms of
 * the 1060 Public License v1.0 and that no warranty is implied or given.
 * D. If you find you are unable to comply with this license you may seek to
 * obtain an alternative license from 1060 Research Limited by contacting
 * license@1060research.com or by visiting www.1060research.com
 *
 * NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC
 * LICENSE V1.0 FOR DETAILS
 *
 * THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ
 * THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license
 *
 * File:          $RCSfile: JMSTransport.java,v $
 * Version:       $Name:  $ $Revision: 1.7 $
 * Last Modified: $Date: 2005/04/14 17:56:21 $
 *****************************************************************************/
package org.ten60.netkernel.jms;

import org.ten60.netkernel.layer1.nkf.impl.NKFTransportImpl;
import org.ten60.netkernel.layer1.nkf.*;
import org.ten60.netkernel.layer1.representation.*;
import org.ten60.netkernel.xml.xda.*;
import com.ten60.netkernel.scheduler.*;
import com.ten60.netkernel.urii.*;
import com.ten60.netkernel.urii.aspect.*;
import com.ten60.netkernel.util.*;
import java.util.*;
import javax.jms.*;


/**
 * A Transport to connect to JMS Queues and Topics
 * @author  tab
 */
public class JMSTransport extends NKFTransportImpl implements MessageListener
{
	private String mDescription="JMSTransport";
	private WorkerThreadPool mPool;
	private IURRepresentation mConfig;
	private List mConsumers;
	private List mSessions;
	private HashSet mNeedAck;
	
	/** Return a generated description of the transport */
	public String getDescription()
	{	return mDescription;
	}
	
	/** Starts the transport
	 */
	protected void startTransport() throws Exception
	{	ThreadGroup root=getContext().getKernelHelper().getKernel().getRootThreadGroup();
		mPool = new WorkerThreadPool("JMSTransport",1,root)
		{	public void process()
			{	try
				{	innerProcess();
				}
				catch (Exception e)
				{	SysLogger.log1(SysLogger.WARNING,this, "Error initialising connection to JMS Message Broker - exception follows...",e.getMessage());
					SysLogger.log(SysLogger.WARNING, this, e.toString());
					try
					{	Thread.currentThread().sleep(10000);
					} catch (InterruptedException e2) {;}
				}
			}
		};
		mPool.start();
	}
	
	/** Closes connections and stops the transport
	 */
	protected void stopTransport() throws Exception
	{	JMSConnectionAspect connection = (JMSConnectionAspect)mConfig.getAspect(JMSConnectionAspect.class);
		cleanup(connection);
		mPool.stop();
		mPool.join();
	}
	
	/** Called once a second to check for updates to configuration
	 */
	private void innerProcess() throws Exception
	{	
		Thread.currentThread().sleep(1000);
		INKFConvenienceHelper context = getContext();
		if (mConfig==null || mConfig.getMeta().isExpired())
		{	IURRepresentation config = context.source("ffcpl:/etc/JMSConfig.xml", JMSConnectionAspect.class);
			if (mConfig!=config)
			{	mConfig=config;
				initialiseConfig();
			}
		}
		
	}
	
	/** Called if the configuration changes.
	 * initializes internal data structures and connects to all queues and topics with a listener
	 */
	private void initialiseConfig() throws Exception
	{	
		JMSConnectionAspect connection = (JMSConnectionAspect)mConfig.getAspect(JMSConnectionAspect.class);
		cleanup(connection);
		QueueConnection qc = connection.getQueueConnection();
		TopicConnection tc = connection.getTopicConnection();
		
		// initialise new listener
		List names = connection.getNames();
		mConsumers = new ArrayList();
		mSessions = new ArrayList();
		mNeedAck = new HashSet();
		mDescription = "JMSTransport:";
		for (Iterator i=names.iterator(); i.hasNext(); )
		{	String name = (String)i.next();
			try
			{	Destination dest = (Destination) connection.getJNDIContext().lookup(name);
				String physicalName;
				IXDAReadOnly config = connection.getConfiguration(name);
				boolean ackOnException = config.getText("ackOnException",true).toLowerCase().equals("true");
				int acknowedgeMode = ackOnException?Session.AUTO_ACKNOWLEDGE:Session.CLIENT_ACKNOWLEDGE;
				String messageSelector=null;
				try
				{	messageSelector = config.getText("messageSelector",true);
				} catch (Exception e)
				{ /* no selector */ }
				Session s;
				if (dest instanceof Queue)
				{	if (qc==null) throw new Exception("No QueueConnectionFactory registered");
					s = qc.createQueueSession(false, acknowedgeMode);
					physicalName = ((Queue)dest).getQueueName();
				}
				else if (dest instanceof Topic)
				{	if (tc==null) throw new Exception("No TopicConnectionFactory registered");
					s = tc.createTopicSession(false, acknowedgeMode);
					physicalName = ((Topic)dest).getTopicName();
				}
				else throw new NKFException("Destination must be queue or topic");
				if (!ackOnException)
				{	mNeedAck.add(physicalName);
				}
				MessageConsumer consumer = s.createConsumer(dest,messageSelector);
				consumer.setMessageListener(this);
				mSessions.add(s);
				mConsumers.add(consumer);
				mDescription+=" "+name;
			}
			catch (Exception e)
			{	SysLogger.log1(SysLogger.WARNING, this, "Failed to initialise consumer for %1",name);
				SysLogger.log(SysLogger.WARNING, this, e.toString());
			}
		}
		if (qc!=null) qc.start();
		if (tc!=null) tc.start();
	}
	
	/** called on stop and when configuration changes to close existing connections and sessions
	 */
	private void cleanup(JMSConnectionAspect aConnection) throws Exception
	{
		QueueConnection qc = aConnection.getQueueConnection();
		TopicConnection tc = aConnection.getTopicConnection();
		if (qc!=null) qc.stop();
		if (tc!=null) tc.stop();
		
		// cleanup last config
		if (mConsumers!=null)
		{	for (Iterator i=mConsumers.iterator(); i.hasNext(); )
			{	MessageConsumer c = (MessageConsumer)i.next();
				c.setMessageListener(null);
				c.close();
			}
			mConsumers=null;
		}
		if (mSessions!=null)
		{	for (Iterator i=mSessions.iterator(); i.hasNext(); )
			{	Session s = (Session)i.next();
				s.close();
			}
			mSessions=null;
		}		
	}
	
	/** Called when a message arrives on a topic or queue we are listening to
	 */
	public void onMessage(Message aMessage)
	{	String uriString=null;
		try
		{	Destination dest = aMessage.getJMSDestination();	
			String name;
			if (dest instanceof Queue)
			{	name = ((Queue)dest).getQueueName();
				uriString="jms-queue-transport:"+name;
			}
			else if (dest instanceof Topic)
			{	name = ((Topic)dest).getTopicName();
				uriString="jms-topic-transport:"+name;
			}
			else throw new Exception("Unknown destination type: "+dest.getClass().getName());
			dispatchRequest(uriString, aMessage);
			if (mNeedAck.contains(name))
			{	aMessage.acknowledge();
			}
		}
		catch (Exception e)
		{
			SysLogger.log2(SysLogger.WARNING, this, "Failed to process message from %1: %2",uriString, e.getMessage());
			SysLogger.log(SysLogger.WARNING, this, e.toString());		
		}
	}	
	
	/** Dispatches the given message to the transport manager for execution
	 */
	private void dispatchRequest(String aDestName, Message aMessage) throws Exception
	{	
		// get message body
		IURAspect body;
		if (aMessage instanceof TextMessage)
		{	TextMessage m = (TextMessage)aMessage;
			body = new org.ten60.netkernel.layer1.representation.StringAspect(m.getText());
		}
		else if (aMessage instanceof BytesMessage)
		{	BytesMessage m = (BytesMessage)aMessage;
			long length = m.getBodyLength();
			if (length>Integer.MAX_VALUE) throw new NKFException("BytesMessage too long",Long.toString(length),null);
			byte[] bytes = new byte[(int)length];
			m.readBytes(bytes);
			body = new ByteArrayAspect(bytes);
		}
		else
		{	throw new NKFException("Unsupported message type",aMessage.getClass().getName(),null);
		}
		
		//get properties
		NVPImpl properties = new NVPImpl();
		for (Enumeration e=aMessage.getPropertyNames(); e.hasMoreElements(); )
		{	String propertyName = (String)e.nextElement();
			String propertyValue = aMessage.getStringProperty(propertyName);
			properties.addNVP(propertyName,propertyValue);
		}
		
		//get headers
		NVPImpl header = new NVPImpl();
		header.addNVP( "JMSCorrelationID", aMessage.getJMSCorrelationID() );
		header.addNVP( "JMSDeliveryMode", (aMessage.getJMSDeliveryMode()==DeliveryMode.PERSISTENT)?"persistent":"non-persistent" );
		header.addNVP( "JMSExpiration", Long.toString(aMessage.getJMSExpiration()) );
		header.addNVP( "JMSMessageID", aMessage.getJMSMessageID() );
		header.addNVP( "JMSPriority", Integer.toString(aMessage.getJMSPriority()) );
		header.addNVP( "JMSRedelivered", Boolean.toString(aMessage.getJMSRedelivered()) );
		header.addNVP( "JMSTimestamp", Long.toString(aMessage.getJMSTimestamp()) );
		header.addNVP( "JMSType", aMessage.getJMSType() );
		Destination replyTo=aMessage.getJMSReplyTo();
		String replyToString="";
		if (replyTo!=null)
		{	JMSConnectionAspect connection = (JMSConnectionAspect)mConfig.getAspect(JMSConnectionAspect.class);
			replyToString = connection.getURIForPhysicalDestination(replyTo.toString());
			if (replyToString==null) replyToString="";
		}
		header.addNVP( "JMSReplyTo", replyToString );
		
		INKFConvenienceHelper context = getContext();
		INKFRequest req = context.createSubRequest();
		req.setURI(aDestName);
		req.addArgument("body",body);
		req.addArgument("properties",new NVPAspect(properties));
		req.addArgument("header",new NVPAspect(header));
		getContext().issueSubRequest(req);
	}
}