package com.streamsicle.fluid;

import java.util.*;
import java.io.*;
import org.ten60.orchextra.*;

/**
 * A MediaInputStream that handles the MP3 format.
 * <P>
 * This class relies heavily on the MP3FrameReader
 * and will read and store a given number of
 * frames in a frame buffer before making this
 * packet available to the overlying layers.
 *
 * @author Lars Samuelsson
 */
public class MP3InputStream extends MediaInputStream {
    private Delay delay;
    private boolean done;
    private MP3FrameReader mp3input;
    private byte[][] framebuf;
        
    /**
     * Creates an MP3InputStream wrapped around the
     * given input stream with a default frame buffer
     * size of 6 frames.
     *
     * @param mediaInput      The input stream containing
     *                        MP3 data
     */
    public MP3InputStream(InputStream mediaInput) {
        super(mediaInput);
        mp3input = new MP3FrameReader(mediaInput);
        framebuf = new byte[6][];
        done = false;
    }
    /**
     * Creates an MP3InputStream that is yet to be
     * associated with a specific media.
     */
    public MP3InputStream() {
        super();
    }
    /**
     * Configures the MP3 input stream.
     * <P>
     * The properties that are read from the configuration
     * are:
     * <P>
     * <UL>
     * <LI> <B> mp3inputstream.buffersize </B><BR>
     *      The size of the frame buffer, ie how
     *      many frames will be stuffed into a
     *      data packet before being sent
     * <LI> <B> mp3inputstream.filename </B><BR>
     *      Optional parameter specifying the name
     *      of an MP3 file that will be used for
     *      reading
     * </UL>
     *
     * @param conf A configuration from which
     *             parameters can be read
     */
    public void configure () {
	try {
            
	    mp3input = new MP3FrameReader(getInputStream());
	    int bufsize = Integer.parseInt(getProperties().getProperty("mp3inputstream.buffersize"));
	    framebuf = new byte[bufsize][];
	    done = false;
	    String filename = getProperties().getProperty("mp3inputstream.filename");
	    if(filename != null)
		setInputStream(new FileInputStream(filename));
	}
	catch(Exception e) {
	    e.printStackTrace();
	    OrchextraAccessor.log(OrchextraAccessor.SEVERE, this, "The MP3InputStream could not be configured");
	}
    }
    /**
     * The delay that the MediaInputStream will have to
     * wait before fetching the next packet.
     *
     * @return Between-packet delay or null if the end
     *         of the file has been reached or if there
     *         is no input stream specified
     */
    public Delay getDelay() {
	// if end of file reached
    	if(done || (mp3input == null))
	        return null;
	    return delay;
    }
    /**
     * This method will read a number of frames
     * using the MP3FrameReader and store these
     * frames in a frame buffer.
     *
     * @return The contents of the frame buffer,
     *         ie a packet, or null when the
     *         end of the file has been reached
     */
    public byte[] read() throws IOException {
        if(done)
            return null;
        delay = new Delay(0);
        byte[] frame;
        for(int i = 0; i < framebuf.length; i++) {
            frame = mp3input.nextFrame();
            if(frame == null) {
	       // done reading so set the done flag and close the stream
	       done = true;
	       getInputStream().close();
	       return Bitmask.assemble(framebuf, i);
            }
            framebuf[i] = frame;
            if (delay != null) {
              delay = delay.add(mp3input.getDelay());
            } 
        }
        return Bitmask.assemble(framebuf, framebuf.length);
    }
    /**
     * To check what kind of media this stream wrapper will
     * handle.
     *
     * @param type A media type (extension)
     * @return     true for "mp3" or "mpeg3"
     */
    public boolean handlesMedia(String type) {
    	return type.toLowerCase().equals("mp3") || type.toLowerCase().equals("mpeg3");
    }
}
