/******************************************************************************
  (c) Copyright 2002,2003, 1060 Research Ltd                                   

  This Software is licensed to You, the licensee, for use under the terms of   
  the 1060 Public License v1.0. Please read and agree to the 1060 Public       
  License v1.0 [www.1060research.com/license] before using or redistributing   
  this software.                                                               

  In summary the 1060 Public license has the following conditions.             
  A. You may use the Software free of charge provided you agree to the terms   
  laid out in the 1060 Public License v1.0                                     
  B. You are only permitted to use the Software with components or applications
  that provide you with OSI Certified Open Source Code [www.opensource.org], or
  for which licensing has been approved by 1060 Research Limited.              
  You may write your own software for execution by this Software provided any  
  distribution of your software with this Software complies with terms set out 
  in section 2 of the 1060 Public License v1.0                                 
  C. You may redistribute the Software provided you comply with the terms of   
  the 1060 Public License v1.0 and that no warranty is implied or given.       
  D. If you find you are unable to comply with this license you may seek to    
  obtain an alternative license from 1060 Research Limited by contacting       
  license@1060research.com or by visiting www.1060research.com                 

  NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC  
  LICENSE V1.0 FOR DETAILS                                                     

  THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ     
  THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license      

  File:          $RCSfile: DependencyMeta.java,v $
  Version:       $Name:  $ $Revision: 1.3 $
  Last Modified: $Date: 2004/06/21 10:23:31 $
 *****************************************************************************/
package org.ten60.netkernel.layer1.meta;

import com.ten60.netkernel.urii.*;
import java.util.*;

/**
 * An implementation of IURMeta that supports chaining of IURRepresentations for
  * derived results such that costs and expiry dependencies are accumulated.
 * @author  tab
 */
public class DependencyMeta extends MetaImpl implements IHasDependencies
{
	private static final int MASK_EXPIRED=1;
	private static final int MASK_INTERMEDIATE=2;
	private static final int MASK_CONTEXT_SENSITIVE=4;
	private int mFlags;
	
	private List mDependencies;
	private long mLastPessimisticExpiry=1;
	private int mUsageCost;
	
	/** Used by accessors for results */
	public DependencyMeta(String aMimeType, int aCreationCost, int aUsageCost)
	{	super(aMimeType, 0, aCreationCost);
		mUsageCost=aUsageCost;
		setIntermediate(true);
	}
	
	/* used by streaming transmutors */
	public DependencyMeta(IURRepresentation aOther, int aCreationCost, int aUsageCost)
	{	super(aOther.getMeta().getMimeType(), 0, aCreationCost);
		mUsageCost=aUsageCost;
		addUsageDependency(aOther);
		setIntermediate(aOther.getMeta().isIntermediate());
		setContextSensitive(aOther.getMeta().isContextSensitive());
	}

	/* used by transreptor to add cost */
	public DependencyMeta(IURRepresentation aOther, int aCreationCost)
	{	super(aOther.getMeta().getMimeType(), 0, aCreationCost);
		IURMeta otherMeta = aOther.getMeta();
		setIntermediate(otherMeta.isIntermediate());
		if (otherMeta instanceof DependencyMeta)
		{	setPessimisticExpiryTime( ((DependencyMeta)otherMeta).getSuperExpiryTime());
		}
		addDependenciesOf(otherMeta);
		setContextSensitive(aOther.getMeta().isContextSensitive());
	}

	/* used to create intermediate flattened results (in DPML) */
	public DependencyMeta(IURRepresentation aOther)
	{	super(aOther.getMeta().getMimeType(), 0, 0);
		setIntermediate(true);
		IURMeta otherMeta = aOther.getMeta();
		if (otherMeta instanceof DependencyMeta)
		{	setPessimisticExpiryTime( ((DependencyMeta)otherMeta).getSuperExpiryTime());
		}
		addDependenciesOf(otherMeta);
	}
	
	/** internal method to increment creation and usage cost
	 */
	private void incrementCost(int aCreationCost, int aUsageCost)
	{	super.incrementCreationCost(aCreationCost);
		mUsageCost+= aUsageCost;
	}
	
	/** return the usage cost
	 */
	public int getUsageCost()
	{	return mUsageCost;
	}
	
	public void addDependency(IURRepresentation aRep)
	{	addDependency(aRep.getMeta());
	}
	public void addUsageDependency(IURRepresentation aRep)
	{	addUsageDependency(aRep.getMeta());
	}
	
	/** add a creation dependency to this meta
	 */
	public void addDependency(IURMeta aDependent)
	{	if (mDependencies==null)
		{	mDependencies = new ArrayList(4);
		}
		mDependencies.add(aDependent);
		incrementCost(aDependent.getCreationCost()+aDependent.getUsageCost(),0);
	}
	
	/** add a usage dependency to this meta
	 */
	public void addUsageDependency(IURMeta aDependent)
	{	if (mDependencies==null)
		{	mDependencies = new ArrayList(4);
		}
		mDependencies.add(aDependent);
		incrementCost(aDependent.getCreationCost(), aDependent.getUsageCost());
	}

	/** This needs a bit of explanation. SuperExpiry time is a baseline time if non-zero
	 * will cause pessimistic expiry of its resources to be ignored. This is useful
	 * for assigning a validity period to external clients over transports.
	 */
	public final long getSuperExpiryTime()
	{	return super.getPessimisticExpiryTime();
	}
	
	/** recursively calculate the pessimistic expiry time
	 */
	public long getPessimisticExpiryTime()
	{	long result = getSuperExpiryTime();
		if (result==0)
		{	long time = System.currentTimeMillis();
			if (mLastPessimisticExpiry==Long.MAX_VALUE || (mLastPessimisticExpiry>0 && mLastPessimisticExpiry<time))
			{	
				long expiry = Long.MAX_VALUE;
				if (mDependencies!=null)
				{	for (int i=mDependencies.size()-1; i>=0; i--)
					{	IURMeta meta = (IURMeta)mDependencies.get(i);
						long pet = meta.getPessimisticExpiryTime();
						if (pet<expiry)
						{	expiry=pet;
							if (expiry<time) break;
						}
					}
				}
				mLastPessimisticExpiry=expiry;
			}
			result = mLastPessimisticExpiry;
		}
		return result;
	}
	
	/** return true if the representation we are the meta for is intermediate
	 */
	public boolean isIntermediate()
	{	return (mFlags&MASK_INTERMEDIATE)!=0;
	}	
	
	/** recursive evaluation of isExpired by looking at dependencies
	 */
	public boolean isExpired()
	{	boolean result = (mFlags&MASK_EXPIRED)!=0;
		if (!result && mDependencies!=null)
		{	long now = System.currentTimeMillis();
			for (int i=mDependencies.size()-1; i>=0; i-- )
			{	IURMeta meta = (IURMeta)mDependencies.get(i);
				if (meta.isExpired())
				{	result=true;
					break;
				}
			}
			setExpired(result);
		}
		return result;
	}
	
	/** Internal method to add all dependencies of the given representation
	 */
	private void addDependenciesOf(IURMeta aDependent)
	{	
		
		if (aDependent instanceof DependencyMeta)
		{	if (mDependencies==null)
			{	mDependencies = new ArrayList(8);
			}
			DependencyMeta dm =  (DependencyMeta)aDependent;
			addDependencies(dm);
			incrementCost(dm.getCreationCost(),dm.getUsageCost());
		}
		else
		{	addDependency(aDependent);
		}
	}
	
	/** Internal method to add all dependencies of the given dependency meta
	 */
	private void addDependencies( DependencyMeta aMeta)
	{	
		if (aMeta.mDependencies!=null)
		{	
			for (int i=aMeta.mDependencies.size()-1; i>=0; i--)
			{	IURMeta meta = (IURMeta)aMeta.mDependencies.get(i);
				if (meta instanceof DependencyMeta /*&& meta.isIntermediate()*/)
				{	addDependencies((DependencyMeta)meta);
				}
				else
				{	if (!mDependencies.contains(meta))
					{	mDependencies.add(meta);
					}
				}
			}
		}
	}
	
	public boolean isContextSensitive()
	{	return (mFlags&MASK_CONTEXT_SENSITIVE)!=0;
	}
	
	public void setIntermediate(boolean aIsIntermediate)
	{	if (aIsIntermediate)
		{	mFlags|=MASK_INTERMEDIATE;
		}
		else
		{	mFlags&=(~MASK_INTERMEDIATE);
		}
	}

	private void setExpired(boolean aIsExpired)
	{	if (aIsExpired)
		{	mFlags|=MASK_EXPIRED;
		}
		else
		{	mFlags&=(~MASK_EXPIRED);
		}
	}
	private void setContextSensitive(boolean aIsContextSensitive)
	{	if (aIsContextSensitive)
		{	mFlags|=MASK_CONTEXT_SENSITIVE;
		}
		else
		{	mFlags&=(~MASK_CONTEXT_SENSITIVE);
		}
	}
	
	/** Introspect the dependencies on the resource representented by this 
	 * meta and return all IURMetas
	 */
	public Set getDependencies()
	{	Set s = new HashSet(32);
		addDependencies(s);
		return s;
	}
	
	private void addDependencies(Set aDependencies)
	{	if (mDependencies!=null)
		{	for (int i=mDependencies.size()-1; i>=0; i--)
			{	IURMeta meta = (IURMeta)mDependencies.get(i);
				if (meta instanceof DependencyMeta)
				{	DependencyMeta dm = (DependencyMeta)meta;
					dm.addDependencies(aDependencies);
				}
				else
				{	aDependencies.add(meta);
				}
			}
		}
	}
	public String toString()
	{	return "DependencyMeta";
	}
}